// Author:	Steven Bergner
// Created:	Oct01

#ifndef _VUVOLUMEREGULARUNIMODAL3D1B1VALSPECTRAL_H_
#define _VUVOLUMEREGULARUNIMODAL3D1B1VALSPECTRAL_H_

#include "../intensity.h"
#include "vuSimpleTypes.h"
#include "vuVector.h"
#include "vuSampleRay.h"
#include "vuThread.h"

#include "DatGrid.h"
#include "vuSpectralImage.h"
#include "vuImage.h"
#include "Util.h"
#include "Material.h"

#define NUM_LIGHTS   2

namespace ns_vu1112112 {
using namespace ns_vu1112112;

/**\todo Find a way to tell doxygen to structure the documentation
         for all helper classes together with this main class under 
	 a meaningful name. Or put the helper classes in a General
	 directory.
*/

class vu1112112;
typedef vu1112112 vuVolumeRegularUnimodal3d1B1ValSpectral;
/** This class performs raycasting using a spectral colour model.
    This allows modelling special physical properties of light like 
    spectral absorption and metamerism.
    long name: vuVolumeRegularUnimodal3d1B1ValSpectral
    \author Steven Bergner (using color models implemented by Steve Kilthau)
*/
class vu1112112 : public vu111211, public vuThread
{
    friend class Parser;
    friend class DatGrid;
    friend class DatPnt;
public:
    //!Default constructor creating an empty instance.
    vu1112112();
    //!Copy constructor which does a deep copy.
    vu1112112(const vu1112112& inst);
    //!Destructor.
    virtual ~vu1112112();

    //!Assignment operator which does a deep copy.
    vu1112112& operator=(const vu1112112& rhs);

    //!Sets the camera viewing position for the render method.
    void setViewVectors(const vuVector& pos,const vuVector& vrp,const vuVector& up);

    //!Initializes open gl for rendering the volume data.
    void initOpenGL(void);
    //!Implements the abstract render() method of the vuVolume base class.
    void render();

    //!Reimplements the read() method to do some extra volume data processing.
    virtual bool read();

    //!This method reads volume data stored using the .raw data type.
    virtual bool readRaw(void);

    //!adds a new material (scattering and absorption information) to the renderer
    bool add_material(Material &mat);
    //!adds a light (just a colour)
    bool add_light(ColourType &lc);
    //!loads a scene setup (materials, lights, camera resolution...)
    /*!The function is intended to be piecewise replaced by additional features of 
      vuVolume (e.g. spectral transfer function panel with load and save...)
     */
    bool load_scene(const char *filename);
    //!load data
    /*!This function copies the entire volume into a seperate DatGrid structure.*/
    bool load_volume(char* name);
    //!adjust light between source 1 and 2; w E [0,1], l intensity scaling
    void setLight(const vuColour31a &light);
    //! set size of the canvas to render to
    void setImageSize(dword sx, dword sy);
    //!tell that the data should be rerendered on next glRender
    /*!default behaviour is to just blit the intermediate framebuffer */
    void doRefresh();
    //!returns the center of the dataset
    vuVector getCenter();

    vuImage* getImage ();

    //! Set whether to do specular illumination. (default true)
    void setDoSpecular(bool doit = true) { m_DoSpecular = doit; }
    //! Is specular illumination calculated?
    bool getDoSpecular() { return m_DoSpecular; }
    //! Set whether to draw a preview. (default true)
    void setDrawPreview(bool doit = true) { m_DrawPreview = doit; }
    //! Will a preview be drawn?
    bool getDrawPreview() {return m_DrawPreview; }
    //! Returns handle to Shininess
    float& getShininess() {return m_Shininess; }
    //! Returns handle to Gloss (intensity of specular reflection)
    float& getGloss() {return m_Gloss; }
    //! Tell whether to move the light with the observer
    void setObserverLight(bool moveit=true) 
	{ m_ObserverLight = moveit; }
    //! Set the direction of the light
    void setLightDir(const vuVector& lightdir) {
	m_LightDir = lightdir;
	Grid.shade(*this);
    }
    //! return the direction of the light
    const vuVector& getLightDir() const { return m_LightDir; }

private:
    /** Overrides virtual vuThread::run().
	Calls shootRays to distribute rendering to two different threads. */
    void run(int whatsup, void* data);
    /** Renders the scene by casting the rays from the camera.
	The parameters can be used to render only specific pixels. That can be used
	for progressive rendering or rendering distributed to different processes.
	The default values setup for full rendering.
	\param xofs pixel offset in each line
	\param xinc increment determining the space between two pixels on one line
	\param yofs line to start with
	\param yinc step size to next line */
    void shootRays(int xofs=0, int xinc=1, int yofs=0, int yinc=1);
    //!Preprocesses volume data for rendering once it's been read.
    void preprocess(void);
    //!Cast a ray through the scene used by render()
    ColourType Cast(vuSampleRay& Vray);

private:
    vuVector		center;		//!<center to rotate about

    DatGrid		Grid;		//!< the grid to be rendered
    ColourType		Background;     //!< Background colour
    vuColour31a		Ambient;        //!< Ambient colour for normaliztion
    ColourType		m_Light[NUM_LIGHTS];	//!< different colours for light source

    /** own internal look-up-table for scattering (color)
	Hopefully substituted by spectral transfer function panel. */
    ColourType		TFLUT_scatt[255];

    /** own internal look-up-table for absorption
	Hopefully substituted by spectral transfer function panel. */
    ColourType		TFLUT_absor[255];

    int			numlights;	//!< number of light sources
    Material	mat[MAT_NUM_MATERIALS];	//!< list of materials
    int			nummat;		//!< number of materials
    vuSpectralImage	spimg;		//!< image with spectral information for each pixel
    vuImage		rgbimg;		//!< The rgb image that will be displayed onto the screen.
    vuVector		m_LightDir;	//!< direction of light vector
    bool		m_ObserverLight;//!< move the light with the viewpoint?
    float		brightness;	//!< scalar to toogle brightness of light source
    float		diffuse;	//!< weight for diffuse gradient shading
    float		light_scale;	//!< max value for the brightness slider
    bool		re_light, re_render; //!<refresh tags
    /** buffer for the image.
	This expires when we have more sophisticated vuDrawTools */
    byte		*imgbuf;
    bool		use_flat_absorption; //!<optimize using plain alpha blending
    bool		pp_gradients;	//!<gradients already preprocessed?
    vuMutex		m_Mutex[2];

    bool		m_DoSpecular;	//!<do specular illumination?
    bool		m_DrawPreview;	//!<draw preview?
    float		m_Shininess;	//!<shininess for specular reflection
    float		m_Gloss;	//!<intensity of specular reflection
};

} // end of namespace

#endif
