#include "spectral.h"
#include <fstream.h>
#include <math.h>

#include "DatGrid.h"
#include "Parser.h"
#include "vuColourRGBa.h"
#include "vuColourXYZa.h"
#include "Util.h"


namespace ns_vu1112112 {
using namespace ns_vu1112112;

float ambient[]={			
  82.8f, 91.6f, 93.5f, 86.8f, 104.9f, 117.1f,
  117.8f, 114.9f, 115.9f, 108.8f, 109.4f,
  107.8f, 104.9f, 107.7f, 104.4f, 104.0f,
  100.0f, 96.4f, 95.7f, 88.6f, 90.0f, 89.6f,
  87.6f, 83.3f, 83.7f, 80.0f, 80.2f, 82.2f,
  78.3f, 69.7f, 71.6f
};


//----------------------------------------------------------------------------
//------------------------- The default constructor --------------------------
//----------------------------------------------------------------------------

vu1112112::vu1112112()
{
    m_Camera = new vuPerspectiveCamera();

    m_Camera->setHeight(240);
    m_Camera->setWidth(320);
    numlights = 0;
    nummat = 0;
    diffuse = 2.0f;
    brightness = 2.0f;
    light_scale = 1.0f;
    m_LightDir = vuVector(1,0,0);
    m_ObserverLight = false;
    imgbuf = NULL;
    re_light = re_render = false;
    ColourType c;
    m_TFunc.resize(c.nComponents(),256);
    m_DrawPreview = true;
    m_DoSpecular = false;
    m_Shininess = 20;
    m_Gloss = 3;
}

//----------------------------------------------------------------------------
//------------------------- The copy constructor -----------------------------
//----------------------------------------------------------------------------

vu1112112::vu1112112(const vu1112112& inst) : vu111211(inst)
{
  operator=(inst);
}

//----------------------------------------------------------------------------
//------------------------- The destructor -----------------------------------
//----------------------------------------------------------------------------

vu1112112::~vu1112112()
{
    if(imgbuf) delete [] imgbuf;
    imgbuf = NULL;
    if (m_Camera != NULL) {
      delete m_Camera;
      m_Camera = NULL;
    }
}

//----------------------------------------------------------------------------
//------------------------- The assignment operator --------------------------
//----------------------------------------------------------------------------

vu1112112& vu1112112::operator=(const vu1112112& rhs)
{
    if (this != &rhs)
    {
        vu111211::operator=(rhs);
		Grid = rhs.Grid;
    }
	re_light = re_render = false;
    return *this;
}

//----------------------------------------------------------------------------
void vu1112112::setImageSize(dword sx, dword sy)
{
    m_Camera->setWidth(sx);
    m_Camera->setHeight(sy);
    //m_Camera->setFOV(...);
    ((vuPerspectiveCamera *)m_Camera)->setAspect(float(sx)/sy);
    m_Camera->init();
    spimg.init(m_Camera->getWidth(),m_Camera->getHeight());

    if(imgbuf) delete [] imgbuf;
	    imgbuf = new byte[3*m_Camera->getWidth()*m_Camera->getHeight()];
}

//----------------------------------------------------------------------------
//------------------------- public setViewVectors() --------------------------
//----------------------------------------------------------------------------

void vu1112112::setViewVectors(const vuVector& pos,const vuVector& vrp,
			       const vuVector& up)
{
	m_Camera->setPosition(pos);
	m_Camera->setLookAtVector(pos+vrp);
	m_Camera->setUpVector(up);
	m_Camera->init();
}

//----------------------------------------------------------------------------
//------------------------- public read() ------------------------------------
//----------------------------------------------------------------------------

bool vu1112112::read()
{
    bool success = vu111211::read();
    if (!success) return false;

    bool ret = Grid.load_vol(m_Data,*this);
    if(ret)
      {
	  if (imgbuf) delete [] imgbuf;
	  imgbuf = NULL;
	  //scale_light(.5f,.5f);
	  center=vuVector((float)m_Dim1Size/2,(float)m_Dim2Size/2,
			  (float)m_Dim3Size/2);
	  m_Camera->setPosition(center);
	  m_Camera->translateXYZ(0,0,-(float)m_Dim1Size);
	  setImageSize(320,200);
      }

    re_light = re_render = false;

    pp_gradients = false;
    preprocess();
    return true;
}

//----------------------------------------------------------------------------
//------------------------- public readRaw() ---------------------------------
//----------------------------------------------------------------------------

bool vu1112112::readRaw(void)
{
    dword len;
    ifstream in;

    in.open(m_FileName, ios::in|ios::binary); //|ios::nocreate);
    if (!in.is_open()) return false;

    in.seekg(0, ios::end);
    len = in.tellg();
    in.seekg(0, ios::beg);

    in >> m_Dim1Size >> m_Dim2Size >> m_Dim3Size;
    if (in.fail()) return false;
    m_DataSize = m_Dim1Size*m_Dim2Size*m_Dim3Size;

    m_Data = new byte[m_DataSize];
    in.read((char*)m_Data, m_DataSize);
    if (in.fail()) return false;

    in.close();

    bool ret = Grid.load_vol(m_Data,*this);
    if(ret)
      {
	if(imgbuf) delete [] imgbuf;
	imgbuf = NULL;

	center=vuVector((float)m_Dim1Size/2,(float)m_Dim2Size/2,
			(float)m_Dim3Size/2);
	m_Camera->setPosition(center);
	m_Camera->translateXYZ(0,0,-(float)m_Dim1Size);
	m_Camera->init();
	setImageSize(320,200);
      }
    re_light = re_render = false;

    pp_gradients = false;
    preprocess();

    return true;
}

//----------------------------------------------------------------------------
//------------------------- private preprocess() -----------------------------
//----------------------------------------------------------------------------

void vu1112112::preprocess(void)
{
  // compute normals
  if(!pp_gradients)
    {
      cout<<"computing normals..."<<endl;
      Grid.calculate_gradients();
      pp_gradients=true;
    }

    cout<<"classifying..."<<endl;

#ifdef USE_VUTFUNC
#elif defined(DO_POST_CLASSIFICATION)
    int data;
    for(data=0;data<256;data++)
    {
	float density=0;
	int m;
	TFLUT_scatt[data] = ColourType(0.0f);
	TFLUT_absor[data] = ColourType(0.0f);
	for(m=0;m<nummat;m++)
	{
		if (data>mat[m].low_th &&
			data<=mat[m].high_th  )
		{
			int d =((mat[m].high_th + mat[m].low_th)>>1)-data;
			density = (.5f-(float)fabs((float)d/(mat[m].high_th 
							     -mat[m].low_th)))*2.f;
		}								// maybe we apply other distributions here
		else density = 0.f;
		TFLUT_scatt[data] += mat[m].scattering*density;
		TFLUT_absor[data] += mat[m].absorption*density;
	}
    }
#endif

    Grid.classify(*this);
    cout << "shading..." << endl;
    Grid.shade(*this);
    
}


//----------------------------------------------------------------------------
void vu1112112::shootRays(int xofs, int xinc, int yofs, int yinc)
{
      vuColourXYZa xyza;
      vuColourRGBa rgba;
      //xyza.setNormalSpectrum(m_Light[0]); //HERE-- fixed scaling

      ColourType col;
      ColourType ill(spimg.get_light());
      glBegin(GL_POINTS);
      int i,j;
	  int w=m_Camera->getWidth();
	  int h=m_Camera->getHeight();
      for(j=yofs;j<h && j>=0;j+=yinc)
	{
	  for(i=xofs;i<w;i+=xinc)
	    {
	      vuSampleRay ray(m_Camera->getRay((float)i,(float)j));
	      col = Cast(ray);

	      spimg.set_xy(i,h-j,col);	//coordinate system is upside down
	      
#if defined USE_RGBA
	      //col.ClampTo01();
	      //rgba = col;
	      //*
	      xyza.from(col);
	      //xyza.Normalize();
	      rgba.from(xyza);
	      rgba.clampTo01();
	      //*/
#elif defined USE_SPECTRUM9A
	      col *= ill;
	      col.to(rgba);
	      rgba.clampTo01();
#else
	      col *= ill;
	      //xyza.from(col);
	      //xyza.normalize();
	      //rgba.from(xyza);
	      rgba.from(col);
	      rgba.clampTo01();
#endif
	      if(m_DrawPreview)
		  if(!yofs) {
		      
		      rgba.glColor();
		      glVertex2f(i,h-j);
		  }
	      
	    }
	  //cout << "."<<flush;
	}
      glEnd();
      
      re_light = true;
}

//----------------------------------------------------------------------------
void vu1112112::run(int whatsup, void* data)
{
  //cout<<"running as "<<whatsup<<endl;
  //shootRays(whatsup,2);
  dword oddheight = (m_Camera->getHeight()-1);
  if(!(oddheight & 0x01)) oddheight--;
  if(whatsup)
    shootRays(0,1,oddheight,-2);
  else
    shootRays(0,1,0,2);
  
  m_Mutex[whatsup].unlock();
}


//----------------------------------------------------------------------------
//------------------------- public render() ----------------------------------
//----------------------------------------------------------------------------

void vu1112112::render(void)
{
  vuVector pos = m_Camera->getPosition();
  vuVector dir = m_Camera->getLookAtVector();

  if(re_render)
    {
      use_flat_absorption = true;
#ifndef USE_VUTFUNC
      int m;
      for(m=0;m<nummat && use_flat_absorption;m++)
	use_flat_absorption = mat[m].flat_absorption;
      if(!use_flat_absorption) cout<<"using spectral absorption."<<endl;
#else
      Grid.classify(*this);
#endif

#ifdef SPRC_SINGLEPROCESS
      shootRays();
#else
      m_Mutex[0].lock();
      m_Mutex[1].lock();

	  if(!startThread(0))
		shootRays();
	  else startThread(1);

      m_Mutex[0].lock();
      m_Mutex[1].lock();
      m_Mutex[0].unlock();
      m_Mutex[1].unlock();
#endif
      spimg.create_black_mask();
      re_render = false;
    }

  int mx,my;
  spimg.get_extents(mx,my);

  if(re_light)
    {
	//copy to framebuffer
	if(mx&&my)
	{
	  if(!imgbuf)
	      imgbuf = new byte[3*mx*my];

	  if(!spimg.get_rgb(imgbuf,3*mx*my)) return;
	} else delete [] imgbuf;
	re_light = false;
    }

  //copy to screen
  if(imgbuf) {
      glRasterPos2i(0,0);
      glPixelStorei(GL_UNPACK_ALIGNMENT,1);
      glDrawPixels(mx, my, GL_RGB, GL_UNSIGNED_BYTE, (void*)imgbuf );
  }
}

//----------------------------------------------------------------------------
//------------------------- public initOpenGL() ------------------------------
//----------------------------------------------------------------------------

void vu1112112::initOpenGL(void)
{
  //glEnable(GL_SECRET_FEATURE);
}

// Cast a ray through the grid
ColourType vu1112112::Cast(vuSampleRay& Sray)
{
    //#define USE_SMPDST		//enable if smpdst != 1.0f OR NOT???
    Sray.SamplingDistance(1.0f);
    if(!Sray.attachToGrid(Grid)) return Background;
    ColourType colour(0.0f);
    ColourType aabsorption(1.0f);
    DatPnt sample;
    int steps=1;
    bool terminate = false;

    while(!terminate && Sray.advanceRay())
    {
	if(m_DoSpecular)
	    Sray.getSamplePointWithGradient(sample);
	else
	    Sray.getSamplePoint(sample);
	if(!(sample.data == 0)) 
	{
	    //use the position information
	    bool matters, xray=false;
#ifdef USE_VUTFUNC
	    ColourType emissive(m_TFunc[sample.data]);
	    
	    ColourType absorption(emissive.getAlpha());
	    if(absorption[0] < 0.0001)
		matters = false;
	    else
		matters = true;
#elif defined(DO_POST_CLASSIFICATION)
	    ColourType& emissive = TFLUT_scatt[sample.data];
	    ColourType& absorption = TFLUT_absor[sample.data];
	
	    if(use_flat_absorption && absorption[0] < 0.001)
		matters = false;
	    else
		matters = true;
#else
	    ColourType emissive(0.0f);
	    ColourType absorption(0.0f);
	    matters=false;
	    int m;
	    for(m=0;m<MAT_NUM_MATERIALS;m++)
	    {
		if(sample.density[m] > 0.01)
		{
		    emissive += mat[m].scattering*(sample.density[m]);
		    if(use_flat_absorption)
			absorption[0] += mat[m].absorption[0]*
			    (sample.density[m]);
		    else
			absorption += mat[m].absorption*(sample.density[m]);
		    matters=true;
		    if(mat[m].xray) {
			xray=true;
		    }
		}
	}
#endif
	    if(matters)
	    {
#ifdef USE_SMPDST	
		if(use_flat_absorption)
		    absorption[0] *= 1;//Sray.SamplingDistance();
		else
		    absorption *= Sray.SamplingDistance();
#endif
		
		if(m_DoSpecular && sample.length>1)
		{
		    //grad has length 128
		    vuVector h((float)sample.grad[0],
			       (float)sample.grad[1],
			       (float)sample.grad[2]);
		    vuVector v(Sray.m_Direction);
		    h*=h.dot(v);
		    h*=(2.0/(128*128));
		    vuVector r =  v - h;
		    r.makeUnit();
		    float spec=r.dot(m_LightDir);
		    //adding specular contribution to voxel intensity
		    //the two values need to be adjusted...
		    if(spec>0)
		    {
			spec=(float)pow((r.dot(m_LightDir)), m_Shininess);
			sample.illum += spec*m_Gloss;
		    }
		}

		if(use_flat_absorption) {
		    colour += emissive*absorption[0]*aabsorption[0]*sample.illum;
		    if(!xray)
			aabsorption[0] *= 1-absorption[0];
		} else {
		    colour += emissive*absorption*aabsorption*sample.illum;
		    if(!xray)
			aabsorption *= absorption.oneMinus();
		}
		
		float mabso;
		if(use_flat_absorption)
		    mabso = aabsorption[0];
		else
		    mabso = aabsorption.maxComponent();
		if(mabso<0.05) 
		{
		    //cout<<"t";
		    terminate = true;
		}
		//#endif
	    }
	    steps++;
	} //end of if skipCell
    } //end of while
    if(!terminate) 
	if(use_flat_absorption)
	    colour += Background*(aabsorption[0]);
	else
	    colour += Background*(aabsorption);
    return colour;
}

void vu1112112::setLight(const vuColour31a &light)
{
  ColourType lc(light);
  spimg.set_light(lc);
  re_light = true;
}

/*
void vu1112112::adjust_camera(float px, float py, float pz, float ux, float uy, float uz)
{
  m_Camera->setPosition(vuVector(px,py,pz));
  m_Camera->setUpVector(vuVector(ux,uy,uz));
  m_Camera->Init();
}

bool vu1112112::get_extents(int &mx, int &my, int &mz)
{

	mx = Grid.maxX;
	my = Grid.maxY;
	mz = Grid.maxZ;
	return true;
}
*/
bool vu1112112::load_scene(const char *filename)
{
	numlights = 0;
	nummat = 0;
	Parser parser;
	cout << "loading scene setup: "<<filename<<endl;
	if(!parser.Parse(filename, *this)) return false;
	
	//Grid.classify(*this);
	preprocess();
	delete [] imgbuf;
	imgbuf = NULL;
	m_Camera->setPosition(center);
	m_Camera->translateXYZ(0,0,-(float)m_Dim1Size);
	m_Camera->init();
	return true;
}

bool vu1112112::add_light(ColourType &lc)
{
	if(numlights<NUM_LIGHTS)
	{
		m_Light[numlights++] = lc;
		return true;
	} else return false;
}

bool vu1112112::add_material(Material &mat)
{
	if(nummat<MAT_NUM_MATERIALS)
	{
		this->mat[nummat++] = mat;
		return true;
	} else return false;
}

void vu1112112::doRefresh()
{
  re_render = true;
}

vuVector vu1112112::getCenter()
{
  return center;
}

vuImage* vu1112112::getImage ()

{
	rgbimg.set_data (imgbuf);

	return &rgbimg;
}

} // end of namespace
