//
// File:   Parser.h
// Author: Steve Kilthau
// Date:   August, 1999
// 

// Modified by Steven Bergner, Aug 2001

// This file defines a parser class.  The parser is
// used to read scene-files describing objects in a
// given scene.
//

#ifndef _PARSER_H_
#define _PARSER_H_

#include "vuVector.h"
#include "Material.h"
#include "vuPerspectiveCamera.h"
#include "spectral.h"
#include "Util.h"

namespace ns_vu1112112 {
using namespace ns_vu1112112;

typedef enum{cLight, cColour, cBackground} ColourUsage;

/** This defines a parser class.  
    The parser is used to read scene-files describing objects in a
    given scene.
    Hopefully, this class expires when we have a transfer function panel
    with support for various colour models and load/save facility.
*/
class Parser
{
public:
    Parser();    //!< Default constructor - set initial values
    ~Parser();   //!< Destructor

    /** Parse the file represented by "filename" and 
      \return the resulting scene in "scene".
    */
    bool Parse(const char* filename, vu1112112& scene);

private:

    /** Check if an error was encountered during parsing.
	Report the position in the file that was successfully
	parsed up to. */
    void CheckError(void);

private:

    //! Try to read the static text given by "s" from the buffer
    bool ReadString(char* s);

    //! Try to read a string prefixed by "prefix" from the buffer
    bool ReadString(char* prefix, char* s);

    //! Try to read a number prefixed by "prefix" from the buffer
    bool ReadNumber(char* prefix, float& t);

    /** Try to read a vector prefixed by "prefix" from the buffer.
	The vector read must have 3 members. */
    bool ReadVector3(char* prefix, vuVector& v);

    /** Try to read a vector prefixed by "prefix" from the buffer.
	The vector read must have 4 members. */
    bool ReadvuVector(char* prefix, vuVector& v);

    /** Try to read an vuColourRGBa prefixed by "prefix" from the buffer.
	Only read the RGB values. */
    bool ReadRGB(char* prefix, vuColourRGBa& r);

    /** Try to read an vuColourRGBa prefixed by "prefix" from the buffer.
	Read all of the vuColourRGBa values. */
    bool ReadvuColourRGBa(char* prefix, vuColourRGBa& r);

    /* Try to read a vuColour31a prefixed by "prefix" from the buffer.
       Do not read the alpha channel. */
    bool ReadSpectrum31(char* prefix, vuColour31a& s);

    /** Try to read a vuColour31a prefixed by "prefix" from the buffer.
	Read all channels. */
    bool ReadvuColour31a(char* prefix, vuColour31a& s);

    /** Try to read a vuColour7a prefixed by "prefix" from the buffer.
	Do not read the alpha channel. */
    bool ReadSpectrum7(char* prefix, vuColour7a& s);

    /** Try to read a vuColour7a prefixed by "prefix" from the buffer.
	Read all channels. */
    bool ReadvuColour7a(char* prefix, vuColour7a& s);

    /** Try to read a vuColour9a prefixed by "prefix" from the buffer.
	Do not read the alpha channel. */
    bool ReadSpectrum9(char* prefix, vuColour9a& s);

    /** Try to read a vuColour9a prefixed by "prefix" from the buffer.
	Read all channels. */
    bool ReadvuColour9a(char* prefix, vuColour9a& s);

    /** Try to read a colour prefixed by "prefix" from the buffer.
	Do not read the alpha channel. */
    bool ReadColourType(char* prefix, ColourType& c, ColourUsage u);

    /** Try to read an Material from the buffer. */
    bool ReadMaterial(vu1112112& r);

    /** Try to read the "general" block from the buffer */
    bool ReadGeneral(vu1112112& s);

    /** Try to read the "image" block from the buffer */
    bool ReadTarga(vu1112112& s);

    /** Try to read a "light" block from the buffer */
    bool ReadLight(vu1112112& scene);

private:
    void FixName(char* str);	//!< crop the string

private:
    /** saves the ambient light of the scene
	This is used to normalize spectra that are read with this parser.
	We would use it if the colour model was RGB which it is not, the case.
    */
    vuColour31a m_Ambient;
    bool        m_ReadAmbient;	//!< a flag

private:
    char* m_Buffer;     //!< The character buffer (contains input file text)
    int   m_Pos;        //!< Current position in the character buffer
};

} // end of namespace
#endif
