//
// File:   Parser.cpp
// Author: Steve Kilthau
// Date:   August, 1999
// 
// This file implements a parser class.  The parser is
// used to read scene-files describing objects in a 
// given scene.
//

#include <stdio.h>
#include <string.h>
#include "Parser.h"
#include "spectral.h"
#include "vuColourRGBa.h"
#include "vuColour31a.h"
#include "vuColour7a.h"
#include "vuColour9a.h"
#include "vuColourXYZa.h"
#include "Util.h"

namespace ns_vu1112112 {
using namespace ns_vu1112112;

// Default constructor - set initial values
Parser::Parser()
{
    m_Buffer = 0;
    m_Pos    = 0;
    m_ReadAmbient = false;
}

// Destructor
Parser::~Parser()
{
    if (m_Buffer)
        delete [] m_Buffer;
}

// Parse the file represented by "filename" and 
// return the resulting scene in "scene".
bool Parser::Parse(const char* filename, vu1112112& scene)
{
    FILE* ifp;

    ifp = fopen(filename, "rb");
    if (ifp)
    {
        int stats[5] = {0,0,0,0,0};
        int length;

        // Get the buffer
		// SB: Here we could add a feature for comment lines
        fseek(ifp, 0, SEEK_END);
        length = ftell(ifp);
        fseek(ifp, 0, SEEK_SET);
        m_Buffer = new char[length+1];
        fread(m_Buffer, 1, length, ifp);
        m_Buffer[length] = 0;
        fclose(ifp);

        // Read the scene
        do
        {
            stats[4] = 0;
            if (!stats[0] && ReadGeneral(scene)){
                stats[0] = 1; stats[4]++;
            }
            if (!stats[1] && ReadTarga(scene)){
                stats[1] = 1; stats[4]++;
            }
            if (!stats[2] && ReadMaterial(scene)){
                //stats[2] = 1; 
				stats[4]++;
            }
            if (!stats[3] && ReadLight(scene)){
                //stats[3] = 1; 
				stats[4]++;
            }
        }while(stats[4]);

        CheckError();
    }

    return true;
}

// Try to read the "general" block from the buffer
bool Parser::ReadGeneral(vu1112112& s)
{
    int stats[7] = {0,0,0,0,0,0,0};
    ColourType c;
    vuVector v;
    float n;

    if (ReadString(" general {"))
    {
		Material mat;
        do
        {
            stats[6] = 0;
            if (!stats[0] && ReadColourType(" background", c, cColour)){
                stats[0] = 1; stats[6]++; s.Background = c;
            }
            if (!stats[1] && ReadNumber(" brightness", n)){
                stats[1] = 1; stats[6]++; s.brightness = n;
            }
            if (!stats[2] && ReadVector3(" light_dir", v)){
                stats[2] = 1; stats[6]++; s.m_LightDir = v.normalize();
            }
            if (!stats[3] && ReadNumber(" diffuse", n)){
                stats[3] = 1; stats[6]++; s.diffuse = n;
            }
            if (!stats[4] && ReadNumber(" light_scale", n)){
                stats[4] = 1; stats[6]++; s.light_scale = n;
            }
            if (ReadString(" }")){
                stats[6]++; break;
            }
        }while(stats[6]);

        if (stats[6])
            return true;
    }
    return false;
}

// Check if an error was encountered during parsing.
// Report the position in the file that was successfully
// parsed up to.
void Parser::CheckError(void)
{
    int pos = m_Pos;

    // Skip trailing whitespace
    ReadString(" ");

    // If not at the end of the buffer then there
    // has been a parse error.
    if (m_Buffer[m_Pos])
    {
        char error[1024];
        char last=0;
        int  line=1;
        int  i=0;

        // Count the number of lines parsed
        while(i < pos)
        {
            if (m_Buffer[i]==10)
            {
                if (last==13)
                {
                    last = 0;
                    ++i;
                }
                else
                {
                    ++line;
                    last = m_Buffer[i++];
                }
            }
            else if (m_Buffer[i]==13)
            {
                if (last==10)
                {
                    last = 0;
                    ++i;
                }
                else
                {
                    ++line;
                    last = m_Buffer[i++];
                }
            }
            else
                last = m_Buffer[i++];
        }

        // Find the position in the current line
        while((i>0) && (m_Buffer[i]!=10) && (m_Buffer[i]!=13))
            --i;
        pos -= i;
        if ((pos==0) && (line>1))
            ++line;

        sprintf(error, "PARSER - Parsed successfully to line %i, character %i.", line, pos);
        throw error;
    }
}

// Try to read the static text given by "s" from the buffer
bool Parser::ReadString(char* s)
{
    char str[1024];
    int ret, len=0;
    strcpy(str, s);
    strcat(str, "%n");
    ret = sscanf(&m_Buffer[m_Pos], str, &len);
    if ((ret!=EOF) && (len!=0))
    {
        m_Pos += len;
        return true;
    }
    return false;
}

// Try to read a string prefixed by "prefix" from the buffer
bool Parser::ReadString(char* prefix, char* s)
{
    char str[1024];
    int i=0;
    strcpy(str, prefix);
    strcat(str, " \"");
    if (ReadString(str))
    {
        while((m_Buffer[m_Pos]!='"') && m_Buffer[m_Pos])
            s[i++] = m_Buffer[m_Pos++];
        s[i] = 0;
        if (m_Buffer[m_Pos]=='"')
        {
            ++m_Pos;
            return true;
        }
    }
    return false;
}

// Try to read a number prefixed by "prefix" from the buffer
bool Parser::ReadNumber(char* prefix, float& t)
{
    char str[1024];
    int ret, len=0;
    strcpy(str, prefix);
    strcat(str, " %f%n");
    ret = sscanf(&m_Buffer[m_Pos], str, (float*)&t, &len);
    if ((ret == 1) && (len!=0))
    {
        m_Pos += len;
        return true;
    }
    return false;
}

// Try to read a vector prefixed by "prefix" from the buffer.
// The vector read must have 3 members.
bool Parser::ReadVector3(char* prefix, vuVector& v)
{
    char str[1024];
    int pos;

    pos = m_Pos;
    strcpy(str, prefix);
    strcat(str, " <");
    if (ReadNumber(str, v[0]))
        if (ReadNumber(" ,", v[1]))
            if (ReadNumber(" ,", v[2]))
                if (ReadString(" >"))
                    return true;
    m_Pos = pos;
    return false;
}

// Try to read a vector prefixed by "prefix" from the buffer.
// The vector read must have 4 members.
bool Parser::ReadvuVector(char* prefix, vuVector& v)
{
    char str[1024];
    int pos;

    pos = m_Pos;
    strcpy(str, prefix);
    strcat(str, " <");
    if (ReadNumber(str, v[0]))
        if (ReadNumber(" ,", v[1]))
            if (ReadNumber(" ,", v[2]))
                if (ReadNumber(" ,", v[3]))
                    if (ReadString(" >"))
                        return true;
    m_Pos = pos;
    return false;
}

// Try to read an vuColourRGBa prefixed by "prefix" from the buffer.
// Only read the RGB values.
bool Parser::ReadRGB(char* prefix, vuColourRGBa& r)
{
    char str[1024];
    int pos;

    pos = m_Pos;
    strcpy(str, prefix);
    strcat(str, " <");
    if (ReadNumber(str, r[0]))
        if (ReadNumber(" ,", r[1]))
            if (ReadNumber(" ,", r[2]))
                if (ReadString(" >"))
                    return true;
    m_Pos = pos;
    return false;
}

// Try to read an vuColourRGBa prefixed by "prefix" from the buffer.
// Read all of the vuColourRGBa values.
bool Parser::ReadvuColourRGBa(char* prefix, vuColourRGBa& r)
{
    char str[1024];
    int pos;

    pos = m_Pos;
    strcpy(str, prefix);
    strcat(str, " <");
    if (ReadNumber(str, r[0]))
        if (ReadNumber(" ,", r[1]))
            if (ReadNumber(" ,", r[2]))
                if (ReadNumber(" ,", r[3]))
                    if (ReadString(" >"))
                        return true;
    m_Pos = pos;
    return false;
}

// Try to read a vuColour31a prefixed by "prefix" from the buffer.
// Do not read the alpha channel.
bool Parser::ReadSpectrum31(char* prefix, vuColour31a& s)
{
    char str[1024];
    bool good;
    int pos;
    int i;

    pos = m_Pos;
    strcpy(str, prefix);
    strcat(str, " <");
    if (ReadNumber(str, s[0]))
    {
        good = true;
        for(i=1;(i<31)&&good;++i)
            good = ReadNumber(" ,", s[i]);
        if (good && ReadString(" >"))
            return true;
    }
    m_Pos = pos;
    return false;
}

// Try to read a vuColour31a prefixed by "prefix" from the buffer.
// Read all channels.
bool Parser::ReadvuColour31a(char* prefix, vuColour31a& s)
{
    char str[1024];
    bool good;
    int pos;
    int i;

    pos = m_Pos;
    strcpy(str, prefix);
    strcat(str, " <");
    if (ReadNumber(str, s[0]))
    {
        good = true;
        for(i=1;(i<32)&&good;++i)
            good = ReadNumber(" ,", s[i]);
        if (good && ReadString(" >"))
            return true;
    }
    m_Pos = pos;
    return false;
}

// Try to read a vuColour7a prefixed by "prefix" from the buffer.
// Do not read the alpha channel.
bool Parser::ReadSpectrum7(char* prefix, vuColour7a& s)
{
    char str[1024];
    bool good;
    int pos;
    int i;

    pos = m_Pos;
    strcpy(str, prefix);
    strcat(str, " <");
    if (ReadNumber(str, s[0]))
    {
        good = true;
        for(i=1;(i<7)&&good;++i)
            good = ReadNumber(" ,", s[i]);
        if (good && ReadString(" >"))
            return true;
    }
    m_Pos = pos;
    return false;
}

// Try to read a vuColour7a prefixed by "prefix" from the buffer.
// Read all channels.
bool Parser::ReadvuColour7a(char* prefix, vuColour7a& s)
{
    char str[1024];
    bool good;
    int pos;
    int i;

    pos = m_Pos;
    strcpy(str, prefix);
    strcat(str, " <");
    if (ReadNumber(str, s[0]))
    {
        good = true;
        for(i=1;(i<8)&&good;++i)
            good = ReadNumber(" ,", s[i]);
        if (good && ReadString(" >"))
            return true;
    }
    m_Pos = pos;
    return false;
}

// Try to read a vuColour9a prefixed by "prefix" from the buffer.
// Do not read the alpha channel.
bool Parser::ReadSpectrum9(char* prefix, vuColour9a& s)
{
    char str[1024];
    bool good;
    int pos;
    int i;

    pos = m_Pos;
    strcpy(str, prefix);
    strcat(str, " <");
    if (ReadNumber(str, s[0]))
    {
        good = true;
        for(i=1;(i<9)&&good;++i)
            good = ReadNumber(" ,", s[i]);
        if (good && ReadString(" >"))
            return true;
    }
    m_Pos = pos;
    return false;
}

// Try to read a vuColour9a prefixed by "prefix" from the buffer.
// Read all channels.
bool Parser::ReadvuColour9a(char* prefix, vuColour9a& s)
{
    char str[1024];
    bool good;
    int pos;
    int i;

    pos = m_Pos;
    strcpy(str, prefix);
    strcat(str, " <");
    if (ReadNumber(str, s[0]))
    {
        good = true;
        for(i=1;(i<10)&&good;++i)
            good = ReadNumber(" ,", s[i]);
        if (good && ReadString(" >"))
            return true;
    }
    m_Pos = pos;
    return false;
}

// Try to read a colour prefixed by "prefix" from the buffer.
// Do not read the alpha channel.
bool Parser::ReadColourType(char* prefix,
                            ColourType& c,
                            ColourUsage u)
{
#if defined USE_vuColourRGBa
    if (u == cColour)
    {
        if (ReadvuColourRGBa(prefix, c))
        {
            return true;
        }
        else
        {
            vuColour31a s;
            vuColourXYZa x;
            if (ReadvuColour31a(prefix, s))
            {
                if (m_ReadAmbient)
                    s *= m_Ambient;
                x.From(s);
                if (m_ReadAmbient)
                    x.SetNormalSpectrum(m_Ambient);
                x.Normalize();
                c.From(x);
                c.ClampTo1();
                return true;
            }
        }
    }
    else if (u == cLight || u == cAmbient || u == cBackground)
    {
        if (ReadRGB(prefix, c))
        {
            return true;
        }
        else
        {
            vuColour31a s;
            vuColourXYZa x;
            if (ReadSpectrum31(prefix, s))
            {
                if (u == cAmbient)
                {
                    m_Ambient = s;
                    m_ReadAmbient = true;
                }
                if (m_ReadAmbient && u == cBackground)
                    s *= m_Ambient;
                x.From(s);
                if (m_ReadAmbient)
                    x.SetNormalSpectrum(m_Ambient);
                x.Normalize();
                c.From(x);
                c.ClampTo1();
                return true;
            }
        }
    }
    return false;
#elif defined USE_SPECTRUM31A
    if (u == cColour)
        return ReadvuColour31a(prefix, c);
    return ReadSpectrum31(prefix, c);
#elif defined USE_SPECTRUM7A
    if (u == cColour)
    {
        if (ReadvuColour7a(prefix, c))
        {
            return true;
        }
        else
        {
            vuColour31a s;
            if (ReadvuColour31a(prefix, s))
            {
                c.from(s);
                return true;
            }
        }
    }
    else if (u == cLight)
    {
        if (ReadSpectrum7(prefix, c))
        {
            return true;
        }
        else
        {
            vuColour31a s;
            if (ReadSpectrum31(prefix, s))
            {
                c.from(s);
                return true;
            }
        }
    }
    return false;
#elif defined USE_SPECTRUM9A
    if (u == cColour)
    {
        if (ReadvuColour9a(prefix, c))
        {
            return true;
        }
        else
        {
            vuColour31a s;
            if (ReadvuColour31a(prefix, s))
            {
                c.from(s);
                return true;
            }
        }
    }
    else if (u == cLight)
    {
        if (ReadSpectrum9(prefix, c))
        {
            return true;
        }
        else
        {
            vuColour31a s;
            if (ReadSpectrum31(prefix, s))
            {
                c.from(s);
                return true;
            }
        }
    }
    return false;
#endif
}

// Try to read an Material from the buffer.
bool Parser::ReadMaterial(vu1112112 &r)
{
    int stats[8] = {0,0,0,0,0,0,0,0};
    ColourType c;
    float n;
    if (ReadString(" material {"))
    {     
        Material m;
	m.scattering=1;
	m.absorption=1;
        do
        {
            stats[7] = 0;
            if (!stats[0] && ReadColourType(" colour", c, cColour)){
                stats[0] = 1; stats[7]++; m.scattering = c;
            }
            if (!stats[1] && ReadColourType(" absorption", c, cColour)){
                stats[1] = 1; stats[7]++; m.absorption = c;
            }
            if (!stats[2] && ReadNumber(" low_th", n)){
                stats[2] = 1; stats[7]++; m.low_th = (int)n;
            }
            if (!stats[3] && ReadNumber(" high_th", n)){
                stats[3] = 1; stats[7]++; m.high_th = (int)n;
            }
            if (!stats[4] && ReadNumber(" xray", n)){
                stats[4] = 1; stats[7]++; m.xray = (n == 1);
            }
	    if (ReadString(" }")){
                stats[7]++; break;
            }
        }while(stats[7]);

        if (stats[7])
		{
		  m.check_absorption();
		  r.add_material(m);
		  return true;
		}
    }
    return false;
}

// Try to read the "image" block from the buffer
bool Parser::ReadTarga(vu1112112& s)
{
    int stats[4] = {0,0,0,0};
    char c[64];
    float n;

    if (ReadString(" image {"))
    {
        do
        {
            stats[3] = 0;
            if (!stats[0] && ReadString(" name", c)){
                FixName(c);
                stats[0] = 1; stats[3]++; //s.Image.Name(c);
            }
            if (!stats[1] && ReadNumber(" width", n)){
                stats[1] = 1; stats[3]++;
                s.m_Camera->setWidth((int)n);
            }
            if (!stats[2] && ReadNumber(" height", n)){
                stats[2] = 1; stats[3]++;
                s.m_Camera->setHeight((int)n);
            }
            if (ReadString(" }")){
                stats[3]++; break;
            }
        }while(stats[3]);

        if (stats[3])
            return true;
    }
    return false;
}

// Try to read a "light" block from the buffer
bool Parser::ReadLight(vu1112112& scene)
{
    int stats[4] = {0,0,0,0};
    vuVector v;
    ColourType c;

    if (ReadString(" light {"))
    {
        do
        {
            stats[3] = 0;
            if (!stats[1] && ReadColourType(" colour", c, cLight)){
                stats[1] = 1; stats[3]++; scene.add_light(c);
            }
            if (ReadString(" }")){
                stats[3]++; break;
            }
        }while(stats[3]);

        if (stats[3])
        {
            return true;
        }
    }
    return false;
}

void Parser::FixName(char* str)
{
    char ext[32];
    int i=0;

    while(str[i] != '.' && str[i])
        ++i;
    strcpy(ext, &str[i]);
    str[i] = 0;
#if defined USE_vuColourRGBa
    strcat(str, "_rgb");
#elif defined USE_SPECTRUM31A
    strcat(str, "_31");
#elif defined USE_SPECTRUM7A
    strcat(str, "_7");
#elif defined USE_SPECTRUM9A
    strcat(str, "_9");
#endif
    strcat(str, ext);
}

} // end of namespace
