// DatGrid.cpp: Implementierung der Klasse DatGrid.
//
//////////////////////////////////////////////////////////////////////

#include "spectral.h"
#include "DatGrid.h"
#include "fstream.h"
#include "iostream.h"
#include "vuSampleRay.h"
#include "Util.h"
#include <math.h>

#define RET_OK	 0
#define RET_ERR	-1

namespace ns_vu1112112 {
using namespace ns_vu1112112;

extern float light[];
extern float ambient[];

//////////////////////////////////////////////////////////////////////
// Konstruktion/Destruktion
//////////////////////////////////////////////////////////////////////

DatGrid::DatGrid()
{
	maxX=maxY=maxZ=layXY = 0;
	vol = NULL;
}

DatGrid::~DatGrid()
{
	free_vol_mem();
}

int DatGrid::free_vol_mem()
{
	if(vol)
	{
		delete [] vol;
		vol = NULL;
	}

	return RET_OK;
}

// allocates memory according to axis extents
int DatGrid::get_vol_mem()
{
	free_vol_mem();
	vol = new DatPnt[maxX*maxY*maxZ];
	if(!vol) return RET_ERR;

	return RET_OK;
}

// initialises the grid
int DatGrid::init(int maxx, int maxy, int maxz)
{
	maxX = maxx; maxY = maxy; maxZ = maxz;
	layXY = maxX*maxY;
	size = layXY*maxZ;
	m_C0 = vuVector(0.0);
	m_C1 = vuVector((float)maxx-1,(float)maxy-1,(float)maxz-1);
	return get_vol_mem();
}

void DatGrid::distributeIllum(ColourType &il)
{
  //that is for a seperate illumination step tracing shadows or even spectral 
  //absorptions
  //by now we only use local illumination
}

float DatGrid::clipRay(vuSampleRay &r, bool bidirectional, int& m_Side) const
{
    vuVector pos;
    vuVector dir;
    float X;
    float Y;
    float Z;
    float tMin=0;
    float t;

    m_Side = -1;

    pos = r.m_Position;
    dir = r.m_Direction;

    // Test the X0 wall
    t = SnapToZero((m_C0[0] - pos[0]) / dir[0]);
    if (t > 0)
    {
        Y = pos[1] + t * dir[1];
        Z = pos[2] + t * dir[2];
        if ((Y >= m_C0[1]) && (Y <= m_C1[1]) &&
            (Z >= m_C0[2]) && (Z <= m_C1[2]) &&
			(bidirectional || t>=0))
        {
            tMin = t;
            m_Side = 0;
        }
    }

    // Test the X1 wall
    t = SnapToZero((m_C1[0] - pos[0]) / dir[0]);
    if (t > 0)
    {
        Y = pos[1] + t * dir[1];
        Z = pos[2] + t * dir[2];
        if ((Y >= m_C0[1]) && (Y <= m_C1[1]) &&
            (Z >= m_C0[2]) && (Z <= m_C1[2]) &&
			(bidirectional || t>=0))
        {
            if ((m_Side==-1) || (t<tMin))
            {
                tMin = t;
                m_Side = 1;
            }
        }
    }

    // Test the Y0 wall
    t = SnapToZero((m_C0[1] - pos[1]) / dir[1]);
    if (t > 0)
    {
        X = pos[0] + t * dir[0];
        Z = pos[2] + t * dir[2];
        if ((X >= m_C0[0]) && (X <= m_C1[0]) &&
            (Z >= m_C0[2]) && (Z <= m_C1[2]) &&
			(bidirectional || t>=0))
        {
            if ((m_Side==-1) || (t<tMin))
            {
                tMin = t;
                m_Side = 2;
            }
        }
    }

    // Test the Y1 walla
    t = SnapToZero((m_C1[1] - pos[1]) / dir[1]);
    if (t > 0)
    {
        X = pos[0] + t * dir[0];
        Z = pos[2] + t * dir[2];
        if ((X >= m_C0[0]) && (X <= m_C1[0]) &&
            (Z >= m_C0[2]) && (Z <= m_C1[2]) &&
			(bidirectional || t>=0))
        {
            if ((m_Side==-1) || (t<tMin))
            {
                tMin = t;
                m_Side = 3;
            }
        }
    }

    // Test the Z0 wall
    t = SnapToZero((m_C0[2] - pos[2]) / dir[2]);
    if (t > 0)
    {
        X = pos[0] + t * dir[0];
        Y = pos[1] + t * dir[1];
        if ((X >= m_C0[0]) && (X <= m_C1[0]) &&
            (Y >= m_C0[1]) && (Y <= m_C1[1]) &&
			(bidirectional || t>=0))
        {
            if ((m_Side==-1) || (t<tMin))
            {
                tMin = t;
                m_Side = 4;
            }
        }
    }

    // Test the Z1 wall
    t = SnapToZero((m_C1[2] - pos[2]) / dir[2]);
    if (t > 0)
    {
        X = pos[0] + t * dir[0];
        Y = pos[1] + t * dir[1];
        if ((X >= m_C0[0]) && (X <= m_C1[0]) &&
            (Y >= m_C0[1]) && (Y <= m_C1[1]) &&
			(bidirectional || t>=0))
        {
            if ((m_Side==-1) || (t<tMin))
            {
                tMin = t;
                m_Side = 5;
            }
        }
    }
	return tMin; 
}

void DatGrid::createSphere()
{
    int i,j,k;
	int radius = maxX/2;
    for(i=0;i<maxZ;i++)
      for(j=0;j<maxY;j++)
        for(k=0;k<maxX;k++)
            {
                float d=(float)k-(maxX/2);
                float dist = d*d;
                d=(float)j-(maxY/2);
                dist += d*d;
                d=(float)i-(maxZ/2);
                dist += d*d;
                dist = (float)sqrt((double)dist);

		float v=0;
                if (dist<radius)
                    v= (float)sqrt((double)(radius-dist)/radius);
//					v=1.0f;

				vol[k+j*maxX+i*layXY].illum = 1.f;
#ifndef DO_POST_CLASSIFICATION
				if(j<maxY/2) vol[k+j*maxX+i*layXY].density[2] = v;			//mat1
				else if(k<maxX/2) vol[k+j*maxX+i*layXY].density[0] = v;		//mat2
				else vol[k+j*maxX+i*layXY].density[1] = v;					//mat3
#else
				vol[k+j*maxX+i*layXY].data = (int)(255*v);
#endif
            }
//	calculate_gradients();
//	classify();

}

/*---------------------------------------------------------------------------*/
/* load .vol file */
bool DatGrid::load_vol(byte* data, vu1112112 &r)
{
	if (!data) return false;
	maxX = r.m_Dim1Size;
	maxY = r.m_Dim2Size;
	maxZ = r.m_Dim3Size;
	//f.eatwhite();
	init(maxX,maxY,maxZ);
	int c;
	for(c=0;c<size;c++)
	  {
	    vol[c].data = data[c];
	  }

	return true;
}

void DatGrid::shade(vu1112112 &r)
{
  int c;
  for(c=0;c<size;c++)
  {
      vol[c].shade(r);
  }
}

void DatGrid::classify(vu1112112 &r)
{
//   int m;
//   for(m=0;m<r.nummat;m++)
//     cout<<"ID "<<m<<": "<<r.mat[m].low_th<<" - "<<r.mat[m].high_th;
//   cout<<endl;
  int c;
  for(c=0;c<size;c++)
  {
      vol[c].classify(r);
  }

#if defined (USE_VUTFUNC)
  for(c=0;c<size-layXY-maxX-1;c++)
  {
      if(vol[c].getFlag() == DatPnt::Invisible &&
	 vol[c+1].getFlag() == DatPnt::Invisible &&
	 vol[c+maxX].getFlag() == DatPnt::Invisible &&
	 vol[c+1+maxX].getFlag() == DatPnt::Invisible &&
	 vol[c+layXY].getFlag() == DatPnt::Invisible &&
	 vol[c+1+layXY].getFlag() == DatPnt::Invisible &&
	 vol[c+maxX+layXY].getFlag() == DatPnt::Invisible &&
	 vol[c+1+maxX+layXY].getFlag() == DatPnt::Invisible)
      {
	  vol[c].setFlag(DatPnt::SkipCell);
      }
  }
#endif 
}


void DatGrid::calculate_gradients()
{
    int i,j,k;
    int offset=0;
    for(k=0;k<maxZ;k++)
      for(j=0;j<maxY;j++)
		for(i=0;i<maxX;i++)
		  {
			  DatPnt *cc = vol + offset;
			  int g[3];
			  
			  g[0]= (i    ? (int)vol[offset-1].data:0) -
			  ((i<maxX-1) ? (int)vol[offset+1].data:0);
			  g[1]= (j    ? (int)vol[offset-maxX].data:0)
			  -((j<maxY-1) ? (int)vol[offset+maxX].data:0);
			  g[2]= (k    ? (int)vol[offset-layXY].data:0)
			  -((k<maxZ-1) ? (int)vol[offset+layXY].data:0);

			  float len= (float)sqrt(
			      (double)(g[0]*g[0] + 
				       g[1]*g[1] + 
				       g[2]*g[2]));
			  if(len!=0.0f)             // normalize to length 128
			  {			  
			      cc->grad[0] = (int)((g[0]<<7)/len);
			      cc->grad[1] = (int)((g[1]<<7)/len);
			      cc->grad[2] = (int)((g[2]<<7)/len);
			  } else cc->grad[0] = cc->grad[1] = cc->grad[2] = 0;
			  cc->length= len>255.0f ? 255 : (char)len;

			  offset++;
		  }
}

} // end of namespace
