#include <string.h>
#include <stdio.h>
#include <iostream.h>
#include "SimpleDefs.h"
#include "Volume.h"
#include "vuVector.h"
#include "vuMatrix.h"
#include "Image_io.h"
#include "Transform.h"

#define ODD(x)	((x)&1)

using namespace SpecFVRNS;


Volume::Volume() : m_XStep(1.0f, 0.0f, 0.0f),
				m_YStep(0.0f, 1.0f, 0.0f),
				m_XAxis(1.0f, 0.0f, 0.0f),
				m_YAxis(0.0f, 1.0f, 0.0f),
				m_ZAxis(0.0f, 0.0f, 1.0f)
{
	m_Volume  = 0;
	m_Slice   = 0;
	m_Image   = 0;
	m_Scale   = 1.0f;
	m_Bias    = 0.0f;

	m_Wrap    = 0;

	m_CurImage = 0;

	m_ImageStep = 0;
	m_SlicePtr = 0;

	SpecFVRNS::initTransforms();

#ifndef FVR_NO_SLICE_COPY
	m_SSlice = 0;
#endif
}

Volume::Volume(Volume& inst)
{
}

Volume::~Volume()
{
    //cout << "destroying\n";
    SpecFVRNS::destroyTransform2D();
    SpecFVRNS::destroyTransforms();
    if (m_Volume) {
	delete [] m_Volume;
	m_Volume = NULL;
    }
    if (m_Slice) {
	delete [] m_Slice;
	m_Slice = NULL;
    }
    if (m_Image) {
	delete [] m_Image;
	m_Image = NULL;
    }
    
#ifndef FVR_NO_SLICE_COPY
    if (m_SSlice) {
	delete [] m_SSlice;
	m_SSlice=NULL;
    }
#endif
}

Volume& Volume::operator=(Volume& rhs)
{
	if (this != &rhs)
	{
	}
	return *this;
}

void Volume::write_fvr(char* out) const
{
    cout << "now writing " << out << endl;
    
    ofstream fout;
    fout.open(out, ios::out|ios::binary);
    if(!fout.good()) cout <<" couldn't open file" << endl;
    
    
    write_fvr_head(fout, m_XSize, m_YSize, m_ZSize, sizeof(float));
    fout.write((char*)m_Volume, m_XSize * m_YSize * m_ZSize * 2 * sizeof(float));
    
    fout.close();
}

void Volume::preprocess(void)
{
	cout << "shifting volume ... " << flush;
	SpecFVRNS::shift3D(m_Volume, m_XSize, m_YSize, m_ZSize);

	cout << "done\ntransforming volume ... " << flush;
	SpecFVRNS::initTransform3D(m_XSize, m_YSize, m_ZSize);
	SpecFVRNS::transform3D(m_Volume);
	SpecFVRNS::destroyTransform3D();

	cout << "done\nshifting volume ... " << flush;
	SpecFVRNS::shift3D(m_Volume, m_XSize, m_YSize, m_ZSize);
	cout << "done\n" << flush;
}

bool Volume::read_fvr(ifstream& fin, dword XSize, dword YSize, dword ZSize, dword d_size)
{
	if (d_size != sizeof(float)) return false;

	m_XSize = XSize + m_Wrap * 2;
	m_YSize = YSize + m_Wrap * 2;
	m_ZSize = ZSize + m_Wrap * 2;

	m_Volume = new float[m_XSize * m_YSize * m_ZSize * 2];

	read_raw_r(fin, (byte*)m_Volume, m_XSize, m_YSize, m_ZSize, XSize, YSize, ZSize, d_size * 2);

	return true;
}

static dword find_dim(dword XSize, dword YSize, dword ZSize, float m_pad, dword a_pad)
{
	dword dim = (XSize > YSize) ? XSize : YSize;
	dim = (dim > ZSize) ? dim : ZSize;
	dim = (dword) ceil((float)dim * m_pad) + a_pad;
	dim = (ODD(dim)) ? dim + 1 : dim;

	cout << "dimension = " << dim << endl;

	return dim;
}

// read file.
bool Volume::convert(byte* data, dword XSize, dword YSize, dword ZSize, 
		     dword d_size, const vuTFIntensity &tfunc, dword component,
		     float osamp, float m_pad, dword a_pad)
{
	if (!data) return 0;

	//raw data, needs to be mapped with tfunc and transformed
	{
		dword dim = find_dim(XSize, YSize, ZSize, m_pad, a_pad);
		m_XSize = m_YSize = m_ZSize = dim;

		m_Volume = new float[m_XSize * m_YSize * m_ZSize * 2];

		read_raw(data, &m_Volume[0], m_XSize, m_YSize, m_ZSize, 
			 XSize, YSize, ZSize, d_size);
//apply transfer function
		dword c;
		float *fp = m_Volume;
		for(c=m_XSize * m_YSize * m_ZSize; c>0; c--) {
		    *fp = tfunc[(dword)(*fp)][component];
		    fp+=2;
		}

		preprocess();
	}

	// FVR file
	cout <<"frequency transform done"<<endl;
	
	setWrap(m_Filter->getWidth() / 2);
	wrapVolume();

	m_Origin[0] = (float)(XSize / 2) * -1.0f;
	m_Origin[1] = (float)(YSize / 2) * -1.0f;
	m_Origin[2] = 0.0f;

	m_ImageXSize = XSize;
	m_ImageYSize = YSize;

	m_Image = new float[m_ImageXSize * m_ImageYSize];

	setOversampling(osamp);

	cout << "initializing transforms ... " << flush;
	SpecFVRNS::initTransform2D(m_SliceXSize, m_SliceYSize);
	cout << "done\n" << flush;

	return true;
}

void Volume::setOversampling(float s) {
	// don't allow undersampling
	if (s < 1.0) s = 1.0;

	m_SliceXSize = (dword) ceil(m_ImageXSize * s);
	m_SliceYSize = (dword) ceil(m_ImageYSize * s);
	m_SliceXSize = (ODD(m_SliceXSize)) ? m_SliceXSize + 1 : m_SliceXSize;
	m_SliceYSize = (ODD(m_SliceYSize)) ? m_SliceYSize + 1 : m_SliceYSize;

	m_XStep = m_XAxis * (m_ImageXSize / (float)m_SliceXSize);
	m_YStep = m_YAxis * (m_ImageYSize / (float)m_SliceYSize);

	m_ImageStep = (m_SliceXSize - m_ImageXSize) * 2;

	cout << "Slice size: " << m_SliceXSize << ' ' << m_SliceYSize << endl;

	if (m_Slice) delete [] m_Slice;
	m_Slice = new float[m_SliceXSize * m_SliceYSize * 2];

#ifndef FVR_NO_SLICE_COPY
	if (m_SSlice) delete [] m_Slice;
	m_SSlice = new float[m_SliceXSize * m_SliceYSize * 2];

	m_SlicePtr = &m_SSlice[scoord( (m_SliceXSize - m_ImageXSize) / 2,
	                               (m_SliceYSize - m_ImageYSize) / 2)];
#else
	m_SlicePtr = &m_Slice[scoord( (m_SliceXSize - m_ImageXSize) / 2,
	                              (m_SliceYSize - m_ImageYSize) / 2)];
#endif

	m_InnerXSize = (dword) floor((float) m_SliceXSize / ROOT_TWO);
	m_InnerYSize = (dword) floor((float) m_SliceYSize / ROOT_TWO);
	m_InnerXSize = (ODD(m_InnerXSize)) ? m_InnerXSize + 1 : m_InnerXSize;
	m_InnerYSize = (ODD(m_InnerYSize)) ? m_InnerYSize + 1 : m_InnerYSize;
}

void Volume::writeImage(void)
{
	char name[32];
	float v;
	ofstream fout;
	dword i;
	byte* img;

	img = new byte[m_SliceXSize*m_SliceYSize];
	for(i=0;i<m_SliceXSize*m_SliceYSize;i++)
	{
		v = m_Image[i]*255.0f + 0.5f;
		if (v > 255.0f)
			img[i] = 255;
		else if (v < 0.0f)
			img[i] = 0;
		else
			img[i] = (byte)v;
	}

	sprintf(name, "%i.pgm", m_CurImage++);
	fout.open(name, ios::out|ios::binary);
	if (fout.is_open())
	{
		sprintf(name, "P5%c%ld %ld%c%d%c", 0x0A, m_SliceXSize, m_SliceYSize, 0x0A, 255, 0x0A);

		fout.write(name, strlen(name));
		fout.write((char*)img, m_XSize*m_YSize);

		fout.close();
	}

	delete [] img;
}

void Volume::setWrap(dword wrap)
{
	m_Wrap = wrap;
}

void Volume::setFilter(Filter* filter)
{
    m_Filter = filter;
}

void Volume::setSliceScale(float scale)
{
	m_Scale = scale;
}

void Volume::setSliceBias(float bias)
{
	m_Bias = bias;
}

void Volume::rotateSliceX(float alpha)
{
	vuMatrix rot;

	rot.makeRotate(m_XAxis, alpha);

	m_YAxis  = rot * m_YAxis;
	m_ZAxis  = rot * m_ZAxis;
	m_XStep  = rot * m_XStep;
	m_YStep  = rot * m_YStep;
	m_Origin = rot * m_Origin;
}

void Volume::rotateSliceY(float alpha)
{
	vuMatrix rot;

	rot.makeRotate(m_YAxis, alpha);

	m_XAxis  = rot * m_XAxis;
	m_ZAxis  = rot * m_ZAxis;
	m_XStep  = rot * m_XStep;
	m_YStep  = rot * m_YStep;
	m_Origin = rot * m_Origin;
}

void Volume::rotateSliceZ(float alpha)
{
	vuMatrix rot;

	rot.makeRotate(m_ZAxis, alpha);

	m_XAxis  = rot * m_XAxis;
	m_YAxis  = rot * m_YAxis;
	m_XStep  = rot * m_XStep;
	m_YStep  = rot * m_YStep;
	m_Origin = rot * m_Origin;
}

void Volume::setViewMatrix(const vuMatrix& mat) 
{
    m_XAxis = mat*vuVector(1,0,0);
    m_YAxis = mat*vuVector(0,1,0);
    m_ZAxis = mat*vuVector(0,0,1);
    m_XStep = m_XAxis;
    m_YStep = m_YAxis;
    m_XStep *= (float)(m_ImageXSize / m_SliceXSize);
    m_YStep *= (float)(m_ImageYSize / m_SliceYSize);
    m_Origin  = m_XAxis*(m_SliceXSize*-0.5f);
    m_Origin += m_YAxis*(m_SliceYSize*-0.5f);
}

void Volume::setCamera(const vuCamera& cam) 
{
    m_XAxis = cam.getRightVector();
    m_XAxis.makeUnit();
    m_YAxis = cam.getUpVector();
    m_YAxis.makeUnit();
    m_ZAxis = cam.getLookAtVector();
    m_XStep = m_XAxis;
    m_YStep = m_YAxis;
    m_XStep *= (float)(m_ImageXSize / m_SliceXSize);
    m_YStep *= (float)(m_ImageYSize / m_SliceYSize);
    m_Origin  = m_XAxis*(m_SliceXSize*-0.5f);
    m_Origin += m_YAxis*(m_SliceYSize*-0.5f);
}

void Volume::getCamera(vuCamera& cam) const
{
    cam.setLookAtVector(m_ZAxis);
    cam.setUpVector(m_YAxis);
    //cam.setRightVector(m_XAxis);
}

dword Volume::getSliceWidth(void) const
{
	return m_ImageXSize;
}

dword Volume::getSliceHeight(void) const
{
	return m_ImageYSize;
}

dword Volume::getInternalSliceWidth(void) const
{
	return m_SliceXSize;
}

dword Volume::getInternalSliceHeight(void) const
{
	return m_SliceYSize;
}

// fill in zeroed wrap-arounds
void Volume::wrapVolume(void)
{
	dword index;
	dword ii, jj ,kk;

	dword hiX = m_XSize - m_Wrap - 1;
	dword hiY = m_YSize - m_Wrap - 1;
	dword hiZ = m_ZSize - m_Wrap - 1;

	dword XSize = m_XSize - m_Wrap * 2;
	dword YSize = m_YSize - m_Wrap * 2;
	dword ZSize = m_ZSize - m_Wrap * 2;

	index = 0;
	for(dword k = 0; k < m_ZSize; ++k) {
		for(dword j = 0; j < m_YSize; ++j) {
			for(dword i = 0; i < m_XSize; ++i) {
				if (i < m_Wrap)
					ii = i + XSize;
				else if (i > hiX)
					ii = i - XSize;
				else {
					index += 2;
					continue;
				}

				if (j < m_Wrap)
					jj = j + YSize;
				else if (j > hiY)
					jj = j - YSize;
				else {
					index += 2;
					continue;
				}

				if (k < m_Wrap)
					kk = k + ZSize;
				else if (k > hiZ)
					kk = k - ZSize;
				else {
					index += 2;
					continue;
				}

				ii = vcoord(ii, jj, kk);

				m_Volume[index++] = m_Volume[ii];
				m_Volume[index++] = m_Volume[ii+1];
			}
		}
	}
}

void Volume::clearSlice(void)
{
	float* s_ptr = m_Slice;
	for (dword i = 0; i < m_SliceXSize * m_SliceYSize * 2; i++)
		*(s_ptr++) = 0.0f;
}

float* Volume::getSliceData(void)
{
	scaleAndBias();

	return m_Image;
}

// and may the gods of good-style forgive me ...
// inner convolution loop, should be fast!
#define CONVOLVE_NO_MOD	\
			int x = (int) p[0];											\
			int y = (int) p[1];											\
			int z = (int) p[2];											\
																		\
			t[0] = p[0] - x;											\
			t[1] = p[1] - y;											\
			t[2] = p[2] - z;											\
																		\
			x = x - XSize + m_Wrap;										\
			y = y - YSize + m_Wrap;										\
			z = z - ZSize + m_Wrap;										\
																		\
			register float* fweight = m_Filter->getWeights(t);					\
																		\
			register float real = 0.0f;											\
			register float imag = 0.0f;											\
																		\
			register float* v_ptr = &m_Volume[vcoord(x + f2, y + f2, z + f2)];	\
																		\
			for (dword l = 0; l < f_len; l++) {							\
				for (dword m = 0; m < f_len; m++) {						\
					for (dword n = 0; n < f_len; n++) {					\
						float tmp = *(fweight++);						\
						real += *(v_ptr)     * tmp;						\
						imag += *(v_ptr + 1) * tmp;						\
						v_ptr -= 2;										\
					}													\
					v_ptr += fXStep;									\
				}														\
				v_ptr += fYStep;										\
			}															\
																		\
			*(s_ptr++) = real;											\
			*(s_ptr++) = imag;											\
																		\
			p += m_XStep;

#define CONVOLVE	\
			int x = (int) p[0];											\
			int y = (int) p[1];											\
			int z = (int) p[2];											\
																		\
			t[0] = p[0] - x;											\
			t[1] = p[1] - y;											\
			t[2] = p[2] - z;											\
																		\
			x = x % XSize + m_Wrap;										\
			y = y % YSize + m_Wrap;										\
			z = z % ZSize + m_Wrap;										\
																		\
			register float* fweight = m_Filter->getWeights(t);					\
																		\
			register float real = 0.0f;											\
			register float imag = 0.0f;											\
																		\
			register float* v_ptr = &m_Volume[vcoord(x + f2, y + f2, z + f2)];	\
																		\
			for (dword l = 0; l < f_len; l++) {							\
				for (dword m = 0; m < f_len; m++) {						\
					for (dword n = 0; n < f_len; n++) {					\
						float tmp = *(fweight++);						\
						real += *(v_ptr)     * tmp;						\
						imag += *(v_ptr + 1) * tmp;						\
						v_ptr -= 2;										\
					}													\
					v_ptr += fXStep;									\
				}														\
				v_ptr += fYStep;										\
			}															\
																		\
			*(s_ptr++) = real;											\
			*(s_ptr++) = imag;											\
																		\
			p += m_XStep;

void Volume::interpolateSlice(dword x_stop, dword y_stop, dword x_pass, dword y_pass)
{
	vuVector t;
	dword f_len = m_Filter->getWidth();
	int f2 = f_len / 2;

	x_stop *= 2;
	y_stop *= 2;
	dword stop_l = (m_SliceXSize - x_stop - 2 * x_pass) / 2;
	dword stop_r =  m_SliceXSize - x_stop - 2 * x_pass - stop_l;
	dword stop_b = (m_SliceYSize - y_stop - 2 * y_pass) / 2;
	dword x_hi = x_stop + 2 * x_pass;
	dword s_step = stop_l + stop_r;

	dword XSize = m_XSize - m_Wrap * 2;
	dword YSize = m_YSize - m_Wrap * 2;
	dword ZSize = m_ZSize - m_Wrap * 2;

	dword fXStep = (f_len - m_XSize) * 2;
	dword fYStep = (f_len - m_YSize) * m_XSize * 2;

	vuVector q = m_Origin;
	q[0] += XSize / 2 + XSize + stop_l * m_XStep[0] + stop_b * m_YStep[0];
	q[1] += YSize / 2 + YSize + stop_l * m_XStep[1] + stop_b * m_YStep[1];
	q[2] += ZSize / 2 + ZSize + stop_l * m_XStep[2] + stop_b * m_YStep[2];

	float* s_ptr = &m_Slice[scoord(stop_l, stop_b)];

	for (dword j = 0; j < y_pass; ++j) {
		vuVector p = q;
		for (dword i = 0; i < x_hi; ++i) {
			CONVOLVE;
		}
		s_ptr += s_step * 2;
		q += m_YStep;
	}

	for (dword j = 0; j < y_stop; ++j) {
		vuVector p = q;
		for (dword i = 0; i < x_pass; ++i) {
			CONVOLVE;
		}

		s_ptr += x_stop * 2;
		p[0] += x_stop * m_XStep[0];
		p[1] += x_stop * m_XStep[1];
		p[2] += x_stop * m_XStep[2];

		for (dword i = 0; i < x_pass; ++i) {
			CONVOLVE;
		}
		s_ptr += s_step * 2;
		q += m_YStep;
	}

	for (dword j = 0; j < y_pass; ++j) {
		vuVector p = q;
		for (dword i = 0; i < x_hi; ++i) {
			CONVOLVE;
		}
		s_ptr += s_step * 2;
		q += m_YStep;
	}
}

void Volume::interpolateSlice(void)
{
	dword i, j;
	vuVector t;
	dword f_len = m_Filter->getWidth();
	int f2 = f_len / 2;

	dword x_pass = (m_SliceXSize - m_InnerXSize) / 2;
	dword y_pass = (m_SliceYSize - m_InnerYSize) / 2;

	dword XSize = m_XSize - m_Wrap * 2;
	dword YSize = m_YSize - m_Wrap * 2;
	dword ZSize = m_ZSize - m_Wrap * 2;

	dword fXStep = (f_len - m_XSize) * 2;
	dword fYStep = (f_len - m_YSize) * m_XSize * 2;

	vuVector q = m_Origin;
	q[0] += XSize / 2 + XSize;
	q[1] += YSize / 2 + YSize;
	q[2] += ZSize / 2 + ZSize;

	float* s_ptr = m_Slice;

	for (j = 0; j < y_pass; ++j) {
		vuVector p = q;
		for (i = 0; i < m_SliceXSize; ++i) {
			CONVOLVE;
		}
		q += m_YStep;
	}

	for (j = 0; j < m_InnerYSize; ++j) {
		vuVector p = q;
		for (i = 0; i < x_pass; ++i) {
			CONVOLVE;
		}

		for (i = 0; i < m_InnerXSize; ++i) {
			CONVOLVE_NO_MOD;
		}

		for (i = 0; i < x_pass; ++i) {
			CONVOLVE;
		}
		q += m_YStep;
	}

	for (j = 0; j < y_pass; ++j) {
		vuVector p = q;
		for (i = 0; i < m_SliceXSize; ++i) {
			CONVOLVE;
		}
		q += m_YStep;
	}
}

#ifndef FVR_NO_SLICE_COPY

void Volume::computeSlice(void)
{
	interpolateSlice();
	SpecFVRNS::shift_copy2D(m_SSlice, m_Slice, m_SliceXSize, m_SliceYSize);
	SpecFVRNS::transform2D(m_SSlice);
	SpecFVRNS::shift2D(m_SSlice, m_SliceXSize, m_SliceYSize);
}

void Volume::computeSlice(dword x_stop, dword y_stop, dword x_pass, dword y_pass)
{
	interpolateSlice(x_stop, y_stop, x_pass, y_pass);
	SpecFVRNS::shift_copy2D(m_SSlice, m_Slice, m_SliceXSize, m_SliceYSize);
	SpecFVRNS::transform2D(m_SSlice);
	SpecFVRNS::shift2D(m_SSlice, m_SliceXSize, m_SliceYSize);
}

#else

void Volume::computeSlice(void)
{
	interpolateSlice();
	shift2D(m_Slice, m_SliceXSize, m_SliceYSize);
	transform2D(m_Slice);
	shift2D(m_Slice, m_SliceXSize, m_SliceYSize);
}

void Volume::computeSlice(dword x_stop, dword y_stop, dword x_pass, dword y_pass)
{
	interpolateSlice(x_stop, y_stop, x_pass, y_pass);
	shift2D(m_Slice, m_SliceXSize, m_SliceYSize);
	transform2D(m_Slice);
	shift2D(m_Slice, m_SliceXSize, m_SliceYSize);
}

#endif

void Volume::scaleAndBias(void)
{
	float* i_ptr = m_Image;
	float* s_ptr = m_SlicePtr;
	for (dword j = 0; j < m_ImageYSize; j++) {
		for (dword i = 0; i < m_ImageXSize; i++) {
			*(i_ptr++) = *s_ptr * m_Scale + m_Bias;
			s_ptr += 2;
		}
		s_ptr += m_ImageStep;
	}
}
