#ifndef _VOLSET_H_
#define _VOLSET_H_

#include "vuTFunc/vuTFIntensity.h"
#include "vuCamera.h"
#include "Volume.h"

//! Apply a Volume member function to all volumes in the array.
#define ITERATE_VOLS(f) do { int c;\
    for( c=0; c<m_NVolumes; c++) \
        if(m_Volumes[c]) m_Volumes[c]->f; \
 }while(0); 

/** This class is managing several volumes.
    Each volume is representing a channel for a spectral colour model
    or RGB. */
class VolumeSet 
{
 public:
    /** Constructor.
	Number of volumes (default = 0) can later be changed. */
    VolumeSet(int nvolumes = 0);
    //! Destructor.
    ~VolumeSet();
    
    /** Init member variables, initialize volume array.
	If initFilter is true, the Filter will also be deleted.
    */
    void createVolumes(int nvolumes = 0, bool initFilter=false);
    
    //!Free the memory of the slice
    void clearSlice(void) 
	{
	    ITERATE_VOLS(clearSlice());
	}
    
    //!computes Slices from all Volumes
    void computeSlice(dword x_stop, dword y_stop, dword x_pass, dword y_pass) 
	{
	    ITERATE_VOLS(computeSlice(x_stop, y_stop, x_pass, y_pass));
	}
    
    //!Returns a Volume of the Array
    Volume* operator[](dword index)
	{
	    if((long)index < m_NVolumes)
		return m_Volumes[index];
	    else
		return NULL;
	}
    
    //!Returns a Volume of the Array (const accessor).
    const Volume* operator[](dword index) const
	{
	    if((long)index < m_NVolumes)
		return m_Volumes[index];
	    else
		return NULL;
	}
    
    /** The rotations are local to the slice. */
    void rotateSliceX(float alpha) {
	ITERATE_VOLS(rotateSliceX(alpha));
    }
    /** The rotations are local to the slice. */
    void rotateSliceY(float alpha) {
	ITERATE_VOLS(rotateSliceY(alpha));
    }
    /** The rotations are local to the slice. */
    void rotateSliceZ(float alpha) {
	ITERATE_VOLS(rotateSliceZ(alpha));
    }
    /** Set the coordinate system of the camera to be the one of the slice. */
    void setCamera(const vuCamera& cam) {
	ITERATE_VOLS(setCamera(cam));
    }
    /** Scale brightness of the slice. */
    void setSliceScale(float scale) {
	ITERATE_VOLS(setSliceScale(scale));
    }
    /** Set slice bias. */
    void setSliceBias(float bias) {
	ITERATE_VOLS(setSliceBias(bias));
    }

    //! Set the filter by number
    bool setFilter(int num);

    /** Set the filter.
	The class becomes owner of the Filter, so don't delete it! */
    bool setFilter(SpecFVRNS::Filter* filter);

    //! Read the volume
    bool read(char* in, float s = 1.0,
	      float m_pad = 1.0f, dword a_pad = 0);

    /** Remap and transform volume data to frequency domain.
        Using transfer function for remapping. If m_SingleChannel is true
        the tfunc won't be used and only a conventional one channel FVR
	is performed. */
    void buildColourVolumes(byte *data, 
			    dword xsize, dword ysize, dword zsize,
			    dword dsize, const vuTFIntensity &tfunc,
			    float osamp=1.0f, float m_pad=1.0f,
			    dword a_pad=0);
    
    //! write the frequency volume
    bool write_fvr(char* fn);

 protected:
    //!clean up the memory
    void destroyData(bool destroyFilter=false);

 public:
    int		m_NVolumes;		//!< Number of volumes
    SpecFVRNS::Filter	*m_Filter;		//!< Filter used for slice extraction
    Volume	**m_Volumes;		//!< Array of Volume pointers
    bool	m_Recomp;		//!< Recompute slice
};

#endif
