#ifndef _VECTOR_H_
#define _VECTOR_H_

#include "vuSimpleTypes.h"
#include <math.h>
#include "Matrix.h"

//
// The Vector class is a 3D homogeneous vector
// that works in conjunction with the Matrix
// class (a 4x4 matrix).  The elements of the
// Vector are all single precision floating-
// point numbers.
//
class Vector
{
friend class Matrix;
public:
    //
    // Constructors and destructor.
    //
    inline Vector();
    inline Vector(const Vector& v);
    inline Vector(float v1, float v2, float v3);
    inline Vector(float v1, float v2, float v3, float v4);
    inline Vector(float v);
    inline Vector(float* v);
    inline ~Vector();

    //
    // Perform the 1-norm and 2-norm on the vector.
    // (The 2-norm is the same as the length).
    //
    inline float Norm(void);
    inline float Norm2(void);

    inline Vector& MakeUnit(void);
    inline Vector& Normalize(void);

    //
    // Pair-wise entry operations.
    //
    inline Vector Inv(void);
    inline Vector& InvEq(void);
    inline Vector Mul(Vector& rhs);
    inline Vector& MulEq(Vector& rhs);
    inline Vector Div(Vector& rhs);
    inline Vector& DivEq(Vector& rhs);

    //
    // Assignment and accessor operators.
    //
    inline Vector& operator=(const Vector& v);
    inline Vector& operator=(const float v);
    inline Vector& operator=(const float* v);
    inline float& operator[](dword index);

    //
    // Get the data pointer.
    //
    inline float* GetData(void);

    //
    // Dot and cross products.
    //
    inline friend float Dot(Vector& v1, Vector& v2);
    inline friend Vector Cross(Vector& v1, Vector& v2);
    inline float Dot(Vector& v);
    inline void Cross(Vector& v, Vector& r);

    //
    // Operators for vector-vector and vector-matrix
    // operations.
    //
    inline Vector operator+(Vector& v);
    inline Vector operator-(Vector& v);
    inline Vector operator*(Matrix& m);
    inline Matrix operator*(Vector& v);
    inline Vector operator*(float s);
    inline Vector operator/(float s);
    inline friend Vector operator*(float s, Vector& v);
    inline Vector& operator+=(Vector& v);
    inline Vector& operator-=(Vector& v);
    inline Vector& operator*=(Matrix& m);
    inline Vector& operator*=(float s);
    inline Vector& operator/=(float s);

    //
    // Equality and inequality operators.
    //
    inline bool operator==(const Vector& v) const;
    inline bool operator!=(const Vector& v) const;

private:
    float val[4];
};

// Default constructor - set initial values
inline Vector::Vector()
{
    val[0] = val[1] = val[2] = 0.0f;
    val[3] = 1.0f;
}

// Copy constructor
inline Vector::Vector(const Vector& v)
{
    val[0] = v.val[0];
    val[1] = v.val[1];
    val[2] = v.val[2];
    val[3] = v.val[3];
}

// Constructor - set to <v1,v2,v3,1.0>
inline Vector::Vector(float v1, float v2, float v3)
{
    val[0] = v1;
    val[1] = v2;
    val[2] = v3;
    val[3] = 1.0f;
}

// Constructor - set to <v1,v2,v3,v4>
inline Vector::Vector(float v1, float v2, float v3, float v4)
{
    val[0] = v1;
    val[1] = v2;
    val[2] = v3;
    val[3] = v4;
}

// Constructor - set all values to v
inline Vector::Vector(float v)
{
    val[0] = val[1] = val[2] = v;
    val[3] = 1.0f;
}

// Constroctor - get values from an array
inline Vector::Vector(float* v)
{
    val[0] = v[0];
    val[1] = v[1];
    val[2] = v[2];
    val[3] = 1.0f;
}

// Destructor
inline Vector::~Vector()
{
}

// Compute the norm of the vector  - |v|
inline float Vector::Norm(void)
{
    return (float)sqrt((double)(val[0]*val[0]+val[1]*val[1]+val[2]*val[2]));
}

// Compute the square norm of the vector
inline float Vector::Norm2(void)
{
    return val[0]*val[0]+val[1]*val[1]+val[2]*val[2];
}

// Normalize the vector - set its length to 1
inline Vector& Vector::MakeUnit(void)
{
    float d = (float)sqrt((double)(val[0]*val[0]+val[1]*val[1]+val[2]*val[2]));
    if (d)
    {
        val[0] /= d;
        val[1] /= d;
        val[2] /= d;
    }
    else
    {
        val[0] = val[1] = val[2] = 0.0f;
    }
    return *this;
}

inline Vector& Vector::Normalize(void)
{
    float d;
    d = (val[3])?(1.0f/val[3]):0.0f;
    val[0] *= d;
    val[1] *= d;
    val[2] *= d;
    val[3] = 1.0f;
    return *this;
}

// Invert the vector - divide all entries into 1
inline Vector Vector::Inv(void)
{
    Vector r;
    r[0] = (val[0])?(1.0f/val[0]):0.0f;
    r[1] = (val[1])?(1.0f/val[1]):0.0f;
    r[2] = (val[2])?(1.0f/val[2]):0.0f;
    return r;
}

// Invert and assign - divide all entries into 1
// for current vector
inline Vector& Vector::InvEq(void)
{
    val[0] = (val[0])?(1.0f/val[0]):0.0f;
    val[1] = (val[1])?(1.0f/val[1]):0.0f;
    val[2] = (val[2])?(1.0f/val[2]):0.0f;
    return *this;
}

// Entry-wise multiplication with another vector
inline Vector Vector::Mul(Vector& rhs)
{
    Vector r;
    r[0] = val[0] * rhs.val[0];
    r[1] = val[1] * rhs.val[1];
    r[2] = val[2] * rhs.val[2];
    return r;
}

// Entry-wise multiplication with another vector, then
// assign result to the current vector.
inline Vector& Vector::MulEq(Vector& rhs)
{
    val[0] *= rhs.val[0];
    val[1] *= rhs.val[1];
    val[2] *= rhs.val[2];
    return *this;
}

// Entry-wise division with another vector
inline Vector Vector::Div(Vector& rhs)
{
    Vector r;
    r[0] = (rhs.val[0])?(val[0]/rhs.val[0]):0.0f;
    r[1] = (rhs.val[1])?(val[1]/rhs.val[1]):0.0f;
    r[2] = (rhs.val[2])?(val[2]/rhs.val[2]):0.0f;
    return r;
}

// Entry-wise division with another vector, then
// assign result to the current vector.
inline Vector& Vector::DivEq(Vector& rhs)
{
    val[0] = (rhs.val[0])?(val[0]/rhs.val[0]):val[0]=0.0f;
    val[1] = (rhs.val[1])?(val[1]/rhs.val[1]):val[1]=0.0f;
    val[2] = (rhs.val[2])?(val[2]/rhs.val[2]):val[2]=0.0f;
    return *this;
}

// Assignment operator
inline Vector& Vector::operator=(const Vector& v)
{
    if (this != &v)
    {
        val[0] = v.val[0];
        val[1] = v.val[1];
        val[2] = v.val[2];
        val[3] = v.val[3];
    }
    return *this;
}

// Assignment operator - set all values to v
inline Vector& Vector::operator=(const float v)
{
    val[0] = val[1] = val[2] = v;
    val[3] = 1.0f;
    return *this;
}

// Assignment operator - get values from an array
inline Vector& Vector::operator=(const float* v)
{
    val[0] = v[0];
    val[1] = v[1];
    val[2] = v[2];
    val[3] = 1.0f;
    return *this;
}

// Access operator - return the value at the given index
inline float& Vector::operator[](dword index)
{
    return val[index];
}

// Return the internal data pointer.
inline float* Vector::GetData(void)
{
    return val;
}

// Dot product friend function
inline float Dot(Vector& v1, Vector& v2)
{
    return v1.val[0]*v2.val[0]+v1.val[1]*v2.val[1]+v1.val[2]*v2.val[2];
}

// Cross product friend function
inline Vector Cross(Vector& v1, Vector& v2)
{
    Vector v;
    v.val[0] = (v1.val[1]*v2.val[2] - v1.val[2]*v2.val[1]);
	v.val[1] = (v1.val[2]*v2.val[0] - v1.val[0]*v2.val[2]);
    v.val[2] = (v1.val[0]*v2.val[1] - v1.val[1]*v2.val[0]);
    v.val[3] = 1.0f;
    return v;
}

// Dot product member function
inline float Vector::Dot(Vector& v)
{
    return val[0]*v.val[0]+val[1]*v.val[1]+val[2]*v.val[2];
}

// Cross product member function
inline void Vector::Cross(Vector& v, Vector& r)
{
	r.val[0] = (val[1]*v.val[2] - val[2]*v.val[1]);
    r.val[1] = (val[2]*v.val[0] - val[0]*v.val[2]);
    r.val[2] = (val[0]*v.val[1] - val[1]*v.val[0]);
    r.val[3] = 1.0f;
}

// Addition operator
inline Vector Vector::operator+(Vector& v)
{
    Vector r;
    r.val[0] = val[0] + v.val[0];
    r.val[1] = val[1] + v.val[1];
    r.val[2] = val[2] + v.val[2];
    r.val[3] = 1.0f;
    return r;
}

// Subtraction operator
inline Vector Vector::operator-(Vector& v)
{
    Vector r;
    r.val[0] = val[0] - v.val[0];
    r.val[1] = val[1] - v.val[1];
    r.val[2] = val[2] - v.val[2];
    r.val[3] = 1.0f;
    return r;
}

// Multiplication operator - vector*matrix
inline Vector Vector::operator*(Matrix& m)
{
    Vector r;
    dword i, j;
    r[3] = 0.0f;
    for(i=0;i<4;++i)
        for(j=0;j<4;++j)
            r.val[i] += m.val[(i<<2)+j]*val[j];
    return r;
}

// Multiplication operator - vector*vector -> matrix
inline Matrix Vector::operator*(Vector& v)
{
    Matrix r;
    dword i, j;
    for(i=0;i<4;++i)
        for(j=0;j<4;++j)
            r.val[(j<<2)+i] = val[i]*v.val[j];
    return r;
}

// Multiplication operator - vector*scalar
inline Vector Vector::operator*(float s)
{
    Vector r;
    r.val[0] = val[0] * s;
    r.val[1] = val[1] * s;
    r.val[2] = val[2] * s;
    r.val[3] = 1.0f;
    return r;
}

// Division operator - vector/scalar
inline Vector Vector::operator/(float s)
{
    Vector r;
    r.val[0] = val[0] / s;
    r.val[1] = val[1] / s;
    r.val[2] = val[2] / s;
    r.val[3] = 1.0f;
    return r;
}

// Multiplication operator - scalar*vector
inline Vector operator*(float s, Vector& v)
{
    Vector r;
    r.val[0] = v.val[0] * s;
    r.val[1] = v.val[1] * s;
    r.val[2] = v.val[2] * s;
    r.val[3] = 1.0f;
    return r;
}

// Add then assign operator
inline Vector& Vector::operator+=(Vector& v)
{
    val[0] += v.val[0];
    val[1] += v.val[1];
    val[2] += v.val[2];
    val[3] = 1.0f;
    return *this;
}

// Subtract then assign operator
inline Vector& Vector::operator-=(Vector& v)
{
    val[0] -= v.val[0];
    val[1] -= v.val[1];
    val[2] -= v.val[2];
    val[3] = 1.0f;
    return *this;
}

// Multiply then assign operator - vector*matrix
inline Vector& Vector::operator*=(Matrix& m)
{
    Vector r;
    dword i, j;
    r[3] = 0.0f;
    for(i=0;i<4;++i)
        for(j=0;j<4;++j)
            r.val[i] += m.val[(i<<2)+j]*val[j];
    return (*this=r);
}

// Multiply then assign operator - vector*scalar
inline Vector& Vector::operator*=(float s)
{
    val[0] *= s;
    val[1] *= s;
    val[2] *= s;
    val[3] = 1.0f;
    return *this;
}

// Divide then assign operator - vector/scalar
inline Vector& Vector::operator/=(float s)
{
    val[0] /= s;
    val[1] /= s;
    val[2] /= s;
    val[3] = 1.0f;
    return *this;
}

// Equality operator
inline bool Vector::operator==(const Vector& v) const
{
    return ((val[0]==v.val[0])&&(val[1]==v.val[1])&&(val[2]==v.val[2]));
}

// Inequality operator
inline bool Vector::operator!=(const Vector& v) const
{
    return !(operator==(v));
}

#endif
