#ifndef _MATRIX_H_
#define _MATRIX_H_

#include "vuSimpleTypes.h"

class Vector;

//
// The Matrix class is a 4x4 matrix that
// works in conjunction with the Vector class
// (a 3D homogeneous vector).  The elements of
// the Matrix are all single precision
// floating-point numbers, stored in column-major
// order to be compatible with OpenGL.
//
class Matrix
{
friend class Vector;
public:
    //
    // Constructors and destructor.
    //
    Matrix();
    Matrix(const Matrix& m);
    Matrix(const float v);
    Matrix(const float* v);
    ~Matrix();

    //
    // Generate various special matrices by
    // changing values in the current instance.
    // All transformation matrices are pre-
    // transforms, meaning they should be left
    // multiplied against a vector.
    //
    Matrix& MakeIdentity(void);
    Matrix& MakeRotate(Vector& axis, float a);
    Matrix& MakeRotateX(float a);
    Matrix& MakeRotateY(float a);
    Matrix& MakeRotateZ(float a);
    Matrix& MakeTranslate(float x, float y, float z);
    Matrix& MakeScale(float x, float y, float z);
    Matrix& MakeShearXY(float s);
    Matrix& MakeShearXZ(float s);
    Matrix& MakeShearYX(float s);
    Matrix& MakeShearYZ(float s);
    Matrix& MakeShearZX(float s);
    Matrix& MakeShearZY(float s);
    Matrix& MakeReflectX(void);
    Matrix& MakeReflectY(void);
    Matrix& MakeReflectZ(void);
    Matrix& MakePerspective(float d);
    Matrix& MakePerspectiveKeepZ(float d);

    //
    // Assignment and accessor operators.
    //
    Matrix& operator=(const Matrix& m);
    Matrix& operator=(const float& v);
    Matrix& operator=(const float* v);
    float* operator[](unsigned int index);

    //
    // Get the data pointer.
    // Same as calling operator[0].
    //
    float* GetData(void);

    //
    // Operators for matrix-matrix and matrix-vector
    // operations.
    //
    Matrix operator+(Matrix& m);
    Matrix operator-(Matrix& m);
    Matrix operator*(Matrix& m);
    Vector operator*(Vector& v);
    Matrix operator*(float s);
    friend Matrix operator*(float s, Matrix& m);
    Matrix& operator+=(Matrix& m);
    Matrix& operator-=(Matrix& m);
    Matrix& operator*=(const Matrix& m);
    Matrix& operator*=(float s);

    //
    // Equality and inequality operators.
    //
    bool operator==(const Matrix& m) const;
    bool operator!=(const Matrix& m) const;

private:
    float val[16];
};

#endif

