#include <math.h>
#include "Matrix.h"
#include "vuVector.h"

#define PI_OVER_180 0.01745329251994

// Default constructor - set initial values
Matrix::Matrix()
{
    for(int i=0;i<16;++i)
        val[i] = 0.0f;
}

// Copy constructor
Matrix::Matrix(const Matrix& m)
{
    for(int i=0;i<16;++i)
        val[i] = m.val[i];
}

// Constructor - set all values to v
Matrix::Matrix(const float v)
{
    for(int i=0;i<16;++i)
        val[i] = v;
}

// Constructor - get values from an array
Matrix::Matrix(const float* v)
{
    for(int i=0;i<16;++i)
        val[i] = v[i];
}

// Destructor
Matrix::~Matrix()
{
}

// Make the current matrix the identity matrix
Matrix& Matrix::MakeIdentity(void)
{
    for(int i=1;i<15;++i)
        val[i] = 0.0f;
    val[0] = val[5] = val[10] = val[15] = 1.0f;
    return *this;
}

Matrix& Matrix::MakeRotate(vuVector& axis, float a)
{
	vuVector axis2;
	float s;
	float c;

	s = (float)sin(a*PI_OVER_180);
	c = (float)cos(a*PI_OVER_180);
	axis2 = axis.Mul(axis);

	val[0] = axis2[0]+(1.0f-axis2[0])*c;
	val[1] = axis[0]*axis[1]*(1.0f-c)+axis[2]*s;
	val[2] = axis[0]*axis[2]*(1.0f-c)-axis[1]*s;
    val[3] = 0.0f;

	val[4] = axis[0]*axis[1]*(1.0f-c)-axis[2]*s;
	val[5] = axis2[1]+(1.0f-axis2[1])*c;
	val[6] = axis[1]*axis[2]*(1.0f-c)+axis[0]*s;
    val[7] = 0.0f;

	val[8]  = axis[0]*axis[2]*(1.0f-c)+axis[1]*s;
	val[9]  = axis[1]*axis[2]*(1.0f-c)-axis[0]*s;
	val[10] = axis2[2]+(1.0f-axis2[2])*c;
    val[11] = 0.0f;

    val[12] = 0.0f;
    val[13] = 0.0f;
    val[14] = 0.0f;
	val[15] = 1.0f;

    return *this;
}

Matrix& Matrix::MakeRotateX(float a)
{
    MakeIdentity();
    val[5] = val[10] = (float)cos((double)a*PI_OVER_180);
    val[9] = -(val[6] = (float)sin((double)a*PI_OVER_180));
    return *this;
}

Matrix& Matrix::MakeRotateY(float a)
{
    MakeIdentity();
    val[0] = val[10] = (float)cos((double)a*PI_OVER_180);
    val[2] = -(val[8] = (float)sin((double)a*PI_OVER_180));
    return *this;
}

Matrix& Matrix::MakeRotateZ(float a)
{
    MakeIdentity();
    val[0] = val[5] = (float)cos((double)a*PI_OVER_180);
    val[4] = -(val[1] = (float)sin((double)a*PI_OVER_180));
    return *this;
}

Matrix& Matrix::MakeTranslate(float x, float y, float z)
{
    MakeIdentity();
    val[12] = x;
    val[13] = y;
    val[14] = z;
    return *this;
}

Matrix& Matrix::MakeScale(float x, float y, float z)
{
    MakeIdentity();
    val[0] = x;
    val[5] = y;
    val[10] = z;
    return *this;
}

Matrix& Matrix::MakeShearXY(float s)
{
    MakeIdentity();
    val[1] = s;
    return *this;
}

Matrix& Matrix::MakeShearXZ(float s)
{
    MakeIdentity();
    val[2] = s;
    return *this;
}

Matrix& Matrix::MakeShearYX(float s)
{
    MakeIdentity();
    val[4] = s;
    return *this;
}

Matrix& Matrix::MakeShearYZ(float s)
{
    MakeIdentity();
    val[6] = s;
    return *this;
}

Matrix& Matrix::MakeShearZX(float s)
{
    MakeIdentity();
    val[8] = s;
    return *this;
}

Matrix& Matrix::MakeShearZY(float s)
{
    MakeIdentity();
    val[9] = s;
    return *this;
}

Matrix& Matrix::MakeReflectX(void)
{
    MakeIdentity();
    val[0] = -1.0f;
    return *this;
}

Matrix& Matrix::MakeReflectY(void)
{
    MakeIdentity();
    val[5] = -1.0f;
    return *this;
}

Matrix& Matrix::MakeReflectZ(void)
{
    MakeIdentity();
    val[10] = -1.0f;
    return *this;
}

Matrix& Matrix::MakePerspective(float d)
{
    MakeIdentity();
    val[10] = 0.0f;
    val[11] = 1.0f/d;
    return *this;
}

Matrix& Matrix::MakePerspectiveKeepZ(float d)
{
    MakeIdentity();
    val[11] = 1.0f/d;
    return *this;
}

// Assignment operator
Matrix& Matrix::operator=(const Matrix& m)
{
    if (this != &m)
    {
        for(int i=0;i<16;++i)
            val[i] = m.val[i];
    }
    return *this;
}

// Assignment operator - set all values to v
Matrix& Matrix::operator=(const float& v)
{
    for(int i=0;i<16;++i)
        val[i] = v;
    return *this;
}

// Assignment operator - get values from an array
Matrix& Matrix::operator=(const float* v)
{
    for(int i=0;i<16;++i)
        val[i] = v[i];
    return *this;
}

// Access operator - get col at the given index
float* Matrix::operator[](unsigned int index)
{
//    if (index > 3)
//        throw "Matrix: Index out of range.";
    return &val[index*4];
}

// Return the internal data pointer.
float* Matrix::GetData(void)
{
    return val;
}

// Addition operator
Matrix Matrix::operator+(Matrix& m)
{
    Matrix r;
    for(int i=0;i<16;++i)
        r.val[i] = val[i] + m.val[i];
    return r;
}

// Subtraction operator
Matrix Matrix::operator-(Matrix& m)
{
    Matrix r;
    for(int i=0;i<16;++i)
        r.val[i] = val[i] - m.val[i];
    return r;
}

// Multiplication operator - matrix*matrix
Matrix Matrix::operator*(Matrix& m)
{
    Matrix r;
    int i, j, k;
    for(i=0;i<4;++i)
        for(j=0;j<4;++j)
            for(k=0;k<4;++k)
                r.val[(j<<2)+i] += val[(k<<2)+i] * m.val[(j<<2)+k];
    return r;
}

// Multiplication operator - matrix*vector -> vector
vuVector Matrix::operator*(vuVector& v)
{
    vuVector r;
    int i, j;
    r[3] = (float)(0.0);
    for(i=0;i<4;++i)
        for(j=0;j<4;++j)
            r.val[i] += val[(j<<2)+i]*v.val[j];
    return r;
}

// Multiplication operator - matrix*scalar
Matrix Matrix::operator*(float s)
{
    Matrix r;
    for(int i=0;i<16;++i)
        r.val[i] = val[i] * s;
    return r;
}

// Multiplication operator - scalar*matrix
Matrix operator*(float s, Matrix& m)
{
    Matrix r;
    for(int i=0;i<16;++i)
        r.val[i] = m.val[i] * s;
    return r;
}

// Add then assign operator
Matrix& Matrix::operator+=(Matrix& m)
{
    for(int i=0;i<16;++i)
        val[i] += m.val[i];
    return *this;
}

// Subtract the assign operator
Matrix& Matrix::operator-=(Matrix& m)
{
    for(int i=0;i<16;++i)
        val[i] -= m.val[i];
    return *this;
}

// Multiply then assign operator - matrix*matrix
Matrix& Matrix::operator*=(const Matrix& m)
{
    Matrix r;
    int i, j, k;
    for(i=0;i<4;++i)
        for(j=0;j<4;++j)
            for(k=0;k<4;++k)
                 r.val[(j<<2)+i] += val[(k<<2)+i] * m.val[(j<<2)+k];
    return (*this=r);
}

// Multiply then assign operator - matrix*scalar
Matrix& Matrix::operator*=(float s)
{
    for(int i=0;i<16;++i)
        val[i] *= s;
    return *this;
}

// Equality operator
bool Matrix::operator==(const Matrix& m) const
{
    for(int i=0;i<16;++i)
        if (val[i] != m.val[i])
            return false;
    return true;
}

// Inequality operator
bool Matrix::operator!=(const Matrix& m) const
{
    return !(operator==(m));
}
