//
// Author: Steve Kilthau
// Date:   January 4, 2001
//

#include "HWTimer.h"

#if defined(WIN32)

    HWTimer::HWTimer()
    {
        LARGE_INTEGER freq;

        if (QueryPerformanceFrequency(&freq))
        {
            m_IsSupported = true;
            m_FloatFreq   = (float)freq.QuadPart;
            m_DoubleFreq  = (double)freq.QuadPart;
        }
        else
        {
            m_IsSupported = false;
            m_FloatFreq  = 1.0f;
            m_DoubleFreq = 1.0;
        }
    }

    HWTimer::HWTimer(HWTimer& t)
    {
        m_LastCount   = t.m_LastCount;
        m_FloatFreq   = t.m_FloatFreq;
        m_DoubleFreq  = t.m_DoubleFreq;
        m_IsSupported = t.m_IsSupported;
    }

    HWTimer::~HWTimer()
    {
    }

    bool HWTimer::IsSupported(void)
    {
        return m_IsSupported;
    }

    float HWTimer::GetElapsedTimeFloat(void)
    {
        LARGE_INTEGER count;
        float time;

        QueryPerformanceCounter(&count);
        time = (float)(count.QuadPart - m_LastCount.QuadPart)/m_FloatFreq;
        m_LastCount = count;

        return time;
    }

    double HWTimer::GetElapsedTimeDouble(void)
    {
        LARGE_INTEGER count;
        double time;

        QueryPerformanceCounter(&count);
        time = (double)(count.QuadPart - m_LastCount.QuadPart)/m_DoubleFreq;
        m_LastCount = count;

        return time;
    }

    void HWTimer::SleepFloat(float s)
    {
        LARGE_INTEGER count;
        float time = -1.0f;

        if (s < 0.0f)
            return;

        while(time < s)
        {
            QueryPerformanceCounter(&count);
            time = (float)(count.QuadPart - m_LastCount.QuadPart)/m_FloatFreq;
        }
        m_LastCount = count;

        return;
    }

    void HWTimer::SleepDouble(double s)
    {
        LARGE_INTEGER count;
        double time = -1.0f;

        if (s < 0.0f)
            return;

        while(time < s)
        {
            QueryPerformanceCounter(&count);
            time = (double)(count.QuadPart - m_LastCount.QuadPart)/m_DoubleFreq;
        }
        m_LastCount = count;

        return;
    }

    HWTimer& HWTimer::operator=(HWTimer& rhs)
    {
        if (this != &rhs)
        {
            m_LastCount   = rhs.m_LastCount;
            m_FloatFreq   = rhs.m_FloatFreq;
            m_DoubleFreq  = rhs.m_DoubleFreq;
            m_IsSupported = rhs.m_IsSupported;
        }
        return *this;
    }

#else

#define TIMEVALDIFFFLOAT(t1, t2)                             \
          (((((float)(t1.tv_sec-t2.tv_sec))*1000000.0f) +     \
             ((float)(t1.tv_usec-t2.tv_usec)))/1000000.0f)
#define TIMEVALDIFFDOUBLE(t1, t2)                            \
          (((((double)(t1.tv_sec-t2.tv_sec))*1000000.0) +     \
             ((double)(t1.tv_usec-t2.tv_usec)))/1000000.0)

    HWTimer::HWTimer()
    {
        struct timeval tv;

        m_IsSupported = (gettimeofday(&tv, 0) == 0);
    }

    HWTimer::HWTimer(HWTimer& t)
    {
        m_LastCount.tv_sec   = t.m_LastCount.tv_sec;
        m_LastCount.tv_usec  = t.m_LastCount.tv_usec;
        m_IsSupported        = t.m_IsSupported;
    }

    HWTimer::~HWTimer()
    {
    }

    bool HWTimer::IsSupported(void)
    {
        return m_IsSupported;
    }

    float HWTimer::GetElapsedTimeFloat(void)
    {
        struct timeval count;
        float time;

        gettimeofday(&count, 0);
        time = TIMEVALDIFFFLOAT(count, m_LastCount);
        m_LastCount.tv_sec  = count.tv_sec;
        m_LastCount.tv_usec = count.tv_usec;

        return time;
    }

    double HWTimer::GetElapsedTimeDouble(void)
    {
        struct timeval count;
        double time;

        gettimeofday(&count, 0);
        time = TIMEVALDIFFDOUBLE(count, m_LastCount);
        m_LastCount.tv_sec  = count.tv_sec;
        m_LastCount.tv_usec = count.tv_usec;

        return time;
    }

    void HWTimer::SleepFloat(float s)
    {
        struct timeval count;
        float time = -1.0f;

        if (s < 0.0f)
            return;

        while(time < s)
        {
            gettimeofday(&count, 0);
            time = TIMEVALDIFFFLOAT(count, m_LastCount);
        }
        m_LastCount.tv_sec  = count.tv_sec;
        m_LastCount.tv_usec = count.tv_usec;

        return;
    }

    void HWTimer::SleepDouble(double s)
    {
        struct timeval count;
        double time = -1.0f;

        if (s < 0.0f)
            return;

        while(time < s)
        {
            gettimeofday(&count, 0);
            time = TIMEVALDIFFDOUBLE(count, m_LastCount);
        }
        m_LastCount.tv_sec  = count.tv_sec;
        m_LastCount.tv_usec = count.tv_usec;

        return;
    }

    HWTimer& HWTimer::operator=(HWTimer& rhs)
    {
        if (this != &rhs)
        {
            m_LastCount.tv_sec   = rhs.m_LastCount.tv_sec;
            m_LastCount.tv_usec  = rhs.m_LastCount.tv_usec;
            m_IsSupported        = rhs.m_IsSupported;
        }
        return *this;
    }

#endif
