#include "vuFourierVolume_IO.h"
#include "Volume/Fourier/Unimodal/3d/3d.h"

#define MAX_FILTER_SIZE 6

template <int S> bool vuFourierVolume_IO<S>::
preprocessSpatialInput(byte *data, dword XSize, dword YSize, dword ZSize,
		       float s, float mult_pad,	dword add_pad,
		       bool doWrapAndInit)
{
  dword dim = computeDimensions(XSize, YSize, ZSize, mult_pad, add_pad);

  m_XSize  = m_YSize = m_ZSize = dim;
  m_Volume = new float[m_XSize * m_YSize * m_ZSize * 2 * S];

  // transporm bytes into complex numbers + do some padding
  readSpatial(data, m_Volume, m_XSize, m_YSize, m_ZSize, XSize, YSize, ZSize);

  // apply fft on spatial domain
  preprocess();

  if (doWrapAndInit) {
    setWrap(MAX_FILTER_SIZE/2);

    XSize   = m_XSize;
    YSize   = m_YSize;
    ZSize   = m_ZSize;

    m_XSize = XSize + m_Wrap * 2;
    m_YSize = YSize + m_Wrap * 2;
    m_ZSize = ZSize + m_Wrap * 2;

    float *dest = new float[m_XSize * m_YSize * m_ZSize * 2 * S];
  
    // make m_Volume larger (according to m_Wrap)
    padFourier(m_Volume, dest, m_XSize, m_YSize, m_ZSize, XSize, YSize, ZSize);
    CHECKNDELETE(m_Volume);
    m_Volume = dest;

    wrapAndInitialize(s);
  }
  else
    m_Wrap = 0;

  return true;
}

template <int S>
void vuFourierVolume_IO<S>::preprocess()
{
  cout << "shifting volume. " << flush;
  shift3D(m_Volume, m_XSize, m_YSize, m_ZSize);

  cout << "transforming volume. " << flush;
  transform3D(m_Volume, m_XSize, m_YSize, m_ZSize);

  cout << "shifting volume. " << flush;
  shift3D(m_Volume, m_XSize, m_YSize, m_ZSize);
  cout << "done\n" << flush;
}

template <int S> void vuFourierVolume_IO<S>::
transform3D(float* vol, dword xx, dword yy, dword zz)
{
  fftwnd_plan plan;

  plan = fftw3d_create_plan(xx, yy, zz, FFTW_FORWARD,
			    FFTW_ESTIMATE | FFTW_IN_PLACE | FFTW_USE_WISDOM);

  fftwnd(plan, S, (fftw_complex*)vol, S, 1, NULL, 0, 0);

  fftwnd_destroy_plan(plan);
  dword count  =  xx * yy * zz * 2 * S;
  float size_f = (float) xx* yy * zz;
  float *ptr   = vol;
  for (dword i = 0; i < count; ++i) *(ptr++) /= size_f;
}

template <int S> void vuFourierVolume_IO<S>::
inverseTransform3D(float* vol, dword xx, dword yy, dword zz)
{
  fftwnd_plan plan;

  plan = fftw3d_create_plan(xx, yy, zz, FFTW_BACKWARD,
			    FFTW_ESTIMATE | FFTW_IN_PLACE | FFTW_USE_WISDOM);

  fftwnd(plan, S, (fftw_complex*)vol, S, 1, NULL, 0, 0);

  fftwnd_destroy_plan(plan);
}


template <int S> void vuFourierVolume_IO<S>::
shift3D(float* data,dword XSize,dword YSize,dword ZSize)
{
  dword xstep  = XSize * 2 * S;
  dword ystep  = YSize * XSize * 2 * S;
  dword index1 = 0;
  int   start1 = 0;
  int   val1   = -1;

  dword index2;
  int   val2, start2;

  for(dword k=0; k<ZSize; k++) {
    index2 = index1;
    start2 = start1;
    val2   = val1;
    for(dword j=0; j<YSize; j++) {
      for(dword i=start2; i<xstep; i+=4*S) {
	float *ptr = &data[index2+i];
	for (int l=0; l<S; l++) {
	  *(ptr++) *= -1.0f;
	  *(ptr++) *= -1.0f;
	}
      }
      val2 *= -1;
      start2 += val2 * 2 * S;
      index2 += xstep;
    }
    val1 *= -1;
    start1 += val1*2*S;
    index1 += ystep;
  }
}

template <int S> bool vuFourierVolume_IO<S>::
scaleAndWriteToFourierFile(const char *fileName, float scale)
{
  if (scale !=1.0 && scale > 0 || true) {
    dword count = m_XSize * m_YSize * m_ZSize * 2 * S;
    float *ptr  = m_Volume;
    for (dword i=0; i<count; i++) *(ptr++) *= scale;
  }

  FILE *file = fopen(fileName, "wb");
  if (file == NULL) return false;

  dword XSize = m_XSize - 2 * m_Wrap;
  dword YSize = m_YSize - 2 * m_Wrap;
  dword ZSize = m_ZSize - 2 * m_Wrap;

  bool status = vu1712<S>::writeHeader(file, XSize, YSize, ZSize);
  if (status) {
    status = vu1712<S>::writeData(file, m_Volume, 
				  m_XSize, m_YSize, m_ZSize,
				  XSize, YSize, ZSize);
  }
  fclose(file);
  return status;
}

/*
   in  -> input data  (spatial domain volume)
   out -> output data (fourier domain volume)
*/
template <int S>
void vuFourierVolume_IO<S>::readSpatial(byte *in, float* out,
                                   dword XX, dword YY, dword ZZ,
                                   dword XXsmall, dword YYsmall, dword ZZsmall)
{
  byte  *i_ptr = in;
  float *v_ptr = out;

  dword lowX = (XX - XXsmall) / 2;
  dword lowY = (YY - YYsmall) / 2;
  dword lowZ = (ZZ - ZZsmall) / 2;

  dword hiX = XX - lowX - XXsmall;
  dword hiY = YY - lowY - YYsmall;
  dword hiZ = ZZ - lowZ - ZZsmall;

  v_ptr = pad(v_ptr, lowZ * XX * YY * 2 * S);
  for (dword k = 0; k < ZZsmall; k++) {
    v_ptr = pad(v_ptr, lowY * XX * 2 * S);
    for (dword j = 0; j < YYsmall; j++) {
      v_ptr = pad(v_ptr, lowX * 2 * S);
      for (dword i = 0; i < XXsmall; i++) {
	for (int l=0; l<S; l++) {
	  *(v_ptr++) = *(i_ptr++);
	  *(v_ptr++) = 0.0f;
	}
      }
      v_ptr = pad(v_ptr, hiX * 2 * S);
    }
    v_ptr = pad(v_ptr, hiY * XX * 2 * S);
  }
  v_ptr = pad(v_ptr, hiZ * XX * YY * 2 * S);
}

// complex pad n spaces
template <int S>
float* vuFourierVolume_IO<S>::pad(float* v, dword n)
{
  for (dword i = 0; i < n; ++i) *(v++) = 0.0f;
  return v;
}

template <int S>
void vuFourierVolume_IO<S>::padFourier(float *in, float* out,
                                   dword XX, dword YY, dword ZZ,
				   dword XXsmall, dword YYsmall, dword ZZsmall)
{
  float *i_ptr = in;
  float *v_ptr = out;

  dword lowX = (XX - XXsmall) / 2;
  dword lowY = (YY - YYsmall) / 2;
  dword lowZ = (ZZ - ZZsmall) / 2;

  dword hiX = XX - lowX - XXsmall;
  dword hiY = YY - lowY - YYsmall;
  dword hiZ = ZZ - lowZ - ZZsmall;

  v_ptr = pad(v_ptr, lowZ * XX * YY * 2 * S);
  for (dword k = 0; k < ZZsmall; k++) {
    v_ptr = pad(v_ptr, lowY * XX * 2 * S);
    for (dword j = 0; j < YYsmall; j++) {
      v_ptr = pad(v_ptr, lowX * 2 * S);
      for (dword i = 0; i < XXsmall; i++) {
	for (int l=0; l<S; l++) {
	  *(v_ptr++) = *(i_ptr++);
	  *(v_ptr++) = *(i_ptr++);
	}
      }
      v_ptr = pad(v_ptr, hiX * 2 * S);
    }
    v_ptr = pad(v_ptr, hiY * XX * 2 * S);
  }
  v_ptr = pad(v_ptr, hiZ * XX * YY * 2 * S);
}

template <int S>
bool vuFourierVolume_IO<S>::getSpatialDataFromVUF(byte *&spatVol,
						  dword &XSize,
						  dword &YSize,
						  dword &ZSize,
						  float scale,
						  vuString fileName)
{
  CHECKNDELETE(spatVol);

  if (S != 1) {
    cerr << "vuFourierVolume_IO::getSpatialDataFromVUF(): ";
    cerr << "only S=1 is supported" << endl;
    return false;
  }

  float *freqVol = NULL;
  bool  status   = true;
  if (fileName.isEmpty()) return false;
  FILE *file  = fopen(fileName, "rb");
  if (file == NULL) return false;

  status = vu1712<S>::readHeader(file, XSize, YSize, ZSize);

  if (status) {
    freqVol = new float[XSize * YSize * ZSize * S * 2];
    status = vu1712<S>::readData(file, freqVol,
				 XSize, YSize, ZSize,
				 XSize, YSize, ZSize);
  }
  fclose(file);

  if (status) {
    vuFourierVolume_IO<S>::inverseTransform3D(freqVol, XSize, YSize, ZSize);

    dword count  = XSize * YSize * ZSize;
    float *s_ptr = freqVol;
    spatVol      = new byte[count];
    byte  *d_ptr = spatVol;

    if (scale == 0) scale = 1.0;

    for (dword i=0; i<count; i++) {
      register float tmp = *s_ptr * scale;

      if (tmp > 255)
	tmp = 255.0f;
      else if (tmp < 0)
	tmp = 0.0f;

      *d_ptr = (byte)tmp;

      d_ptr += 1;
      s_ptr += 2;
    }
  }

  CHECKNDELETE(freqVol);
  return status;
}
