////////////////////////////////////////////////////////////////////////
//                                                                    //
// Informatikpraktikum I "Implementation of the ShearWarp Algorithm"  //
// Februar - October 2002                                             //
//                                                                    //
// author: Sebastian Zambal                                           //
//         e9826978@student.tuwien.ac.at                              //
//                                                                    //
// file:   shearWarp.cpp (regular)                                    //
//                                                                    //
////////////////////////////////////////////////////////////////////////

#include "shearWarp.h"
#include "vuGui/vuDrawTools.h"
#include "../simpleFuncs.h"
#include <fstream.h>
#include <math.h>

//-----------------------------------------------------------------------
//-- Constructor                                                      ---
//-----------------------------------------------------------------------
vu111211A::vu111211A()
{
    //initially no zoom
    zoomValue = 1;

    //initially don't use OpenGL for warping
    orthoWarpOpenGL = 0;

    //initially no normals
    m_Normals = 0;

    //initially there is no runlength encoded data
    dataX = 0;
    dataY = 0;
    dataZ = 0;

    //initially no current runlength encoding
    curr = 0;
    //initially no intermediate image
    intermediate = 0;
    //initially orthogonal projection
    viewingMode = VIEWING_MODE_ORTHO;
    //initially no specular shading
    specular = 0;
    //initially no fast classification
    fastClassification = 0;

    m_GLShearWarp = 0;
}

//-----------------------------------------------------------------------
//-- Destructor                                                       ---
//-----------------------------------------------------------------------
vu111211A::~vu111211A() 
{
    if (glImage) {
	delete [] glImage;
	glImage = 0;
    }
    if (m_Normals) delete [] m_Normals;
    if (dataX) {
	for (dword i = 0; i < m_Dim1Size; ++i) { 
	    if (dataX[i].size>0) {
		delete [] dataX[i].voxel;
		delete [] dataX[i].runlength;
	    }
	}
	delete [] dataX;
    }
    if (dataY) {
	for (dword i = 0; i < m_Dim2Size; ++i) { 
	    if (dataY[i].size>0) {
		delete [] dataY[i].voxel;
		delete [] dataY[i].runlength;
	    }
	}
	delete [] dataY;
    }
    if (dataZ) {
	for (dword i = 0; i < m_Dim3Size; ++i) { 
            if (dataZ[i].size>0) {
		delete [] dataZ[i].voxel;
		delete [] dataZ[i].runlength;
	    }
	}
	delete [] dataZ;
    }
    if (intermediate) {
	delete [] intermediate;
    }
    if (fastClassificationObj) {
	delete fastClassificationObj;
    }
}

//-----------------------------------------------------------------------
//-- returns the size of dimensions of the volume data                ---
//-----------------------------------------------------------------------
void vu111211A::getDimensions(int &x, int &y, int &z) {
    x = m_Dim1Size;
    y = m_Dim2Size;
    z = m_Dim3Size;
}

//-----------------------------------------------------------------------
//-- Construct OpenGL image                                           ---
//-----------------------------------------------------------------------
void vu111211A::createGLImage(void) {

    word glSize = 64;
    while (glSize < maxSize*2) {
	glSize *= 2;
    }
//    glSize *= 2;

    glImageWidth = glSize;
    glImageHeight = glSize;
    glImage = new GLubyte[glImageWidth*glImageHeight*4];
}

//-----------------------------------------------------------------------
//-- Compute maximum stretch-out of volume                            ---
//-----------------------------------------------------------------------
void vu111211A::computeMaxSize(void) 
{
    maxSize = m_Dim1Size;
    if (m_Dim2Size > maxSize) maxSize = m_Dim2Size;
    if (m_Dim3Size > maxSize) maxSize = m_Dim3Size;
    maxSize += 2;                         
}

//-----------------------------------------------------------------------
//-- Set view vectors                                                 ---
//-----------------------------------------------------------------------
void vu111211A::setViewVectors(const vuVector& view, const vuVector& up, const vuVector& right)
{
  vuVector view1;
  vuVector up1;
  vuVector right1;

  m_View = view;

  m_Up = up;
  m_Right = right;

}

//-----------------------------------------------------------------------
//-- Set viewing Mode                                                 ---
//-----------------------------------------------------------------------
void vu111211A::setViewing(int vm)
{
    viewingMode = vm;
}

//-----------------------------------------------------------------------
//-- Turn on/off specular light                                       ---
//-----------------------------------------------------------------------
void vu111211A::setSpecular(int spec)
{
    specular = spec;
}

//-----------------------------------------------------------------------
//-- Turn on/off fast classification                                  ---
//-----------------------------------------------------------------------
void vu111211A::setFastClassification(int fastClass)
{
    fastClassification = fastClass;
    if (fastClassification == 1) {
	initFastClassification();
    } else {
	runlengthEncode();
    }
}

//-----------------------------------------------------------------------
//-- Get the classification mode                                      ---
//-----------------------------------------------------------------------
int vu111211A::getFastClassification()
{
    return fastClassification;
}

//-----------------------------------------------------------------------
//-- Classifies the volume represented by the octree                  ---
//-----------------------------------------------------------------------
void vu111211A::classify() 
{
    fastClassificationObj->classify();
}

//-----------------------------------------------------------------------
//-- Zoom                                                             ---
//-----------------------------------------------------------------------
void vu111211A::zoom(float value) 
{
    if (value > 0) zoomValue = value;
}

void vu111211A::setOrthogonalWarpOpenGL(int useOpenGL) 
{
    orthoWarpOpenGL = useOpenGL;
}

//-----------------------------------------------------------------------
//-- Set size of canvas                                               ---
//-----------------------------------------------------------------------
void vu111211A::setCanvasSize(int width, int height) 
{
    canvasWidth = width;
    canvasHeight = height;
}

//-----------------------------------------------------------------------
//-- Reimplements the read() method to do some extra volume data      ---
//-- processing.                                                      ---
//-----------------------------------------------------------------------
//!Reimplements the read() method to do some extra volume data processing
bool vu111211A::read()
{
    bool success = vu111211::read();
    if (!success) return false;

    //Allocate memory for the normals.
    if (m_Normals != 0) delete [] m_Normals;

    m_Normals = new float[m_DataSize*3];

    computeMaxSize();

    //initially value for distance between eye and projection plane
    eye_distance = getMinEyeDistance();

    createGLImage();
    computeNormals();

    //Allocate memory for intermediate image
    intermediate = new intermediatePixel[maxSize*maxSize*4];

    if (fastClassification == 1) {
	initFastClassification();
    } else {
	runlengthEncode();
    }

    return true;
}

//-----------------------------------------------------------------------
//-- Implements readRaw                                               ---
//-----------------------------------------------------------------------
bool vu111211A::readRaw(void)
{
    dword len;
    ifstream in;

    in.open(m_FileName, ios::in|ios::binary
#ifdef IS_NOCREATE_NEEDED
	    |ios::nocreate
#endif
	);
    if (!in.is_open()) return false;

    in.seekg(0, ios::end);
    len = in.tellg();
    in.seekg(0, ios::beg);

    in >> m_Dim1Size >> m_Dim2Size >> m_Dim3Size;
    if (in.fail()) return false;
    m_DataSize = m_Dim1Size*m_Dim2Size*m_Dim3Size;

    m_Data = new byte[m_DataSize];
    in.read((char*)m_Data, m_DataSize);
    if (in.fail()) return false;

    in.close();

    //Allocate memory for the normals.
    if (m_Normals != 0) delete [] m_Normals;

    m_Normals = new float[m_DataSize];

    computeNormals();

    return true;
}

//-----------------------------------------------------------------------
//-- Removes the runlength encoding                                   ---
//-----------------------------------------------------------------------
void vu111211A::removeRunlengthEncoding() 
{
    // if neccessary delete existing runlength-encoding
    if (dataX) {
	for (dword i = 0; i < m_Dim1Size; ++i) { 
	    if (dataX[i].size>0) {
		delete [] dataX[i].voxel;
		delete [] dataX[i].runlength;
	    }
	}
	delete [] dataX;
    }
    if (dataY) {
	for (dword i = 0; i < m_Dim2Size; ++i) { 
	    if (dataY[i].size>0) {
		delete [] dataY[i].voxel;
		delete [] dataY[i].runlength;
	    }
	}
	delete [] dataY;
    }
    if (dataZ) {
	for (dword i = 0; i < m_Dim3Size; ++i) { 
            if (dataZ[i].size>0) {
		delete [] dataZ[i].voxel;
		delete [] dataZ[i].runlength;
	    }
	}
	delete [] dataZ;
    }
}

//-----------------------------------------------------------------------
//-- Do the runlength encoding                                        ---
//-----------------------------------------------------------------------
void vu111211A::runlengthEncode() 
{
    //If there exists already a runlength encoding, remove it
    removeRunlengthEncoding();

    dword voxels = 0;           // the number of voxels in the current plane
    dword runs = 0;             // the number of runs in the current plane
    byte lengthr = 0;           // length of the run
    byte state = 0;             // are we in a run of transparent (0) 
                                // or non-transparent (1) voxels?
    dword voxelIndex = 0;       // index of voxel in array voxel
    dword runIndex = 0;         // index of run in array runlength
    dword indexRunlength = 0;   // index in the runlength encoded data
    dword indexVolume = 0;      // index in the original volume data
    dword maxPlaneSize = m_Dim2Size * m_Dim1Size;  // uncompressed plane size

    dataZ = new RLEplane[m_Dim3Size];

    for (word z = 0; z < m_Dim3Size; ++z) {

	voxels = 0;
	runs = 0;
	state = 0;
	lengthr = 0;

        // find out the number of voxels and runs in the plane
	for (dword i = 0; i < maxPlaneSize; ++i) {
            if (m_TFunc[m_Data[maxPlaneSize*z + i]][3] > THRESHOLD_RUNLENGTH) {
		++voxels;
		if (state == 0) {
		    ++runs;
		    lengthr = 0;
		    state = 1;
		}
	    } else {
		if (state == 1) {
		    ++runs;
		    lengthr = 0;
		    state = 0;
		}
	    }
	    if ((lengthr == 255) || ((i+1) % m_Dim1Size == 0)) {
		runs += 3;
		lengthr = 0;
		if (state == 0) state = 1; else state = 0;
	    }
	}

        // set data of the plane
	dataZ[z].size = voxels;
	dataZ[z].dim1_pos = 0;
	dataZ[z].dim2_pos = 0;
	dataZ[z].scale = 1;

        // if plane is not empty...
	if (voxels > 0) {

	    dataZ[z].voxel = new RLEvoxel[voxels+1];
	    dataZ[z].runlength = new byte[runs+1];

	    state = 0;
	    indexRunlength = 0;
	    voxelIndex = 0;
	    runIndex = 0;

	    dataZ[z].runlength[0] = 0; // init first runlength

            // construct data for current plane
	    for (dword i = 0; i < maxPlaneSize; ++i) {

		if (m_TFunc[m_Data[maxPlaneSize*z + i]][3] > THRESHOLD_RUNLENGTH) {

		    byte val = m_Data[maxPlaneSize*z + i];
		    dataZ[z].voxel[voxelIndex].value = val;

		    dataZ[z].voxel[voxelIndex].normal[XDIR] = 
			m_Normals[(maxPlaneSize*z + i)*3 + XDIR];
		    dataZ[z].voxel[voxelIndex].normal[YDIR] = 
			m_Normals[(maxPlaneSize*z + i)*3 + YDIR];
		    dataZ[z].voxel[voxelIndex].normal[ZDIR] = 
			m_Normals[(maxPlaneSize*z + i)*3 + ZDIR];

		    dataZ[z].voxel[voxelIndex].red = m_TFunc[val][0];
		    dataZ[z].voxel[voxelIndex].green = m_TFunc[val][1];
		    dataZ[z].voxel[voxelIndex].blue = m_TFunc[val][2];
		    dataZ[z].voxel[voxelIndex].opacity = m_TFunc[val][3];

		    ++voxelIndex;
		    if (state == 0) {
			++runIndex;
			dataZ[z].runlength[runIndex] = 0;
			state = 1;
		    }
		    ++dataZ[z].runlength[runIndex];
		} else {
		    if (state == 1) {
			++runIndex;
			dataZ[z].runlength[runIndex] = 0;
			state = 0;
		    }
		    ++dataZ[z].runlength[runIndex];
		}
		if ((dataZ[z].runlength[runIndex] == 255) || ((i+1) % m_Dim1Size == 0)) {
		    ++runIndex;
		    dataZ[z].runlength[runIndex] = 0;
		    if (state == 0) state = 1; else state = 0;
		}
	    }

	} else {
	    dataZ[z].voxel = 0;
	    dataZ[z].runlength = 0;
	}
    }

    //--------------------------------------------------------------
    // Now we do the same for the main viewing direction in y

    dword start = 0;
    indexVolume = 0;
    maxPlaneSize = m_Dim1Size * m_Dim3Size;  // uncompressed plane size

    dataY = new RLEplane[m_Dim2Size];

    for (word y = 0; y < m_Dim2Size; ++y) {

	voxels = 0;
	runs = 0;
	state = 0;
	lengthr = 0;

	dword step = m_Dim1Size * m_Dim2Size;  // size of the steps in a row
	dword row_length = (m_Dim3Size-1)*step;  // length of a row


	indexVolume = y * m_Dim1Size;

	start = indexVolume;

        // find out the number of voxels and runs in the plane
	for (dword i = 0; i < maxPlaneSize; ++i) {
            if (m_TFunc[m_Data[indexVolume]][3] > THRESHOLD_RUNLENGTH) {
		++voxels;
		if (state == 0) {
		    ++runs;
		    lengthr = 0;
		    state = 1;
		}
	    } else {
		if (state == 1) {
		    ++runs;
		    lengthr = 0;
		    state = 0;
		}
	    }
	    if ((lengthr == 255) || ((i+1) % m_Dim3Size == 0)) {
		runs += 3;
		lengthr = 0;
		if (state == 0) state = 1; else state = 0;
	    }


	    if (i > 0) {
                //if one row is finished -> increment start position for next row and
                //continue at this new start position
	        if (indexVolume >= y*m_Dim1Size+row_length) {
	    	++start;
		    indexVolume = start;
                //...otherwise just go on tracing through the row
	        } else if (i > 0) {    
		    indexVolume += step;
		}
	    }
	}

        // set data of the plane
	dataY[y].size = voxels;
	dataY[y].dim1_pos = 0;
	dataY[y].dim2_pos = 0;
	dataY[y].scale = 1;

        // if plane is not empty...
	if (voxels > 0) {

	    dataY[y].voxel = new RLEvoxel[voxels+1];
	    dataY[y].runlength = new byte[runs+1];

	    state = 0;
	    indexRunlength = 0;
	    voxelIndex = 0;
	    runIndex = 0;

	    dataY[y].runlength[0] = 0; // init first runlength

	    indexVolume = y * m_Dim1Size;
	    start = indexVolume;

            // construct data for current plane
	    for (dword i = 0; i < maxPlaneSize; ++i) {

		if (i > 0) {
                    //if one row is finished -> increment start position for next row and
                    //continue at this new start position
		    if (indexVolume >= y*m_Dim1Size+row_length) {
			++start;
			indexVolume = start;
                    //...otherwise just go on tracing through the row
		    } else if (i > 0) {    
			indexVolume += step;
		    }
		}

		if (m_TFunc[m_Data[indexVolume]][3] > THRESHOLD_RUNLENGTH) {

		    byte val = m_Data[indexVolume];
		    dataY[y].voxel[voxelIndex].value = val;
		    dataY[y].voxel[voxelIndex].normal[XDIR] = m_Normals[indexVolume*3 + XDIR];
		    dataY[y].voxel[voxelIndex].normal[YDIR] = m_Normals[indexVolume*3 + YDIR];
		    dataY[y].voxel[voxelIndex].normal[ZDIR] = m_Normals[indexVolume*3 + ZDIR];
		    dataY[y].voxel[voxelIndex].red = m_TFunc[val][0];
		    dataY[y].voxel[voxelIndex].green = m_TFunc[val][1];
		    dataY[y].voxel[voxelIndex].blue = m_TFunc[val][2];
		    dataY[y].voxel[voxelIndex].opacity = m_TFunc[val][3];

		    ++voxelIndex;
		    if (state == 0) {
			++runIndex;
			dataY[y].runlength[runIndex] = 0;
			state = 1;
		    }
		    ++dataY[y].runlength[runIndex];
		} else {
		    if (state == 1) {
			++runIndex;
			dataY[y].runlength[runIndex] = 0;
			state = 0;
		    }
		    ++dataY[y].runlength[runIndex];
		}

                //if the capacity of the current runlength is exhausted or if there
                //is a line-break: increase runIndex (new run with length 0) and change 
                //current state 
		if ((dataY[y].runlength[runIndex] == 255) || ((i+1) % m_Dim3Size == 0)) {
		    ++runIndex;
		    dataY[y].runlength[runIndex] = 0;
		    if (state == 0) state = 1; else state = 0;
		}
	    }

	} else {
	    dataY[y].voxel = 0;
	    dataY[y].runlength = 0;
	}
    }

    //--------------------------------------------------------------
    // Now we do the same for the main viewing direction in x

    indexVolume = 0;
    maxPlaneSize = m_Dim2Size * m_Dim3Size;  // uncompressed plane size

    dataX = new RLEplane[m_Dim1Size];

    for (word x = 0; x < m_Dim1Size; ++x) {

	voxels = 0;
	runs = 0;
	state = 0;
	lengthr = 0;

	indexVolume = x;

        // find out the number of voxels and runs in the plane
	for (dword i = 0; i < maxPlaneSize; ++i) {
            if (m_TFunc[m_Data[indexVolume]][3] > THRESHOLD_RUNLENGTH) {
		++voxels;
		if (state == 0) {
		    ++runs;
		    lengthr = 0;
		    state = 1;
		}
	    } else {
		if (state == 1) {
		    ++runs;
		    lengthr = 0;
		    state = 0;
		}
	    }
	    if ((lengthr == 255) || ((i+1) % m_Dim2Size == 0)) {
		runs += 3;
		lengthr = 0;
		if (state == 0) state = 1; else state = 0;
	    }

	    indexVolume += m_Dim1Size;
	}

        // set data of the plane
	dataX[x].size = voxels;
	dataX[x].dim1_pos = 0;
	dataX[x].dim2_pos = 0;
	dataX[x].scale = 1;

        // if plane is not empty...
	if (voxels > 0) {

	    dataX[x].voxel = new RLEvoxel[voxels+1];
	    dataX[x].runlength = new byte[runs+1];

	    state = 0;
	    indexRunlength = 0;
	    voxelIndex = 0;
	    runIndex = 0;

	    dataX[x].runlength[0] = 0; // init first runlength

	    indexVolume = x;

            // construct data for current plane
	    for (dword i = 0; i < maxPlaneSize; ++i) {

		if (m_TFunc[m_Data[indexVolume]][3] > THRESHOLD_RUNLENGTH) {

		    byte val = m_Data[indexVolume];
		    dataX[x].voxel[voxelIndex].value = val;
		    dataX[x].voxel[voxelIndex].normal[XDIR] = m_Normals[indexVolume*3 + XDIR];
		    dataX[x].voxel[voxelIndex].normal[YDIR] = m_Normals[indexVolume*3 + YDIR];
		    dataX[x].voxel[voxelIndex].normal[ZDIR] = m_Normals[indexVolume*3 + ZDIR];

		    dataX[x].voxel[voxelIndex].red = m_TFunc[val][0];
		    dataX[x].voxel[voxelIndex].green = m_TFunc[val][1];
		    dataX[x].voxel[voxelIndex].blue = m_TFunc[val][2];
		    dataX[x].voxel[voxelIndex].opacity = m_TFunc[val][3];

		    ++voxelIndex;
		    if (state == 0) {
			++runIndex;
			dataX[x].runlength[runIndex] = 0;
			state = 1;
		    }
		    ++dataX[x].runlength[runIndex];
		} else {
		    if (state == 1) {
			++runIndex;
			dataX[x].runlength[runIndex] = 0;
			state = 0;
		    }
		    ++dataX[x].runlength[runIndex];
		}
		if ((dataX[x].runlength[runIndex] == 255) || ((i+1) % m_Dim2Size == 0)) {
		    ++runIndex;
		    dataX[x].runlength[runIndex] = 0;
		    if (state == 0) state = 1; else state = 0;
		}

		indexVolume += m_Dim1Size;
	    }

	} else {
	    dataX[x].voxel = 0;
	    dataX[x].runlength = 0;
	}
    }
}

//-----------------------------------------------------------------------
//-- Preprocesses volume data for rendering once it's been read       ---
//-----------------------------------------------------------------------
void vu111211A::computeNormals(void)
{
    float norm[3];
    float len;
    dword w, wh;
    dword i, j, k, index;

    w = m_Dim1Size;
    wh = m_Dim1Size*m_Dim2Size;

    //
    // Compute all of the normals
    //
    index = 0;

    for(k=0;k<m_Dim3Size;++k)
    {
        for(j=0;j<m_Dim2Size;++j)
        {
            for(i=0;i<m_Dim1Size;++i)
            {
                if (i == 0) {
                    norm[0] = m_Data[index+1]-m_Data[index];
		}
                else if (i == m_Dim1Size-1) {
                    norm[0] = m_Data[index]-m_Data[index-1];
		}
                else {
                    norm[0] = (m_Data[index+1]-m_Data[index-1])*1;

		    if ((j > 0) && (j < m_Dim2Size) && (k > 0) && (k < m_Dim3Size)) {
		    norm[0] += (m_Data[index  +w +1]-m_Data[index  +w -1])*0.2;
		    norm[0] += (m_Data[index  -w +1]-m_Data[index  -w -1])*0.2;
		    norm[0] += (m_Data[index +wh +1]-m_Data[index +wh -1])*0.2;
		    norm[0] += (m_Data[index -wh +1]-m_Data[index -wh -1])*0.2;

		    norm[0] += (m_Data[index +wh +w +1]-m_Data[index +wh +w -1])*0.1;
		    norm[0] += (m_Data[index +wh -w +1]-m_Data[index +wh -w -1])*0.1;
		    norm[0] += (m_Data[index -wh +w +1]-m_Data[index -wh +w -1])*0.1;
		    norm[0] += (m_Data[index -wh -w +1]-m_Data[index -wh -w -1])*0.1;
		    }
		}

                if (j == 0) {
                    norm[1] = m_Data[index+w]-m_Data[index];
		}
                else if (j == m_Dim2Size-1) {
                    norm[1] = m_Data[index]-m_Data[index-w];
		}
                else {
                    norm[1] = (m_Data[index+w]-m_Data[index-w])*1;

		    if ((i > 0) && (i < m_Dim1Size) && (k > 0) && (k < m_Dim3Size)) {
                    norm[1] += (m_Data[index  +1 +w]-m_Data[index +1  -w])*0.2;
                    norm[1] += (m_Data[index  -1 +w]-m_Data[index -1  -w])*0.2;
                    norm[1] += (m_Data[index +wh +w]-m_Data[index +wh -w])*0.2;
                    norm[1] += (m_Data[index -wh +w]-m_Data[index -wh -w])*0.2; 

                    norm[1] += (m_Data[index +wh +1 +w]-m_Data[index +wh +1 -w])*0.1;
                    norm[1] += (m_Data[index +wh -1 +w]-m_Data[index +wh -1 -w])*0.1;
                    norm[1] += (m_Data[index -wh +1 +w]-m_Data[index -wh +1 -w])*0.1;
                    norm[1] += (m_Data[index -wh -1 +w]-m_Data[index -wh -1 -w])*0.1;
		    }
		}

                if (k == 0) {
                    norm[2] = m_Data[index+wh]-m_Data[index];
		}
                else if (k == m_Dim3Size-1) {
                    norm[2] = m_Data[index]-m_Data[index-wh];
		}
                else {
                    norm[2] = (m_Data[index+wh]-m_Data[index-wh])*1;

		    if ((i > 0) && (i < m_Dim1Size) && (j > 0) && (j < m_Dim2Size)) {
                    norm[2] += (m_Data[index +1 +wh]-m_Data[index +1 -wh])*0.2;
                    norm[2] += (m_Data[index -1 +wh]-m_Data[index -1 -wh])*0.2;
                    norm[2] += (m_Data[index +w +wh]-m_Data[index +w -wh])*0.2;
                    norm[2] += (m_Data[index -w +wh]-m_Data[index -w -wh])*0.2;

                    norm[2] += (m_Data[index +w +1 +wh]-m_Data[index +w +1 -wh])*0.1;
                    norm[2] += (m_Data[index +w -1 +wh]-m_Data[index +w -1 -wh])*0.1;
                    norm[2] += (m_Data[index -w +1 +wh]-m_Data[index -w +1 -wh])*0.1;
                    norm[2] += (m_Data[index -w -1 +wh]-m_Data[index -w -1 -wh])*0.1;
		    }
		}

                len = (float)sqrt((double)((norm[0]*norm[0])+
                                           (norm[1]*norm[1])+
                                           (norm[2]*norm[2])));

                if (len > 0.0f)
                {
                    norm[0] /= len;
                    norm[1] /= len;
                    norm[2] /= len;
                }

                m_Normals[index*3+XDIR] = norm[0];
                m_Normals[index*3+YDIR] = norm[1];
                m_Normals[index*3+ZDIR] = norm[2];

                ++index;
            }
        }
    }
}

//-----------------------------------------------------------------------
//-- Initializes the Fast Classification                              ---
//-----------------------------------------------------------------------
void vu111211A::initFastClassification() 
{
    removeRunlengthEncoding();

    //Set up the Arrays for saving the sheared positions of the planes
    dataX = new RLEplane[m_Dim1Size];
    for (unsigned int i = 0; i < m_Dim1Size; ++i) dataX[i].size = 0;
    dataY = new RLEplane[m_Dim2Size];
    for (unsigned int i = 0; i < m_Dim2Size; ++i) dataY[i].size = 0;
    dataZ = new RLEplane[m_Dim3Size];
    for (unsigned int i = 0; i < m_Dim3Size; ++i) dataZ[i].size = 0;

    //init the FastClassification object
    fastClassificationObj = new FastClassification(m_Data,m_Dim1Size,m_Dim2Size,m_Dim3Size);
}

//-----------------------------------------------------------------------
//-- Determines the main viewing direction from the viewing vector    ---
//-----------------------------------------------------------------------
void vu111211A::computeMainViewingDirection() {
    mainViewingDir = XDIR;

    double vXsq = m_View[XDIR]*m_View[XDIR];
    double vYsq = m_View[YDIR]*m_View[YDIR];
    double vZsq = m_View[ZDIR]*m_View[ZDIR];

    if ((vXsq >= vYsq) && (vXsq >= vZsq)) {        // main viewing direction: X
	mainViewingDir = XDIR;
    } else if ((vYsq >= vXsq) && (vYsq >=vZsq)) {  // main viewing direction: Y
	mainViewingDir = YDIR;
    } else {                                       // main viewing direction Z
	mainViewingDir = ZDIR;
    }
}

//-----------------------------------------------------------------------
//-- Determines the permutation matrix                                ---
//-----------------------------------------------------------------------
void vu111211A::computePermutationMatrix() 
{
  // depending on the main viewing direction a permutation matrix is choosen
  switch(mainViewingDir) {
      case XDIR: {
	  permMatrix[0][0] = 0; permMatrix[0][1] = 1; permMatrix[0][2] = 0; permMatrix[0][3] = 0;
	  permMatrix[1][0] = 0; permMatrix[1][1] = 0; permMatrix[1][2] = 1; permMatrix[1][3] = 0;
	  permMatrix[2][0] = 1; permMatrix[2][1] = 0; permMatrix[2][2] = 0; permMatrix[2][3] = 0;
	  permMatrix[3][0] = 0; permMatrix[3][1] = 0; permMatrix[3][2] = 0; permMatrix[3][3] = 1;
	  break;
      }
      case YDIR: {
	  permMatrix[0][0] = 0; permMatrix[0][1] = 0; permMatrix[0][2] = 1; permMatrix[0][3] = 0;
	  permMatrix[1][0] = 1; permMatrix[1][1] = 0; permMatrix[1][2] = 0; permMatrix[1][3] = 0;
	  permMatrix[2][0] = 0; permMatrix[2][1] = 1; permMatrix[2][2] = 0; permMatrix[2][3] = 0;
	  permMatrix[3][0] = 0; permMatrix[3][1] = 0; permMatrix[3][2] = 0; permMatrix[3][3] = 1;
	  break;
      }
      case ZDIR: {
	  permMatrix[0][0] = 1; permMatrix[0][1] = 0; permMatrix[0][2] = 0; permMatrix[0][3] = 0;
	  permMatrix[1][0] = 0; permMatrix[1][1] = 1; permMatrix[1][2] = 0; permMatrix[1][3] = 0;
	  permMatrix[2][0] = 0; permMatrix[2][1] = 0; permMatrix[2][2] = 1; permMatrix[2][3] = 0;
	  permMatrix[3][0] = 0; permMatrix[3][1] = 0; permMatrix[3][2] = 0; permMatrix[3][3] = 1;
	  break;
      }
  }
}


//-----------------------------------------------------------------------
//-- Computes the viewing matrix according to the given view- and     ---
//-- up-vectors.                                                      ---
//-----------------------------------------------------------------------
void vu111211A::computeViewMatrix(void)
{
  vuVector puffer;
  vuVector cross;
  vuVector up = m_Up;
  vuVector v = m_View;

  double to_degree = 180/M_PI;  // for dealing with angles in degree
  double alpha = 0;
  double beta = 0;

  vuMatrix rotate_view;  // a rotation matrix to bring the 
                         // view-vector in correct position 

  vuMatrix rotate_up;    // a rotation matrix to bring the 
                         // up-vector in correct position

  // init the matrices with the identity
  viewMatrix.makeIdentity();
  rotate_view.makeIdentity();
  rotate_up.makeIdentity();

  // unitVecMain ist the unit vector pointing in the direction of the 
  // main viewing direction which is Z
  vuVector unitVecMain;
  unitVecMain[0] = 0;
  unitVecMain[1] = 0;
  unitVecMain[2] = 1;
  unitVecMain[3] = 1;

  // compute the standard object coordinates for view- and up-vector
  puffer = v;
  v = puffer * permMatrix;
  puffer = up;
  up = puffer * permMatrix;

  // compute the direction-vector of the axis for the following
  // rotation (= v x unitVecMain)
  cross = v.cross(unitVecMain);
  cross.makeUnit();

  // compute angle between view-vector and unit-vector of main viewing
  // direction and set step2-matrix to the according rotation around cross
  v.makeUnit();
  alpha = -to_degree * acos(v[ZDIR]);
  rotate_view.makeRotate(cross,alpha);

  // --> now step2 represents the matrix that turns the viewing vector 
  //     into exactly the same direction as the main viewing vector

  // compute the value of the up vector when it is transformed by steps 1 and 2
  puffer = up;
  up = puffer * rotate_view;

  // compute angle between up-vector (without main-component)
  // and set step2-matrix to the according rotation around the main viewing axis
  up[ZDIR] = 0;
  up.makeUnit();

  // compute the angle for rotation to bring the up vector in correct position
  beta = to_degree * acos(up[YDIR]);
  if (up[XDIR] >= 0) beta *= (-1);
  rotate_up.makeRotateZ(beta);

  // compute final view-matrix and return the value
  viewMatrix = rotate_view * rotate_up;      // in [Lacroute] this is M'
  worldMatrix = permMatrix * viewMatrix;

  if (viewingMode == VIEWING_MODE_PERSPECTIVE) {
      projMatrix[0][0] = eye_distance; projMatrix[0][1] = 0; projMatrix[0][2] = 0; projMatrix[0][3] = 0;
      projMatrix[1][0] = 0; projMatrix[1][1] = eye_distance; projMatrix[1][2] = 0; projMatrix[1][3] = 0;
      projMatrix[2][0] = 0; projMatrix[2][1] = 0; projMatrix[2][2] = 1; projMatrix[2][3] = 0;
      projMatrix[3][0] = 0; projMatrix[3][1] = 0; projMatrix[3][2] = 1; projMatrix[3][3] = eye_distance;
      viewMatrix = worldMatrix * projMatrix;
  }
}

//-----------------------------------------------------------------------
//-- Determines width, height and depth of the Volume and saves these ---
//-- values in volumeWidth, volumeHeight and volumeDepth.             ---
//-----------------------------------------------------------------------
void vu111211A::computeRunlengthSizes()
{
    switch(mainViewingDir) {
	case XDIR: {
	    volumeDepth = m_Dim1Size;
	    volumeWidth = m_Dim2Size;
	    volumeHeight = m_Dim3Size;
	    break;
	}
	case YDIR: {
	    volumeDepth = m_Dim2Size;  
	    volumeWidth = m_Dim3Size;  
	    volumeHeight = m_Dim1Size; 
	    break;
	}
	case ZDIR: {
	    volumeDepth = m_Dim3Size;
	    volumeWidth = m_Dim1Size;
	    volumeHeight = m_Dim2Size;
	    break;
	}
    }
}

//-----------------------------------------------------------------------
//-- Computes the Shear Factors si, sj and the translation ti, tj     ---
//-- for orthogonal projection.                                       ---
//-----------------------------------------------------------------------
void vu111211A::computeShearAndWarpFactorsOrtho()
{
    if ((viewMatrix[0][0]*viewMatrix[1][1]-viewMatrix[1][0]*viewMatrix[0][1]) == 0) si = 1; 
    else si = ((viewMatrix[1][1]*viewMatrix[0][2]-viewMatrix[0][1]*viewMatrix[1][2])/
	       (viewMatrix[0][0]*viewMatrix[1][1]-viewMatrix[1][0]*viewMatrix[0][1]));
    if ((viewMatrix[0][0]*viewMatrix[1][1]-viewMatrix[1][0]*viewMatrix[0][1]) == 0) sj = 1;
    else sj = ((viewMatrix[0][0]*viewMatrix[1][2]-viewMatrix[1][0]*viewMatrix[0][2])/
	       (viewMatrix[0][0]*viewMatrix[1][1]-viewMatrix[1][0]*viewMatrix[0][1]));

    if (si >= 0) ti = 0.0; else ti = (-si*volumeDepth);
    if (sj >= 0) tj = 0.0; else tj = (-sj*volumeDepth);
}

//-----------------------------------------------------------------------
//-- Shear the planes of the volume for orthogonal projection         ---
//-----------------------------------------------------------------------
void vu111211A::shearOrtho()
{
    // normalize the viewing vector
    m_View.makeUnit();

    double shearPos_i = 0;
    double shearPos_j = 0;

    if (orthoWarpOpenGL == 1) {
	shearPos_i = ti;
	shearPos_j = tj;
    } else {
	shearPos_i = maxSize - ((volumeWidth  + si * volumeDepth) / 2);
	shearPos_j = maxSize - ((volumeHeight + sj * volumeDepth) / 2);
    }

    // select the used dataset...
    switch(mainViewingDir) {
	case ZDIR: {
	    curr = dataZ;
	    break;
	}
	case YDIR: {
	    curr = dataY;
	    break;
	}
	case XDIR: {
	    curr = dataX;
	    break;
	}
    }

    if (m_View[mainViewingDir] >= 0) direction = 0; else direction = 1;

    for (dword i = 0; i < volumeDepth; ++i) {
	curr[i].dim1_pos = shearPos_i;
	curr[i].dim2_pos = shearPos_j;
	shearPos_i += si;
	shearPos_j += sj;
	if (shearPos_i < 0) shearPos_i = 0;
	if (shearPos_j < 0) shearPos_j = 0;
    }
}

//-----------------------------------------------------------------------
//-- Computes the length of the view vector normalized in Z           ---
//-----------------------------------------------------------------------
float vu111211A::computeZNormalizedViewingVectorLength() {
    //precompute length of z-normalized viewing vector (later used for opacity correction)
    double z,x,y;
    m_View.makeUnit();
    if      (mainViewingDir == XDIR) { z = m_View[XDIR]; y = m_View[YDIR]; x = m_View[ZDIR]; }
    else if (mainViewingDir == YDIR) { z = m_View[YDIR]; y = m_View[ZDIR]; x = m_View[XDIR]; }
    else                          { z = m_View[ZDIR]; y = m_View[XDIR]; x = m_View[YDIR]; }
    return sqrt((x/z)*(x/z)+(y/z)*(y/z)+1);
}

//-----------------------------------------------------------------------
//-- Reset the intermediate image                                     ---
//-----------------------------------------------------------------------
void vu111211A::resetIntermediateImage(int intermediateSize) {
    //reset intermediate image
    for (int i = 0; i < intermediateSize; ++i) {
	intermediate[i].trans = 1;
	intermediate[i].red = 0;
	intermediate[i].green = 0;
	intermediate[i].blue = 0;
	intermediate[i].offset = 0;
    }
}

//-----------------------------------------------------------------------
//-- Pre-classify the voxels                                          ---
//-----------------------------------------------------------------------
void vu111211A::preClassification(RLEplane current, float viewVectorLength) {

    // do the shading and the opacity correction
    for (dword k = 0; k < current.size; ++k) {
	current.voxel[k].normal.makeUnit();

	float diffuse = m_View.dot(current.voxel[k].normal);

	float spec = 0;

	if (diffuse <= 0) { // <= 0
	    current.voxel[k].red_shaded = 0.0;
	    current.voxel[k].green_shaded = 0.0;
	    current.voxel[k].blue_shaded = 0.0;
	    current.voxel[k].opacity_corr = 0.0;
	} else {
	    current.voxel[k].opacity_corr = 
		1 - pow((1-current.voxel[k].opacity),viewVectorLength);

	    if (specular == 1) {
	        spec = pow(diffuse,100);
	    } else {
	        specular = 0;
	    }

	    float fac = current.voxel[k].opacity_corr * (0.1 + 0.9 * diffuse);

	    current.voxel[k].red_shaded   = current.voxel[k].red   * fac;
	    current.voxel[k].green_shaded = current.voxel[k].green * fac;
	    current.voxel[k].blue_shaded  = current.voxel[k].blue  * fac;

	    current.voxel[k].red_shaded   += spec;
	    current.voxel[k].green_shaded += spec;
	    current.voxel[k].blue_shaded  += spec;
	    current.voxel[k].opacity_corr += spec;

	    if (current.voxel[k].red_shaded   > 1.0) current.voxel[k].red_shaded   = 1.0;
	    if (current.voxel[k].green_shaded > 1.0) current.voxel[k].green_shaded = 1.0;
	    if (current.voxel[k].blue_shaded  > 1.0) current.voxel[k].blue_shaded  = 1.0;
	    if (current.voxel[k].opacity_corr > 1.0) current.voxel[k].opacity_corr = 1.0;
	}
    }
}

//-----------------------------------------------------------------------
//-- Init the start-position of the first pointer                     ---
//-----------------------------------------------------------------------
void vu111211A::initRunPtr1(RLEplane current, int &volumeIndex, 
			    int &voxelPtr1, int &runPtr1, byte &state1) {
    do {
        volumeIndex += current.runlength[runPtr1]; 
	if (state1 == 0) {
	    state1 = 1;
	} else {
	    state1 = 0;
	    voxelPtr1 += current.runlength[runPtr1];
	}
	++runPtr1;
    } while ((dword)volumeIndex < volumeWidth);
}

//-----------------------------------------------------------------------
//-- Skip runs with zero length                                       ---
//-----------------------------------------------------------------------
void vu111211A::skipZeroLengthRuns(RLEplane slice, int &runPtr, 
				   byte &count, byte &state) {
    while (slice.runlength[runPtr] == 0) {
	++runPtr;
	count = 0;
	if (state == 0) state = 1; else state = 0;
    }
}

//-----------------------------------------------------------------------
//-- Skip transparent voxels                                          ---
//-----------------------------------------------------------------------
int vu111211A::skipTransparentVoxels(RLEplane slice, 
				     int runPtr1, byte &count1, 
				     int runPtr2, byte &count2, 
				     int &add, 
                                     int intermediateWidth, int &pos) {

    int skip;

    if (slice.runlength[runPtr1]-count1 < slice.runlength[runPtr2]-count2) {
	skip = slice.runlength[runPtr1]-count1-1;
    } else {
        skip = slice.runlength[runPtr2]-count2-1;
    }

    add += skip;
    if ((dword)add >= volumeWidth) {
	pos += (intermediateWidth - volumeWidth);
	add = add-volumeWidth;
	pos += skip-add;
    } else {
	pos += skip;
    }

    count1 += skip;
    count2 += skip;

    return skip;
}

//-----------------------------------------------------------------------
//-- Interpolate a Pixel                                              ---
//-----------------------------------------------------------------------
void vu111211A::interpolatePixel(RLEplane slice,
				 intermediatePixel &pixel,
				 float w1, float w2, float w3, float w4, int volumeIndex,
				 int &voxelPtr1, int runPtr1, int count1, int state1,
				 int &voxelPtr2, int runPtr2, int count2, int state2) {

    double red     = 0;
    double green   = 0;
    double blue    = 0;
    double opacity = 0;

    if ((dword)voxelPtr1 < slice.size) {

        //does state1 point to a non-transparent voxel?
	if (state1 == 1) {

	    red     += w1*slice.voxel[voxelPtr1].red_shaded;
	    green   += w1*slice.voxel[voxelPtr1].green_shaded;
	    blue    += w1*slice.voxel[voxelPtr1].blue_shaded;
	    opacity += w1*slice.voxel[voxelPtr1].opacity_corr;

	    if ((slice.runlength[runPtr1]-count1 > 1) &&
		((dword)voxelPtr1+1 < slice.size)) {

		red     += w2*slice.voxel[voxelPtr1+1].red_shaded;
		green   += w2*slice.voxel[voxelPtr1+1].green_shaded;
		blue    += w2*slice.voxel[voxelPtr1+1].blue_shaded;
		opacity += w2*slice.voxel[voxelPtr1+1].opacity_corr;

	    }
	    ++voxelPtr1;
	} else {
	    if ((slice.runlength[runPtr1]-count1 == 1) &&   //non-transp. run ends after ptr1?
		(slice.runlength[runPtr1+1]!=0) &&          //no zero-length run after current?
		((volumeIndex+1) % volumeWidth)!=0) {   //no line break?

		red     += w2*slice.voxel[voxelPtr1].red_shaded;  // vorher voxelPtr1+1
		green   += w2*slice.voxel[voxelPtr1].green_shaded;
		blue    += w2*slice.voxel[voxelPtr1].blue_shaded;
		opacity += w2*slice.voxel[voxelPtr1].opacity_corr;
	    }
	}
    }

    if ((dword)voxelPtr2 < slice.size) {
	if (state2 == 1) {

	    red     += w3*slice.voxel[voxelPtr2].red_shaded;
	    green   += w3*slice.voxel[voxelPtr2].green_shaded;
	    blue    += w3*slice.voxel[voxelPtr2].blue_shaded;
	    opacity += w3*slice.voxel[voxelPtr2].opacity_corr;

	    if ((slice.runlength[runPtr2]-count2 > 1) && 
		((dword)voxelPtr2+1 < slice.size)) {

		red     += w4*slice.voxel[voxelPtr2+1].red_shaded;
		green   += w4*slice.voxel[voxelPtr2+1].green_shaded;
		blue    += w4*slice.voxel[voxelPtr2+1].blue_shaded;
		opacity += w4*slice.voxel[voxelPtr2+1].opacity_corr;

	    }
	    ++voxelPtr2;
	} else {
	    if ((slice.runlength[runPtr2]-count2 == 1) &&  //non-transp. run ends after ptr1?
		(slice.runlength[runPtr2+1]!=0) &&         //no zero-length run after current?
		((volumeIndex+1) % volumeWidth)!=0) {  //no line break?

		red     += w4*slice.voxel[voxelPtr2].red_shaded;  //vorher voxelPtr2+1
		green   += w4*slice.voxel[voxelPtr2].green_shaded;
		blue    += w4*slice.voxel[voxelPtr2].blue_shaded;
		opacity += w4*slice.voxel[voxelPtr2].opacity_corr;

	    }
	}
    }

    if (opacity > 0) {

	pixel.offset = 1;

	double fac = pixel.trans;

	pixel.red   += red * fac;
	pixel.green += green * fac;
	pixel.blue  += blue * fac;

	pixel.trans *= (1-opacity);
    }
}

//-----------------------------------------------------------------------
//-- Skip opaque pixel                                                ---
//-----------------------------------------------------------------------
void vu111211A::skipOpaquePixel(byte state1,int &voxelPtr1,byte state2,int &voxelPtr2) {

    if (state1 == 1) {
	++voxelPtr1;
    }

    if (state2 == 1) {
	++voxelPtr2;
    }
}

//-----------------------------------------------------------------------
//-- Goto next voxel                                                  ---
//-----------------------------------------------------------------------
void vu111211A::volumeNext(RLEplane slice, int &volumeIndex, int &add, 
			   int intermediateWidth, int &pos,
			   int &runPtr1, byte &count1, byte &state1,
                           int &runPtr2, byte &count2, byte &state2) {

    //one step foreward through the volume
    ++volumeIndex;

    ++add;
    if ((dword)add >= volumeWidth) {
	pos += (intermediateWidth - volumeWidth);
	add = add-volumeWidth;
	++pos;
    } else {
	++pos;
    }

    ++count1;
    ++count2;

    if (count1 >= slice.runlength[runPtr1]) {
	++runPtr1;
	count1 = 0;
	if (state1 == 1) state1 = 0; else state1 = 1;
    }

    if (count2 >= slice.runlength[runPtr2]) {
	++runPtr2;
	count2 = 0;
	if (state2 == 1) state2 = 0; else state2 = 1;
    }
}

//-----------------------------------------------------------------------
//-- compute the inverse warp matrix for orthogonal viewing           ---
//-----------------------------------------------------------------------
void vu111211A::computeInvWarpOrtho() 
{
    // the 2D-warp-Matrix
    float m00 = viewMatrix[0][0];
    float m01 = viewMatrix[0][1];
    float m10 = viewMatrix[1][0];
    float m11 = viewMatrix[1][1];

    // cmompute the inverse of the 2D-warp-Matrix
    float invDet = 1 / (m00 * m11 - (m01 * m10));

    invWarpOrtho00 = invDet * m11;
    invWarpOrtho01 = - invDet * m01;
    invWarpOrtho10 = - invDet * m10;
    invWarpOrtho11 = invDet * m00;
}

//-----------------------------------------------------------------------
//-- inverse orthogonal warping of a single point                     ---
//-----------------------------------------------------------------------
void vu111211A::warpOrthoInv(float x, float y, float &x_result, float &y_result)
{
    // apply the inverse warp to the given coordinates
    x_result = invWarpOrtho00 * x + invWarpOrtho01 * y;
    y_result = invWarpOrtho10 * x + invWarpOrtho11 * y;
}

//-----------------------------------------------------------------------
//-- orthogonal warping                                               ---
//-----------------------------------------------------------------------
void vu111211A::warpOrtho()
{
    int glSize = glImageWidth * glImageHeight;
    int glHalf = glImageWidth / 2;

    int intermediateWidth = maxSize * 2;

    float int_x = 0.0f;
    float int_y = 0.0f;

    float gl_x = 0.0f;
    float gl_y = 0.0f;

    float maxSizef = (float)maxSize-1;

    int int_index = 0;
    int gl_index  = 0;
    while(gl_index < glSize) {

	gl_x = (gl_index % glImageWidth) - glHalf;
	gl_y = (gl_index / glImageWidth) - glHalf;

	warpOrthoInv(gl_x, gl_y, int_x, int_y);

	float d1 = (int_x - floor(int_x));
	float d2 = (int_y - floor(int_y));

	float w1 = 1-d1-d2+d1*d2;
	float w2 = d1-d1*d2;
	float w3 = d2-d1*d2;
	float w4 = d1*d2;

	int_index = (int)(floor(int_x + maxSizef) + 
			  floor(int_y + maxSizef) * intermediateWidth);


	if ((int_x > -maxSizef) && (int_x < maxSizef - 1) &&
	    (int_y > -maxSizef) && (int_y < maxSizef - 1)) {


  	    float red = intermediate[int_index].red*w1 + 
 	            intermediate[int_index+1].red*w2 +
 	            intermediate[int_index+intermediateWidth].red*w3 +
	            intermediate[int_index+intermediateWidth+1].red*w4;
	    float green = intermediate[int_index].green*w1 +
 	            intermediate[int_index+1].green*w2 +
 	            intermediate[int_index+intermediateWidth].green*w3 +
  	            intermediate[int_index+intermediateWidth+1].green*w4;
	    float blue = intermediate[int_index].blue*w1 + 
 	            intermediate[int_index+1].blue*w2 +
 	            intermediate[int_index+intermediateWidth].blue*w3 +
 	            intermediate[int_index+intermediateWidth+1].blue*w4;


	    glImage[gl_index*4+0] = (GLubyte)(red*255);
	    glImage[gl_index*4+1] = (GLubyte)(green*255);
	    glImage[gl_index*4+2] = (GLubyte)(blue*255);
	    glImage[gl_index*4+3] = 255;
	} else {
	    glImage[gl_index*4+0] = 0;
	    glImage[gl_index*4+1] = 0;
	    glImage[gl_index*4+2] = 0;
	    glImage[gl_index*4+3] = 255;
	}

	gl_index++;
    }
}


//-----------------------------------------------------------------------
//-- Produces the intermediate image for orthogonal projection        ---
//-----------------------------------------------------------------------
void vu111211A::makeIntermediateImageOrtho()
{
    int from = 0;
    int to = 0;

    int pos = 0;
    int add = 0;

    int intermediateWidth = maxSize*2;
    int intermediateSize = intermediateWidth*intermediateWidth;

    int step = 0;

    double l = computeZNormalizedViewingVectorLength();

    resetIntermediateImage(intermediateSize);

    if (direction == 0) { to = volumeDepth-1; step = +1; }
    else { from = volumeDepth-1; step = -1; }

    int runIndex = 0;
    byte state1 = 0;
    byte state2 = 0;
    byte count1 = 0;
    byte count2 = 0;
    int voxelIndex = 0;
    int volumeIndex = 0;

    //for every plane in the runlength encoded volume...
    for (int i = from; i != to + step; i += step) {

	runIndex = 0;
	state1 = 0;
	state2 = 0;
	count1 = 0;
	count2 = 0;
	voxelIndex = 0;
	volumeIndex = 0;

        //index of the voxels of each of the two voxel scanlines
        int voxelPtr1 = 0;
	int voxelPtr2 = 0;

        //index of the current runs
	int runPtr1 = 0;
	int runPtr2 = 0;

        //compute the rounded total shearing (rounding dim1_pos and dim2_pos)
	int shearX = (int)(curr[i].dim1_pos);
	int shearY = (int)(curr[i].dim2_pos);

        //compute the bilinear interpolation-coefficients
	float s1 = 1 - (curr[i].dim1_pos - shearX);
	float s2 = 1 - (curr[i].dim2_pos - shearY);

	float w1 = s2-s1*s2;
	float w2 = s1*s2;
	float w3 = 1-s1-s2+s1*s2;
	float w4 = s1-s1*s2;

        vuVector normal;

        //if current slice is not empty
	if (curr[i].size > 0) {

            // pre-classify the current slice
	    preClassification(curr[i],l);

            // voxel-pointer 2 is already in right position; 
            // find value for voxel-pointer 1
	    initRunPtr1(curr[i], volumeIndex, voxelPtr1, runPtr1, state1);

	    volumeIndex = 0;

            // position of the pixel that is affected by the
            // four surrounding voxels
	    pos = intermediateWidth * shearY + shearX;
	    add = 0;

            //trace through the slice
	    while((dword)voxelPtr2 < curr[i].size) {

                //if neccessary, skip zero-length runs
		skipZeroLengthRuns(curr[i], runPtr1, count1, state1);
		skipZeroLengthRuns(curr[i], runPtr2, count2, state2);

                //if inside a transparent run, skip voxels
		if ((state1 == 0) && (curr[i].runlength[runPtr1]-count1 > 1) && 
		    (state2 == 0) && (curr[i].runlength[runPtr2]-count2 > 1)) {

		    volumeIndex += skipTransparentVoxels(curr[i], 
                                                         runPtr1, 
                                                         count1, 
                                                         runPtr2, 
                                                         count2, 
                                                         add, 
                                                         intermediateWidth,
							 pos);

		} else {

                    // only check this pixel, if it still is transparent
		    if (intermediate[pos].trans > MIN_TRANSPARENCY_MAKING_INTERMEDIATE) {

			interpolatePixel(curr[i], intermediate[pos],
                                         w1, w2, w3, w4, volumeIndex,
                                         voxelPtr1, runPtr1, count1, state1,
                                         voxelPtr2, runPtr2, count2, state2);

		    } else {

			skipOpaquePixel(state1,voxelPtr1,state2,voxelPtr2);

		    }

                    //one step foreward through the volume
		    volumeNext(curr[i], volumeIndex, add, intermediateWidth, pos, 
			       runPtr1, count1, state1, 
			       runPtr2, count2, state2);

		}
	    }
	}
    }
}

//-----------------------------------------------------------------------
//-- Convert format of intermediate image to OpenGL texture format    ---
//-----------------------------------------------------------------------
void vu111211A::changeFormatIntermediate() {
    int w = (int)(si*volumeDepth);
    if (w < 0) w *= -1;
    w += volumeWidth;

    int h = (int)(sj*volumeDepth);
    if (h < 0) h *= -1;
    h += volumeHeight;

    int diffX = (int)((glImageWidth-w)/2.0);
    int diffY = (int)((glImageHeight-h)/2.0);

    for (word i = 0; i < glImageWidth; ++i) {
	for (word j = 0; j < glImageHeight; ++j) {
	    dword pos_gl = (i + j*glImageWidth)*4;
	    if ((i-diffX >= 0) && (j-diffY >= 0) && (i-diffX < w) && (j-diffY < h)) {

		dword pos_int = (i-diffX) + (j-diffY)*maxSize*2;

		glImage[pos_gl+0] = (GLubyte)(intermediate[pos_int].red*255);
		glImage[pos_gl+1] = (GLubyte)(intermediate[pos_int].green*255);
		glImage[pos_gl+2] = (GLubyte)(intermediate[pos_int].blue*255);
		glImage[pos_gl+3] = 255;

	    } else {
		glImage[pos_gl+0] = (GLubyte)0;
		glImage[pos_gl+1] = (GLubyte)0;
		glImage[pos_gl+2] = (GLubyte)0;
		glImage[pos_gl+3] = 255;
	    }
	}
    }
}

//-----------------------------------------------------------------------
//-- Warp and draw image (warp is performed by OpenGL)                ---
//-----------------------------------------------------------------------
void vu111211A::drawWarpOpenGL() {
    //the positions of the edges of the OpenGL-polygon
    float pos[4][2];

    //the 2D warping matrix
    float m00 = viewMatrix[0][0];
    float m01 = viewMatrix[0][1];
    float m10 = viewMatrix[1][0];
    float m11 = viewMatrix[1][1];

    //compute the warped positions of the edges of the OpenGL-polygon
    pos[0][XDIR] = 0 * m00 + 0 * m01;
    pos[0][YDIR] = 0 * m10 + 0 * m11;

    pos[1][XDIR] = 1 * m00 + 0 * m01;
    pos[1][YDIR] = 1 * m10 + 0 * m11;

    pos[2][XDIR] = 1 * m00 + 1 * m01;
    pos[2][YDIR] = 1 * m10 + 1 * m11;

    pos[3][XDIR] = 0 * m00 + 1 * m01;
    pos[3][YDIR] = 0 * m10 + 1 * m11;

    //compute the translation of the OpenGL-plane to place 
    //it in the center of the screen
    float t[2];
    t[XDIR] = 0.5 * m00 + 0.5 * m01;
    t[YDIR] = 0.5 * m10 + 0.5 * m11;

    //clear screen, enable texturing...
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    glEnable(GL_TEXTURE_2D);
    glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

    //set parameters for texture mapping
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);

    //create new texture
    glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, glImageWidth,
		 glImageHeight, 0, GL_RGBA, GL_UNSIGNED_BYTE, glImage);

    //work with projection matrix
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();

    // the size of the OpenGL plane with the texture upon it
    float planeSize = 40*zoomValue*zoomValue;  

    glOrtho(-10,10,-10,10,-1000,1000);
	
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();

    glBegin(GL_QUADS);

    glTexCoord2f(0, 0); 
    glVertex3f(planeSize*(pos[0][XDIR]-t[XDIR]), planeSize*(pos[0][YDIR]-t[YDIR]), 0.0);
    glTexCoord2f(1, 0); 
    glVertex3f(planeSize*(pos[1][XDIR]-t[XDIR]), planeSize*(pos[1][YDIR]-t[YDIR]), 0.0);
    glTexCoord2f(1, 1); 
    glVertex3f(planeSize*(pos[2][XDIR]-t[XDIR]), planeSize*(pos[2][YDIR]-t[YDIR]), 0.0);
    glTexCoord2f(0, 1); 
    glVertex3f(planeSize*(pos[3][XDIR]-t[XDIR]), planeSize*(pos[3][YDIR]-t[YDIR]), 0.0);

    glEnd();

    glFlush();

    glDisable(GL_TEXTURE_2D);
}

//-----------------------------------------------------------------------
//-- get the value of the voxel at the given coordinates              ---
//-----------------------------------------------------------------------
byte vu111211A::getSample(dword x, dword y, dword z)
{
    dword index = x + y * m_Dim1Size + z * m_Dim1Size * m_Dim2Size;
    byte result = 0;
    if (index < m_Dim1Size*m_Dim2Size*m_Dim3Size) {
	result = m_Data[index];
    } 
    return result;
}

//-----------------------------------------------------------------------
//-- Produces the intermediate image for orthogonal projection using  ---
//-- Fast Classification                                              ---
//-----------------------------------------------------------------------
void vu111211A::makeIntermediateImageOrthoFastClass()
{
    dword from = 0;
    dword to = 0;

    dword pos = 0;

    dword row = 0;
    dword col = 0;

    dword intermediateWidth = maxSize*2;
    dword intermediateSize = intermediateWidth*intermediateWidth;

    int step = 0;

    double l = computeZNormalizedViewingVectorLength();

    //initialize intermediate image
    for (dword i = 0; i < intermediateSize; ++i) {
	intermediate[i].trans = 1;
	intermediate[i].red = 0;
	intermediate[i].green = 0;
	intermediate[i].blue = 0;
	intermediate[i].offset = 0;
    }

    //Select start-, end-position and step-size
    if (direction == 0) { to = volumeDepth-1; step = +1; }
    else { from = volumeDepth-1; step = -1; }

    //for every plane in the runlength encoded volume...
    for (dword i = from; i != to + step; i += step) {

        //precompute the rounded total shearing (rounding dim1_pos and dim2_pos)
	dword shearX = (dword)(curr[i].dim1_pos);
	dword shearY = (dword)(curr[i].dim2_pos);

        //precompute the bilinear interpolation-coefficients
	double s1 = 1 - (curr[i].dim1_pos - shearX);
	double s2 = 1 - (curr[i].dim2_pos - shearY);

	double w[4];
	w[0] = s2-s1*s2;
	w[1] = s1*s2;
	w[2] = 1-s1-s2+s1*s2;
	w[3] = s1-s1*s2;

        //position in intermediate image
	pos = intermediateWidth * shearY + shearX;

	switch(mainViewingDir) {
	    case XDIR: {
		row = m_Dim2Size;
		col = m_Dim3Size;
		break;
	    }
	    case YDIR: {
		row = m_Dim3Size;
		col = m_Dim1Size;
		break;
	    }
	    case ZDIR: {
		row = m_Dim1Size;
		col = m_Dim2Size;
		break;
	    }
	}

        //The coordinates of all 4 sample points
	dword xVol[4] = {0, 0, 0, 0};
	dword yVol[4] = {0, 0, 0, 0};
	dword zVol[4] = {0, 0, 0, 0};

	dword j = 0;

	while (j < col) {

	    dword k = 0;

	    while (k < row) {

		switch(mainViewingDir) {
		    case XDIR: {
			xVol[2] = i;
			yVol[2] = k;
			zVol[2] = j;

			xVol[3] = i;
			yVol[3] = k + 1;
			zVol[3] = j;

			xVol[0] = i;
			yVol[0] = k;
			zVol[0] = j + 1;

			xVol[1] = i;
			yVol[1] = k + 1;
			zVol[1] = j + 1;

			break;
		    }
		    case YDIR: {
			xVol[2] = j;
			yVol[2] = i;
			zVol[2] = k;

			xVol[3] = j;
			yVol[3] = i;
			zVol[3] = k + 1;

			xVol[0] = j + 1;
			yVol[0] = i;
			zVol[0] = k;

			xVol[1] = j + 1;
			yVol[1] = i;
			zVol[1] = k + 1;

			break;
		    }
		    case ZDIR: {
			xVol[2] = k;
			yVol[2] = j;
			zVol[2] = i;

			xVol[3] = k + 1;
			yVol[3] = j;
			zVol[3] = i;

			xVol[0] = k;
			yVol[0] = j + 1;
			zVol[0] = i;

			xVol[1] = k + 1;
			yVol[1] = j + 1;
			zVol[1] = i;

			break;
		    }
		}


		if (intermediate[pos].trans > MIN_TRANSPARENCY_MAKING_INTERMEDIATE) {

		    float red     = 0;
		    float green   = 0;
		    float blue    = 0;
		    float opacity = 0;

                    //the normal vector
		    vuVector normal;
		    normal[XDIR] = 0;
		    normal[YDIR] = 0;
		    normal[ZDIR] = 0;

		    for (int m = 0; m < 4; ++m) {
                        //get the value of the voxel
                        byte sample = getSample(xVol[m],yVol[m],zVol[m]);

                        //get the normal vector of the voxel
			vuVector n;
			dword indexNormal = (xVol[m] + yVol[m] * m_Dim1Size + 
					     zVol[m] * m_Dim1Size * m_Dim2Size) * 3;
			n[XDIR] = m_Normals[indexNormal+0];
			n[YDIR] = m_Normals[indexNormal+1];
			n[ZDIR] = m_Normals[indexNormal+2];
			n.makeUnit();

                        //weigh the normal vector
			normal[XDIR] += w[m] * n[XDIR];
			normal[YDIR] += w[m] * n[YDIR];
			normal[ZDIR] += w[m] * n[ZDIR];

			red     += w[m] * m_TFunc[sample][0];
			green   += w[m] * m_TFunc[sample][1];
			blue    += w[m] * m_TFunc[sample][2];
			opacity += w[m] * m_TFunc[sample][3];
		    }

		    opacity = 1 - pow((1-opacity),(float)l);

		    intermediate[pos].offset = 1;

                    //compute shading
		    normal.makeUnit();
		    float diffuse = normal.dot(m_View);

		    if (diffuse > 0) {

			double fac = intermediate[pos].trans * opacity * (0.1 + 0.9 * diffuse);

			intermediate[pos].red   += red   * fac;
			intermediate[pos].green += green * fac;
			intermediate[pos].blue  += blue  * fac;
			intermediate[pos].trans *= (1-opacity);

			if (specular == 1) {
			    float spec = pow(diffuse,150);
			    intermediate[pos].red     += spec * opacity;
			    intermediate[pos].green   += spec * opacity;
			    intermediate[pos].blue    += spec * opacity;
			    intermediate[pos].trans   -= spec * opacity;

			    if (intermediate[pos].red     > 1.0) intermediate[pos].red     = 1.0;
			    if (intermediate[pos].green   > 1.0) intermediate[pos].green   = 1.0;
			    if (intermediate[pos].blue    > 1.0) intermediate[pos].blue    = 1.0;
			    if (intermediate[pos].trans   < 0.0) intermediate[pos].trans   = 0.0;
			}
		    }
		}

		int skip1 = fastClassificationObj->skip(xVol[1],yVol[1],zVol[1],mainViewingDir);
		int skip2 = fastClassificationObj->skip(xVol[3],yVol[3],zVol[3],mainViewingDir);

		if (skip2 < skip1) skip1 = skip2;

		pos += skip1;
		k   += skip1;

	    }

	    pos += (intermediateWidth - volumeWidth);

	    ++j;
	}
    }

    //-- change format of intermediate Image ------------------

    int w = (int)(si*volumeDepth);
    if (w < 0) w *= -1;
    w += volumeWidth;

    int h = (int)(sj*volumeDepth);
    if (h < 0) h *= -1;
    h += volumeHeight;

    int diffX = (int)((glImageWidth-w)/2.0);
    int diffY = (int)((glImageHeight-h)/2.0);

    for (word i = 0; i < glImageWidth; ++i) {
	for (word j = 0; j < glImageHeight; ++j) {
	    dword pos_gl = (i + j*glImageWidth)*4;
	    if ((i-diffX >= 0) && (j-diffY >= 0) && (i-diffX < w) && (j-diffY < h)) {

		dword pos_int = (i-diffX) + (j-diffY)*maxSize*2;

		glImage[pos_gl+0] = (GLubyte)(intermediate[pos_int].red*255);
		glImage[pos_gl+1] = (GLubyte)(intermediate[pos_int].green*255);
		glImage[pos_gl+2] = (GLubyte)(intermediate[pos_int].blue*255);
		glImage[pos_gl+3] = 255;

	    } else {
		glImage[pos_gl+0] = (GLubyte)0;
		glImage[pos_gl+1] = (GLubyte)0;
		glImage[pos_gl+2] = (GLubyte)0;
		glImage[pos_gl+3] = 255;
	    }
	}
    }
}

int vu111211A::getMaxSize() {
    return maxSize;
}

float vu111211A::getMinEyeDistance() {
    return maxSize * 1.0f;
}

void vu111211A::setEyeDistance(float distance) {
    eye_distance = distance;
}

//-----------------------------------------------------------------------
//-- Computes the position of the eye point in object space           ---
//-----------------------------------------------------------------------
void vu111211A::computeEyePoint() 
{
    vuVector eye_w;
    eye_w[XDIR] = 0.0f;
    eye_w[YDIR] = 0.0f;
    eye_w[ZDIR] = - eye_distance;
    eye_w[WDIR] = 1.0f;

    vuMatrix invWorldMatrix = worldMatrix.inverse();

    eye_o = eye_w * invWorldMatrix * permMatrix;
}

//-----------------------------------------------------------------------
//-- Computes the Shear Factors si, sj and the Warp Factors ti, tj    ---
//-- for perspective projection.                                      ---
//-----------------------------------------------------------------------
void vu111211A::computeShearPerspective()
{
    //init the shearMatrix with the identity
    shearMatrix.makeIdentity();

    //change the three values in the matrix to get the final shear matrix
    shearMatrix[0][2] = - eye_o[XDIR] / eye_o[ZDIR];
    shearMatrix[1][2] = - eye_o[YDIR] / eye_o[ZDIR];
    shearMatrix[3][2] = - eye_o[WDIR] / eye_o[ZDIR];
}


//-----------------------------------------------------------------------
//-- Shear the planes of the volume for perspective projection        ---
//-----------------------------------------------------------------------
void vu111211A::computePlanePositionScale(float &pos_i, 
					  float &pos_j,
					  float &scale,
					  int plane)
{
    int halfX = volumeWidth  / 2;
    int halfY = volumeHeight / 2;

    vuVector lu = vuVector(-halfX, -halfY, (float)plane, 1.0f);  // left upper point of slice
    vuVector ru = vuVector( halfX, -halfY, (float)plane, 1.0f);   // right upper point of slice
    vuVector ll = vuVector(-halfX,  halfY, (float)plane, 1.0f);   // left lower point of slice
    vuVector rl = vuVector( halfX,  halfY, (float)plane, 1.0f);    // right lower point of slice

    vuVector lu_t = lu * shearMatrix;
    vuVector ru_t = ru * shearMatrix;
    vuVector ll_t = ll * shearMatrix;
    vuVector rl_t = rl * shearMatrix;

    // homogenous coordinates! -> divide by 4th coordinate
    for (int i = 0; i < 4; ++i) {
	lu_t[i] = lu_t[i] / lu_t[WDIR];
	ru_t[i] = ru_t[i] / ru_t[WDIR];
	ll_t[i] = ll_t[i] / ll_t[WDIR];
	rl_t[i] = rl_t[i] / rl_t[WDIR];
    }

    if (scale_total < 0) {
	scale_total = (lu_t[XDIR] - rl_t[XDIR]) / (lu[XDIR] - rl[XDIR]);
    }

    scale = ((lu_t[XDIR] - ru_t[XDIR]) / (lu[XDIR] - ru[XDIR])) / scale_total;

    pos_i = lu_t[XDIR] / scale_total + maxSize;
    pos_j = lu_t[YDIR] / scale_total + maxSize;
}

//-----------------------------------------------------------------------
//-- Shear the planes of the volume for perspective projection        ---
//-----------------------------------------------------------------------
void vu111211A::shearPerspective()
{
    // normalize the viewing vector
    m_View.makeUnit();

    // in which direction and from where shall the volume be traversed:
    int step = 0;
    int start = 0;
    int end = 0;

// !!!!
//    dword intermediateWidth = maxSize*2;
//    dword intermediateSize = intermediateWidth*intermediateWidth;

//    float shearPos_i = intermediateWidth / 2;
//    float shearPos_j = intermediateWidth / 2;

    // select the used dataset...
    switch(mainViewingDir) {
	case XDIR: {
	    curr = dataX;
	    break;
	}
	case YDIR: {
	    curr = dataY;
	    break;
	}
	case ZDIR: {
	    curr = dataZ;
	    break;
	}
    }

    if (m_View[mainViewingDir] >= 0) {
	direction = 0;
	step = 1;
	start = 0;
	end = volumeDepth;
    } else {
	direction = 1;
	step = -1;
	start = volumeDepth-1;
	end = -1;
    }

    int z = 0;

    if (direction == 0) {
	z = -(volumeDepth / 2);
    } else {
	z = volumeDepth / 2;
    }

    // needs to be assigned a negative value -> computePlanePositionScale
    // computes the total scale-value
    scale_total = -100.0f;

    for (int i = start; i != end; i += step) {
	computePlanePositionScale(curr[i].dim1_pos,curr[i].dim2_pos,curr[i].scale,z);
	if (direction == 0) z += 1; else z -= 1;
    }
}


//-----------------------------------------------------------------------
//-- Compute the warp- and the inverse warp-matrix for                ---
//-- perspective viewing (scaling of the most front plane is          ---
//-- included in the computation)                                     ---
//-----------------------------------------------------------------------
void vu111211A::computeWarpPerspective()
{
    vuMatrix scaleMatrix;
    scaleMatrix.makeIdentity();
    scaleMatrix.makeScale(scale_total,scale_total,1.0f);

    warpMatrix = scaleMatrix * shearMatrix.inverse() * permMatrix.inverse() * viewMatrix;

    invWarpMatrix = warpMatrix.inverse();
}

//-----------------------------------------------------------------------
//-- inverse perspective warping of a single point                    ---
//-----------------------------------------------------------------------
void vu111211A::warpPerspectiveInv(float x, 
				   float y, 
				   float &x_result, 
				   float &y_result)
{
    vuVector v;
    v[XDIR] = x;
    v[YDIR] = y;
    v[ZDIR] = 0.0f;
    v[WDIR] = 1.0f;

    // apply the inverse warp to the given coordinates
    vuVector result = v * invWarpMatrix;

    // the result must be divided by the 4th coordinate (homogenous coordinates!)
    x_result = result[XDIR] / result[WDIR];
    y_result = result[YDIR] / result[WDIR];
}

//-----------------------------------------------------------------------
//-- perspective warping                                              ---
//-----------------------------------------------------------------------
void vu111211A::warpPerspective()
{
    int glSize = glImageWidth * glImageHeight;
    int glHalf = glImageWidth / 2;

    int intermediateWidth = maxSize * 2;
//!!!!    int intermediateSize = intermediateWidth * intermediateWidth;

    float int_x = 0.0f;
    float int_y = 0.0f;

    float gl_x = 0.0f;
    float gl_y = 0.0f;

    float maxSizef = (float)maxSize;

    int int_index = 0;
    int gl_index  = 0;
    while(gl_index < glSize) {

	gl_x = (gl_index % glImageWidth) - glHalf;
	gl_y = (gl_index / glImageWidth) - glHalf;

	warpPerspectiveInv(gl_x, gl_y, int_x, int_y);

	float d1 = (int_x - floor(int_x));
	float d2 = (int_y - floor(int_y));

	float w1 = 1-d1-d2+d1*d2;
	float w2 = d1-d1*d2;
	float w3 = d2-d1*d2;
	float w4 = d1*d2;

	int_index = (int)(floor(int_x + maxSizef) + floor(int_y + maxSizef) * intermediateWidth);

	if ((int_x > -maxSizef) && (int_x < maxSizef-1) &&
	    (int_y > -maxSizef) && (int_y < maxSizef-1)) {

	    float red = intermediate[int_index].red*w1 + 
 	            intermediate[int_index+1].red*w2 +
 	            intermediate[int_index+intermediateWidth].red*w3 +
	            intermediate[int_index+intermediateWidth+1].red*w4;
	    float green = intermediate[int_index].green*w1 + 
 	            intermediate[int_index+1].green*w2 +
 	            intermediate[int_index+intermediateWidth].green*w3 +
  	            intermediate[int_index+intermediateWidth+1].green*w4;
	    float blue = intermediate[int_index].blue*w1 + 
 	            intermediate[int_index+1].blue*w2 +
 	            intermediate[int_index+intermediateWidth].blue*w3 +
 	            intermediate[int_index+intermediateWidth+1].blue*w4;

	    glImage[gl_index*4+0] = (GLubyte)(red*255);
	    glImage[gl_index*4+1] = (GLubyte)(green*255);
	    glImage[gl_index*4+2] = (GLubyte)(blue*255);
	    glImage[gl_index*4+3] = 255;
	} else {
	    glImage[gl_index*4+0] = 0;
	    glImage[gl_index*4+1] = 0;
	    glImage[gl_index*4+2] = 0;
	    glImage[gl_index*4+3] = 255;
	}

	gl_index++;
    }
}

//-----------------------------------------------------------------------
//-- Produces the intermediate image for perspective projection       ---
//-----------------------------------------------------------------------
void vu111211A::makeIntermediateImagePerspective()
{
    dword from = 0;
    dword to = 0;

    dword pos = 0;

    dword intermediateWidth = maxSize*2;
    dword intermediateSize = intermediateWidth*intermediateWidth;

    int step = 0;

    double l = computeZNormalizedViewingVectorLength();

    //initialize intermediate image
    for (dword i = 0; i < intermediateSize; ++i) {
	intermediate[i].trans = 1;
	intermediate[i].red = 0;
	intermediate[i].green = 0;
	intermediate[i].blue = 0;
	intermediate[i].offset = 0;
    }

    if (direction == 0) { to = volumeDepth-1; step = +1; }
    else { from = volumeDepth-1; step = -1; }

    dword runIndex = 0;
    byte state1 = 0;
    byte state2 = 0;
    byte count1 = 0;
    byte count2 = 0;
    dword voxelIndex = 0;
    dword indexRunlength = 0;

    //for every plane in the runlength encoded volume...
    for (dword i = from; i != to + step; i += step) {
//    for (dword i = 10; i == 10; i += step) {

	runIndex = 0;
	state1 = 0;
	state2 = 0;
	count1 = 0;
	count2 = 0;
	voxelIndex = 0;
	indexRunlength = 0;

        //index of the voxels of each of the two voxel scanlines
        dword voxelPtr1 = 0;
	dword voxelPtr2 = 0;

        //index of the current runs
	dword runPtr1 = 0;
	dword runPtr2 = 0;

        //precompute the rounded total shearing (rounding dim1_pos and dim2_pos)
// !!!!	dword shearX = (dword)(curr[i].dim1_pos);
//	dword shearY = (dword)(curr[i].dim2_pos);

        vuVector normal;

        //value of sampled point resample = w1*p1 + w2*p2 + w3*p3 + w4*p4
        // wi precomputed weights for every voxel;
        // pi points:
        //               p3    p4
        //                  sp
        //               p1    p2

	if (curr[i].size > 0) {

//---------------------------------------
            // do the shading and the opacity correction
	    for (dword k = 0; k < curr[i].size; ++k) {
		curr[i].voxel[k].normal.makeUnit();

		float diffuse = m_View.dot(curr[i].voxel[k].normal);

		float spec = 0;

		if (diffuse <= 0) { // <= 0
		    curr[i].voxel[k].red_shaded = 0.0;
		    curr[i].voxel[k].green_shaded = 0.0;
		    curr[i].voxel[k].blue_shaded = 0.0;
		    curr[i].voxel[k].opacity_corr = 0.0;
		} else {
		    curr[i].voxel[k].opacity_corr = 1 - pow((1-curr[i].voxel[k].opacity),(float)l);

		    float fac = curr[i].voxel[k].opacity_corr * (0.1 + 0.9 * diffuse);

		    curr[i].voxel[k].red_shaded = curr[i].voxel[k].red * fac;
		    curr[i].voxel[k].green_shaded = curr[i].voxel[k].green * fac;
		    curr[i].voxel[k].blue_shaded = curr[i].voxel[k].blue * fac;

                    // if neccessary, add specular light
		    if (specular == 1) {
			spec = pow(diffuse,150);

			curr[i].voxel[k].red_shaded   += spec;
			curr[i].voxel[k].green_shaded += spec;
			curr[i].voxel[k].blue_shaded  += spec;
			curr[i].voxel[k].opacity_corr += spec;

			if (curr[i].voxel[k].red_shaded > 1.0) curr[i].voxel[k].red_shaded = 1.0;
			if (curr[i].voxel[k].green_shaded > 1.0) curr[i].voxel[k].green_shaded = 1.0; 
			if (curr[i].voxel[k].blue_shaded > 1.0) curr[i].voxel[k].blue_shaded = 1.0;
			if (curr[i].voxel[k].opacity_corr > 1.0) curr[i].voxel[k].opacity_corr = 1.0;
		    }
		}
	    }

//---------------------------------------

            //voxel-pointer 2 is already in right position; find value for voxel-pointer 1
	    do {
		indexRunlength += curr[i].runlength[runPtr1];
		if (state1 == 0) {
		    state1 = 1;
		} else {
		    state1 = 0;
		    voxelPtr1 += curr[i].runlength[runPtr1];
		}
		++runPtr1;
	    } while (indexRunlength < volumeWidth);

	    indexRunlength = 0;

	    dword skip;

            //trace through the plane
	    while(voxelPtr2 < curr[i].size) {

                //if neccessary, skip zero-length runs
		while (curr[i].runlength[runPtr1] == 0) { 
		    ++runPtr1;
		    count1 = 0;
		    if (state1 == 0) state1 = 1; else state1 = 0;
		}
		while (curr[i].runlength[runPtr2] == 0) {
		    ++runPtr2;
		    count2 = 0;
		    if (state2 == 0) state2 = 1; else state2 = 0;
		}

                //if inside a transparent run, skip voxels
		if ((state1 == 0) && (curr[i].runlength[runPtr1]-count1 > 1) && 
		    (state2 == 0) && (curr[i].runlength[runPtr2]-count2 > 1)) {

		    if (curr[i].runlength[runPtr1]-count1 < curr[i].runlength[runPtr2]-count2) {
			skip = curr[i].runlength[runPtr1]-count1-1;
		    } else {
			skip = curr[i].runlength[runPtr2]-count2-1;
		    }

		    indexRunlength += skip;

		    count1 += skip;
		    count2 += skip;
		} else {

		    word xpos = indexRunlength % volumeWidth;
		    word ypos = indexRunlength / volumeWidth;

		    // positions of the left upper and right lower sample for the intermediate image
		    float sx1 = curr[i].dim1_pos + (float)xpos * curr[i].scale;
		    float sy1 = curr[i].dim2_pos + (float)ypos * curr[i].scale;
		    float sx4 = curr[i].dim1_pos + (float)(xpos+1) * curr[i].scale;
		    float sy4 = curr[i].dim2_pos + (float)(ypos+1) * curr[i].scale;

		    pos = intermediateWidth * ((dword)sy4) + ((dword)sx4);

                    // only check this pixel, if it still is transparent
		    if ((intermediate[pos].trans > MIN_TRANSPARENCY_MAKING_INTERMEDIATE) &&
			(floor(sx1) != floor(sx4)) && (floor(sy1) != floor(sy4))) {

			double red = 0;
			double green = 0;
			double blue = 0;
			double opacity = 0;

                        //precompute the bilinear interpolation-coefficients
			double s1 = 1 - (sx1 - (int)sx1);
			double s2 = 1 - (sy1 - (int)sy1);

			s1 /= curr[i].scale;
			s2 /= curr[i].scale;

			double w1 = s2-s1*s2;
			double w2 = s1*s2;
			double w3 = 1-s1-s2+s1*s2;
			double w4 = s1-s1*s2;

			if (voxelPtr1 < curr[i].size) {
			    if (state1 == 1) {

				red     += w1*curr[i].voxel[voxelPtr1].red_shaded;
				green   += w1*curr[i].voxel[voxelPtr1].green_shaded;
				blue    += w1*curr[i].voxel[voxelPtr1].blue_shaded;
				opacity += w1*curr[i].voxel[voxelPtr1].opacity_corr;

				if ((curr[i].runlength[runPtr1]-count1 > 1) &&
				    (voxelPtr1+1 < curr[i].size)) {

				    red     += w2*curr[i].voxel[voxelPtr1+1].red_shaded;
				    green   += w2*curr[i].voxel[voxelPtr1+1].green_shaded;
				    blue    += w2*curr[i].voxel[voxelPtr1+1].blue_shaded;
				    opacity += w2*curr[i].voxel[voxelPtr1+1].opacity_corr;

				}
				++voxelPtr1;
			    } else {
			        if ((curr[i].runlength[runPtr1]-count1 == 1) &&   //non-transp. run ends after ptr1?
			            (curr[i].runlength[runPtr1+1]!=0) &&          //no zero-length run after current?
			            ((indexRunlength+1) % volumeWidth)!=0) {   //no line break?

				    red     += w2*curr[i].voxel[voxelPtr1].red_shaded;  // vorher voxelPtr1+1
				    green   += w2*curr[i].voxel[voxelPtr1].green_shaded;
				    blue    += w2*curr[i].voxel[voxelPtr1].blue_shaded;
				    opacity += w2*curr[i].voxel[voxelPtr1].opacity_corr;

				}
			    }
			}

			if (voxelPtr2 < curr[i].size) {
			    if (state2 == 1) {

				red     += w3*curr[i].voxel[voxelPtr2].red_shaded;
				green   += w3*curr[i].voxel[voxelPtr2].green_shaded;
				blue    += w3*curr[i].voxel[voxelPtr2].blue_shaded;
				opacity += w3*curr[i].voxel[voxelPtr2].opacity_corr;

				if ((curr[i].runlength[runPtr2]-count2 > 1) &&
				    (voxelPtr2+1 < curr[i].size)) {

				    red     += w4*curr[i].voxel[voxelPtr2+1].red_shaded;
				    green   += w4*curr[i].voxel[voxelPtr2+1].green_shaded;
				    blue    += w4*curr[i].voxel[voxelPtr2+1].blue_shaded;
				    opacity += w4*curr[i].voxel[voxelPtr2+1].opacity_corr;

				}
				++voxelPtr2;
			    } else {
				if ((curr[i].runlength[runPtr2]-count2 == 1) &&  //non-transp. run ends after ptr1?
				    (curr[i].runlength[runPtr2+1]!=0) &&         //no zero-length run after current?
				    ((indexRunlength+1) % volumeWidth)!=0) {  //no line break?
				    
				    red     += w4*curr[i].voxel[voxelPtr2].red_shaded;  //vorher voxelPtr2+1
				    green   += w4*curr[i].voxel[voxelPtr2].green_shaded;
				    blue    += w4*curr[i].voxel[voxelPtr2].blue_shaded;
				    opacity += w4*curr[i].voxel[voxelPtr2].opacity_corr;
				}
			    }
			}


			if (opacity > 0) {

			    intermediate[pos].offset = 1;

			    double fac = intermediate[pos].trans;

			    intermediate[pos].red   += red * fac;
			    intermediate[pos].green += green * fac;
			    intermediate[pos].blue  += blue * fac;

			    intermediate[pos].trans *= (1-opacity);
			}

		    } else {

			if (state1 == 1) {
			    ++voxelPtr1;
			}

			if (state2 == 1) {
			    ++voxelPtr2;
			} 

		    }

                    //one step foreward through the volume
		    ++indexRunlength;

		    ++count1;
		    ++count2;

		    if (count1 >= curr[i].runlength[runPtr1]) {
			++runPtr1;
			count1 = 0;
			if (state1 == 1) state1 = 0; else state1 = 1;
		    }

		    if (count2 >= curr[i].runlength[runPtr2]) {
			++runPtr2;
			count2 = 0;
			if (state2 == 1) state2 = 0; else state2 = 1;
		    }
		}
	    }
	}
    }
}

//-----------------------------------------------------------------------
//-- Draws the perspective intermediate Image using OpenGL --------------
//-----------------------------------------------------------------------
void vu111211A::drawOpenGL() 
{
    //clear screen, enable texturing...
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    glEnable(GL_TEXTURE_2D);
    glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

    //set parameters for texture mapping
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);

    //create new texture
    glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, glImageWidth,
		 glImageHeight, 0, GL_RGBA, GL_UNSIGNED_BYTE, glImage);

    //work with projection matrix
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();

    // the size of the OpenGL plane with the texture upon it
    float planeSize = 40*zoomValue*zoomValue;  

    glOrtho(-10,10,-10,10,-1000,1000);

    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();

    glBegin(GL_QUADS);

    glTexCoord2f(0, 0); glVertex3f(planeSize*(-1), planeSize*(-1), 0.0);
    glTexCoord2f(1, 0); glVertex3f(planeSize*(1), planeSize*(-1), 0.0);
    glTexCoord2f(1, 1); glVertex3f(planeSize*(1), planeSize*(1), 0.0);
    glTexCoord2f(0, 1); glVertex3f(planeSize*(-1), planeSize*(1), 0.0);

    glEnd();

    glFlush();

    glDisable(GL_TEXTURE_2D);
}

//-----------------------------------------------------------------------
//-- Initializes OpenGL -------------------------------------------------
//-----------------------------------------------------------------------
void vu111211A::initOpenGL(void)
{
    // selecting background color
    glClearColor(0.0, 0.0, 0.0, 0.0);

    glPixelStorei(GL_UNPACK_ALIGNMENT, 1);

    glGenTextures(1, &m_GLShearWarp);

    glBindTexture(GL_TEXTURE_2D, m_GLShearWarp);

    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);

    glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, glImageWidth,
		 glImageHeight, 0, GL_RGBA, GL_UNSIGNED_BYTE, glImage);
}


//-----------------------------------------------------------------------
//-- Renders the volume -------------------------------------------------
//-----------------------------------------------------------------------
void vu111211A::render() 
{
    computeMainViewingDirection();
    computePermutationMatrix();
    computeViewMatrix();
    computeRunlengthSizes();

    switch(viewingMode) {
	case VIEWING_MODE_ORTHO: {
	    computeShearAndWarpFactorsOrtho();
	    shearOrtho();
	    computeInvWarpOrtho();
	    if (fastClassification == 1) {

                //Save the transfer function in an array
		float *tf = new float[256];
		for (int i = 0; i < 256; ++i) {
		    tf[i] = m_TFunc[i][3];
		}
		fastClassificationObj->buildSummedAreaTable(tf,THRESHOLD_RUNLENGTH);
		fastClassificationObj->classify();

		makeIntermediateImageOrthoFastClass();
	    } else {
		makeIntermediateImageOrtho();
	    }
	    if (orthoWarpOpenGL == 0) {
		warpOrtho();
		drawOpenGL();
	    } else {
		changeFormatIntermediate();
		drawWarpOpenGL();
	    }
	    break;
	}
	case VIEWING_MODE_PERSPECTIVE: {
	    computeEyePoint();
	    computeShearPerspective();
	    shearPerspective();
	    computeWarpPerspective();
	    makeIntermediateImagePerspective();
	    warpPerspective();
	    drawOpenGL();
	    break;
	}
    }
}








