////////////////////////////////////////////////////////////////////////
//                                                                    //
// Informatikpraktikum I "Implementation of the ShearWarp Algorithm"  //
// Februar - October 2002                                             //
//                                                                    //
// author: Sebastian Zambal                                           //
//         e9826978@student.tuwien.ac.at                              //
//                                                                    //
// file:   octree.h (fast Classification                              //
//                                                                    //
////////////////////////////////////////////////////////////////////////

//!Type of node (all children are transparent) in the octree
#define ALL_TRANSPARENT     0

//!Type of node (all children are non-transparent) in the octree
#define ALL_NON_TRANSPARENT 1

//!Type of node (mixed types of children) in the octree
#define COMBINATION         2

//!Defines numerical value representing X
#define XDIR 0
//!Defines numerical value representing Y
#define YDIR 1
//!Defines numerical value representing Z
#define ZDIR 2
//!Defines numerical value representing W (for homogenous coordinates)
#define WDIR 3

#include "../intensity.h"
#include "glos.h"
#include <GL/gl.h>
#include <GL/glu.h>
#include "vuSimpleTypes.h"
#include "vuNormalTable.h"
#include "vuVector.h"
#include "vuMatrix.h"

//!Structure for the nodes of the octree for fast classification
/*!
 * One instance of this structure represents one node in the
 * octree.
 */
struct Node {
    //!The minimum value of all voxels in this subtree
    byte min;
    //!The maximum value of all voxels in this subtree
    byte max;
    //!The maximum x-coordinate of the subcube
    int maxX;
    //!The maximum y-coordinate of the subcube
    int maxY;
    //!The maximum z-coordinate of the subcube
    int maxZ;
    //!The minimum x-coordinate of the subcube
    int minX;
    //!The minimum y-coordinate of the subcube
    int minY;
    //!The minimum z-coordinate of the subcube
    int minZ;
    //!The result of a classification
    /*!
     * 0 = all voxels in this subcube are transparent
     * 1 = some voxels in this subcube are non-transparent 
     *     and some are transparent
     */
    byte classification;
    //!The children of the node
    Node *children[8] ;
    //!Are there no children?
    int noChildren;
};

// Prototypes of classes Octree, SummedAreaTable 
// and FastClassification
class Octree;
class SummedAreaTable;
class FastClassification;

//!Represents an octree for fast classification.
/*!
 * Using this octree speeds up classification. When used,
 * the volume needn't be runlength-encoded each time after
 * the transfer-function is changed.
 */
class Octree
{
  public:
    //!Constructor.
    /*!
     * @param volumeData a pointer to the whole volume data
     * @param maxX       maximum size of the volume in X
     * @param maxY       maximum size of the volume in Y
     * @param maxZ       maximum size of the volume in Z
     */
    Octree(byte *volumeData, int maxX, int maxY, int maxZ);

    //!Destructor.
    /*!
     * Calls remove() to free memory.
     */
    ~Octree();

    //!Removes the octree.
    /*!
     * Frees memory from the octree.
     *
     * @param tree pointer to the root of the octree.
     */
    void remove(Node *tree);

    //!Builds the octree.
    /*!
     * @param minX lower boundary of subvolume in X
     * @param minY lower boundary of subvolume in Y
     * @param minZ lower boundary of subvolume in Z
     * @param maxX upper boundary of subvolume in X
     * @param maxY upper boundary of subvolume in Y
     * @param maxZ upper boundary of subvolume in Z
     *
     * @return the tree representing the given subvolume.
     */
    Node *build(int minX, int minY, int minZ, 
		int maxX, int maxY, int maxZ, 
		int count);

    //!Get the value of a voxel with given coordinates
    /*!
     * @param x the x-coordinate of the voxel
     * @param y the y-coordinate of the voxel
     * @param z the z-coordinate of the voxel
     *
     * @return the value of a voxel with given coordinates
     */
    byte getData(int x, int y, int z);

    //!Classifies the octree.
    /*!
     * Calls classify(SumedAreaTable *, Node *).
     *
     * @param sat a pointer to the summed area table.
     */
    void classify(SummedAreaTable *sat);

    //!Does the classification.
    /*!
     * Recursively classifies the octree.
     *
     * @param sat   the pointer to the summed area table.
     * @param tree  the pointer to the root-node of the octree. 
     */
    void classify(SummedAreaTable *sat, Node *tree);

    //!Finds out the number of voxels that can be skipped
    /*!
     * The number of voxels that can be skipped is determinded.
     *
     * @param x the x-coordinate of the current position.
     * @param y the y-coordinate of the current position.
     * @param z the z-coordinate of the current position.
     * @param mainViewingDir the main viewing direction 
     *                       (can either be X, Y or Z)
     *
     * @return the number of voxels that can be skipped.
     */
    int skip(int x, int y, int z, int mainViewingDir);

    //!Find out the number of voxels that can be skipped rekursively.
    /*!
     * @param x the x-coordinate of the current position.
     * @param y the y-coordinate of the current position.
     * @param z the z-coordinate of the current position.
     * @param mainViewingDir the main viewing direction.
     * @param tree the pointer to the root of the octree.
     *
     * @return the number of voxels that can be skipped.
     */
    int skip(int x, int y, int z, int mainViewingDir, Node *tree);

  private:
    //!Pointer to the root of the total octree.
    Node *root;

    //!Pointer to the volume data.
    byte *data;

    //!Width of the octree
    int width;
    //!Height of the octree
    int height;
    //!Depth of the octree
    int depth;

    //!Stores the value of dim1Size*dim2Size.
    /*!
     * Is used for computing indices.
     */
    int step_dim3;
};

//!This class represents the summed area table.
/*!
 * The summed area table deals with summing up 
 * the opacities in the transferfunction.
 * This way it can be checked out whether sub-trees
 * of the total octree are transparent or not.
 */
class SummedAreaTable
{
  public: 
    //!Constructor.
    /*! 
     * Constructs the summed area table
     */
    SummedAreaTable(void);

    //!Destructor.
    /*!
     * Removes the summed area table.
     */
    ~SummedAreaTable();

    //!Build the summed area table.
    /*!
     * build up the summed area table (for each voxel-value i
     * compute the sum of all opacities greater than a given 
     * threshold from 0 up to i.
     * 
     * @param transferFunction a pointer to the transferFunction.
     * @param threshold_runlength the threshold for summation.
     */
    void build(float *transferFunction, float threshold_runlength);

    //!Given range transparent?
    /*!
     * Check whether the given range is total 
     * transparent (0 = false, 1 = true)
     *
     * @param min the lower boundary of the interval
     * @param max the upper boundary of the interval
     * 
     * @return 1 if the interval is transparent, 0 else
     */
    int transparent(byte min, byte max);

  private:
    //!The summed area table itself
    /*!
     * The summed area table is an array of integers.
     */
    int *table;
};

//!The class that controls octree and summed area table.
/*!
 * FastClassification makes octree and summed area table
 * work together and offers the interface for the Shear-
 * Warp Algorithm to use fast classification.
 */
class FastClassification
{
  public:
    //!Constructor
    /*!
     * @param volumeData the volume data
     * @param dim1Size the size of the volume data in X
     * @param dim2Size the size of the volume data in Y
     * @param dim3Size the size of the volume data in Z
     */
    FastClassification(byte *volumeData, 
		       int dim1Size, 
		       int dim2Size, 
		       int dim3Size);

    //!Destructor
    ~FastClassification(void);

    //!Builds the summed area table
    /*!
     * @param transferFunction a pointer to the transfer-function
     * @param threshold_runlength the threshold for a value 
     *                     in the sum to count.
     */
    void buildSummedAreaTable(float *transferFunction, 
			      float threshold_runlength);

    //!Classifies the volume-data in the octree
    void classify(void);

    //!Compute the maximum possible skipping of transparent voxels
    /*!
     * @param x the x-coordinate of the current position.
     * @param y the y-coordinate of the current position.
     * @param z the z-coordinate of the current position.
     * @param mainViewingDirection the main viewing direction.
     */
    int skip(int x, int y, int z, int mainViewingDirection);

  private:

    //!The min-max octree
    Octree *octree;

    //!The summed area table
    SummedAreaTable *sTable;

    //!The volume data
    byte *data;

    //!The width of the volume data
    int width;
    //!The height of the volume data
    int height;
    //!The depth of the volume data
    int depth;

    //!The transfer function
    float *TFunc;
};













