////////////////////////////////////////////////////////////////////////
//                                                                    //
// Informatikpraktikum I "Implementation of the ShearWarp Algorithm"  //
// Februar - October 2002                                             //
//                                                                    //
// author: Sebastian Zambal                                           //
//         e9826978@student.tuwien.ac.at                              //
//                                                                    //
// file:   shearWarp.cpp (regular)                                    //
//                                                                    //
////////////////////////////////////////////////////////////////////////

#include "octree.h"
#include "math.h"

//-----------------------------------------------------------------------
//-- Constructor                                                      ---
//-----------------------------------------------------------------------
Octree::Octree(byte *volumeData, int dim1Size, int dim2Size, int dim3Size)
{
    width  = dim1Size;
    height = dim2Size;
    depth  = dim3Size;
    data   = volumeData;

    step_dim3 = dim1Size * dim2Size;

    int max = width;
    if (height > max) max = height;
    if (depth  > max) max = depth;

    int treeDepth = (int)(log((double)max)/log(2.0f)) - 1;
    if (treeDepth < 1) treeDepth = 1;

    root = build(0,0,0,width-1,height-1,depth-1,treeDepth);
}

//-----------------------------------------------------------------------
//-- Destructor                                                       ---
//-----------------------------------------------------------------------
Octree::~Octree() 
{
    remove(root);
}

//-----------------------------------------------------------------------
//-- Delete octree, free memory                                       ---
//-----------------------------------------------------------------------
void Octree::remove(Node *tree) 
{
    for (int i = 0; i < 8; ++i) {
	if (tree->children[i] != 0) remove(tree->children[i]);
    }

    delete tree;
    tree = NULL;
}

//-----------------------------------------------------------------------
//-- Build octree                                                     ---
//-----------------------------------------------------------------------
Node *Octree::build(int minX, int minY, int minZ,
		    int maxX, int maxY, int maxZ, int count)
{
    Node *result;

    int partX = 1;
    int partY = 1;
    int partZ = 1;

    if (minX > maxX) minX = maxX;
    if (minY > maxY) minY = maxY;
    if (minZ > maxZ) minZ = maxZ;

    result = new Node();                 // create new node/subcube

    result->maxX = maxX;                 // save the coordinates of the subcube
    result->maxY = maxY;
    result->maxZ = maxZ;
    result->minX = minX;
    result->minY = minY;
    result->minZ = minZ;

    if (count == 0) {

	for (int i = 0; i < 8; ++i) {
	    result->children[i] = 0;                           // no children needed
	}
	result->noChildren = 1;

	result->min = 255;
	result->max = 0;

	for (int x = minX; x <= maxX; ++x) {
	    for (int y = minY; y <= maxY; ++y) {
		for (int z = minZ; z <= maxZ; ++z) {
		    byte voxel = data[x+y*width+z*step_dim3];
		    if (voxel > result->max) result->max = voxel;
		    if (voxel < result->min) result->min = voxel;
		}
	    }
	}

    } else {
	if ((minX == maxX) && (minY == maxY) && (minZ == maxZ)) {
            // domain represents only one voxel

	    for (int i = 0; i < 8; ++i) {
	        result->children[i] = 0;                           // no children needed
	    }
	    result->noChildren = 1;

	    result->min = result->max = data[minX+minY*width+minZ*step_dim3]; // get the value of the voxel

	} else {

	    int midX = (minX + maxX) / 2;
	    int midY = (minY + maxY) / 2;
	    int midZ = (minZ + maxZ) / 2;

	    if (minX == maxX) partX = 0;
	    if (minY == maxY) partY = 0;
	    if (minZ == maxZ) partZ = 0;

	    for (int i = 0; i < 8; ++i) result->children[i] = 0;

	    result->children[0] = build(minX,   minY,   minZ,   midX, midY, midZ, count-1);
	    if (partX == 1)
		result->children[1] = build(midX+1, minY,   minZ,   maxX, midY, midZ, count-1);
	    else result->children[1] = 0;
	    if (partY == 1)
		result->children[2] = build(minX,   midY+1, minZ,   midX, maxY, midZ, count-1);
	    else result->children[2] = 0;
	    if ((partX == 1) && (partY == 1)) 
		result->children[3] = build(midX+1, midY+1, minZ,   maxX, maxY, midZ, count-1);
	    else result->children[3] = 0;
	    if (partZ == 1)
		result->children[4] = build(minX,   minY,   midZ+1, midX, midY, maxZ, count-1);
	    else result->children[4] = 0;
	    if ((partX == 1) && (partZ == 1)) 
		result->children[5] = build(midX+1, minY,   midZ+1, maxX, midY, maxZ, count-1);
	    else result->children[5] = 0;
	    if ((partY == 1) && (partZ == 1))
		result->children[6] = build(minX,   midY+1, midZ+1, midX, maxY, maxZ, count-1);
	    else result->children[6] = 0;
	    if ((partX == 1) && (partY == 1) && (partZ == 1))
		result->children[7] = build(midX+1, midY+1, midZ+1, maxX, maxY, maxZ, count-1);
	    else result->children[7] = 0;

	    result->noChildren = 0;                 // there is at least one child-node

	    result->min = 255;
	    result->max = 0;

	    for (int i = 0; i < 8; ++i) {
		if (result->children[i] != 0) {
		    if (result->children[i]->min < result->min) result->min = result->children[i]->min;
		    if (result->children[i]->max > result->max) result->max = result->children[i]->max;
		}
	    }
	}
    }

    return result;
}

//-----------------------------------------------------------------------
//-- Do classification                                                ---
//-----------------------------------------------------------------------
void Octree::classify(SummedAreaTable *sat)
{
    printf("Starting classification");
    classify(sat,root);
    printf("ready.");
}

//-----------------------------------------------------------------------
//-- Do classification recursively                                    ---
//-----------------------------------------------------------------------
void Octree::classify(SummedAreaTable *sat, Node *tree)
{
    if (sat->transparent(tree->min,tree->max) == 1) {   // there are only transparent voxels in the subcube
	tree->classification = ALL_TRANSPARENT;
    } else {                                            // there are also non-transparent voxels in the subcube
	tree->classification = ALL_NON_TRANSPARENT;
	for (int i = 0; i < 8; ++i) {
	    if (tree->children[i] != 0) {
		tree->classification = COMBINATION;
		classify(sat,tree->children[i]);
	    }
	}
    }
}

//-----------------------------------------------------------------------
//-- Skip transparent voxels                                          ---
//-----------------------------------------------------------------------
int Octree::skip(int x, int y, int z, int mainViewingDir)
{
    return skip(x, y, z, mainViewingDir, root);
}

//-----------------------------------------------------------------------
//-- Skip transparent voxels recursively                              ---
//-----------------------------------------------------------------------
int Octree::skip(int x, int y, int z, int mainViewingDir, Node *tree)
{
    int result = 1;

    if (tree->classification == ALL_TRANSPARENT) {    // only transparent voxels => skip the whole subtree
	switch(mainViewingDir) {
	    case XDIR: {
		result = tree->maxY - y;
		break;
	    }
	    case YDIR: {
		result = tree->maxZ - z;
		break;
	    }
	    case ZDIR: {
		result = tree->maxX - x;
		break;
	    }
	}
    } else {

	if (tree->classification == COMBINATION) {   // both, transparent and non-transparent voxels

	    int midX = (tree->minX + tree->maxX) / 2;
	    int midY = (tree->minY + tree->maxY) / 2;
	    int midZ = (tree->minZ + tree->maxZ) / 2;

	    if (x <= midX) {
		if (y <= midY) {
		    if (z <= midZ) {
			result = skip(x, y, z, mainViewingDir, tree->children[0]);
		    } else {
			result = skip(x, y, z, mainViewingDir, tree->children[4]);
		    }
		} else {
		    if (z <= midZ) {
			result = skip(x, y, z, mainViewingDir, tree->children[2]);
		    } else {
			result = skip(x, y, z, mainViewingDir, tree->children[6]);
		    }
		}
	    } else {
		if (y <= midY) {
		    if (z <= midZ) {
			result = skip(x, y, z, mainViewingDir, tree->children[1]);
		    } else {
			result = skip(x, y, z, mainViewingDir, tree->children[5]);
		    }
		} else {
		    if (z <= midZ) {
			result = skip(x, y, z, mainViewingDir, tree->children[3]);
		    } else {
			result = skip(x, y, z, mainViewingDir, tree->children[7]);
		    }
		}
	    }
	} else {                                         // only non-transparent voxels
	    result = 1;
	}
    }

    if (result <= 0) result = 1;

    return result;
}

//-----------------------------------------------------------------------
//-- Constructor                                                      ---
//-----------------------------------------------------------------------
SummedAreaTable::SummedAreaTable() 
{
    table = new int[256];
}

//-----------------------------------------------------------------------
//-- Destructor                                                       ---
//-----------------------------------------------------------------------
SummedAreaTable::~SummedAreaTable() 
{
    if (table) delete [] table;
    table = NULL;
}

//-----------------------------------------------------------------------
//-- Build the summed area table                                      ---
//-----------------------------------------------------------------------
void SummedAreaTable::build(float *transferFunction, float threshold_runlength)
{
    byte current = 0;
    for (int i = 0; i < 256; ++i) {
	if (transferFunction[i] > threshold_runlength) {
	    current++;
	}
	table[i] = current;
    }
}

//-----------------------------------------------------------------------
//-- Given interval transparent?                                      ---
//-----------------------------------------------------------------------
int SummedAreaTable::transparent(byte min, byte max) 
{
    int result;

    if (min == max) {
	if (max == 0) {
	    if (table[max] >= 1) result = 0; else result = 1;
	} else {
	    if (table[max] > table[max-1]) result = 0; else result = 1;
	}
    } else {
        if (table[max] == table[min]) {   // transparent domain
	    result = 1;
	} else {
	    result = 0;
	}
    }

    return result;
}

//-----------------------------------------------------------------------
//-- Constructor                                                      ---
//-----------------------------------------------------------------------
FastClassification::FastClassification(byte *volumeData, int dim1Size, int dim2Size, int dim3Size)
{
    sTable = new SummedAreaTable();
    octree = new Octree(volumeData, dim1Size, dim2Size, dim3Size);

    width  = dim1Size;
    height = dim2Size;
    depth  = dim3Size;

    data = volumeData;
}

//-----------------------------------------------------------------------
//-- Destructor                                                       ---
//-----------------------------------------------------------------------
FastClassification::~FastClassification() 
{
  if (octree) delete(octree); // octree->~Octree();
  if (sTable) delete(sTable); //  sTable->~SummedAreaTable();
  octree = NULL;
  sTable = NULL;
}

//-----------------------------------------------------------------------
//-- Do classification                                                ---
//-----------------------------------------------------------------------
void FastClassification::classify() 
{
    octree->classify(sTable);
}

//-----------------------------------------------------------------------
//-- Skip transparent voxels                                          ---
//-----------------------------------------------------------------------
int FastClassification::skip(int x, int y, int z, int mainViewingDirection)
{
    int result;

    result = octree->skip(x, y, z, mainViewingDirection);

    return result;
}

//-----------------------------------------------------------------------
//-- Build summed area table                                          ---
//-----------------------------------------------------------------------
void FastClassification::buildSummedAreaTable(float *transferFunction, float threshold_runlength)
{
    sTable->build(transferFunction,threshold_runlength);
}














