//! Defines a Ray using the direction vector as sampling step.
//
// File:   vuSampleRay.cpp
// Author: Steve Kilthau
// Date:   August, 1999
// Adapted to vuVolume by Steven Bergner, 2001
/*! Derived from the vuRay class and adds a feature to change position by
  a certain sampling step, keeping track of current position in Grid.
  Because of its orientation to specific grid type probably not a general tool.
 */
#include <stddef.h>
#include <math.h>
#include "vuSampleRay.h"

using namespace ns_vu1112113;

// Default constructor
vuSampleRay::vuSampleRay() : vuRay()
{
  steps = 0;
  m_Grid = NULL;
  m_SmpDist = 1.f;
}

// Copy constructor
vuSampleRay::vuSampleRay(const vuSampleRay &r) : vuRay(r)
{
  steps = 0;
  m_Grid    = r.m_Grid;
  m_SmpDist = 1.f;
}

// Copy constructor
vuSampleRay::vuSampleRay(const vuRay &r) : vuRay(r)
{
    m_Grid    = NULL;
    m_SmpDist = 1.f;
}

// Destructor
vuSampleRay::~vuSampleRay()
{
}

// Assignment operator
vuSampleRay& vuSampleRay::operator=(vuSampleRay &rhs)
{
  vuRay::operator=(rhs);
  m_Grid    = rhs.m_Grid;
  steps	    = rhs.steps;
  //there is more to do now...
  return *this;
}

float vuSampleRay::SamplingDistance() const
{
  return m_SmpDist;
}

void vuSampleRay::SamplingDistance(float sd)
{
  m_Direction *= sd/m_Direction.norm();
  m_SmpDist = sd;
  steps = 0;
  m_Grid = NULL;
}

bool vuSampleRay::advanceRay()
{
	if(!steps || !m_Grid) return false;

	m_Position += m_Direction;
	//maintain cell indexing
	p[0]=(int)(m_Position[0]);
	if(p[0]>=m_Grid->maxX-1) return false;
	p[1]=(int)(m_Position[1]);
	if(p[1]>=m_Grid->maxY-1) return false;
	p[2]=(int)(m_Position[2]);
	if(p[2]>=m_Grid->maxZ-1) return false;
	cell[0] = m_Grid->vol+p[0]+p[1]*m_Grid->maxX+p[2]*m_Grid->layXY;
	cell[1] = cell[0]+1;
	cell[2] = cell[0]+m_Grid->maxX;
	cell[3] = cell[1]+m_Grid->maxX;
	cell[4] = cell[0]+m_Grid->layXY;
	cell[5] = cell[1]+m_Grid->layXY;
	cell[6] = cell[2]+m_Grid->layXY;
	cell[7] = cell[3]+m_Grid->layXY;

	float s[3];
	s[0] = m_Position[0]-(float)p[0];
	s[1] = m_Position[1]-(float)p[1];
	s[2] = m_Position[2]-(float)p[2];
	t[0] = 1.f-s[0];
	t[1] = 1.f-s[1];
	t[2] = 1.f-s[2];
	w[0] = w[4] = t[0]*t[1];
	w[1] = w[5] = s[0]*t[1];
	w[2] = w[6] = t[0]*s[1];
	w[3] = w[7] = s[0]*s[1];
	w[0] *= t[2];
	w[1] *= t[2];
	w[2] *= t[2];
	w[3] *= t[2];
	w[4] *= s[2];
	w[5] *= s[2];
	w[6] *= s[2];
	w[7] *= s[2];

	steps--;
	return true;
}

float vuSampleRay::getSamplePoint(DatPnt &dat, bool getnormal) const
{
	if(!m_Grid) return 0.0f;
	dat.reset();
	int v;
	vuVector norm(0.0f);
	float fdata=0, flen=0;
	for(v=0;v<8;v++)
	{
	  if( getnormal && cell[v]->len)
	    {
	      norm += cell[v]->normalVec()*w[v];
	      flen += cell[v]->len*w[v];
	    }
	  
	  dat.illum += cell[v]->illum*w[v];
	  
	  fdata += cell[v]->data*w[v];
	}
	dat.setNormalVec(norm);
	dat.data = byte(fdata);
	dat.len = byte(flen);
	return fdata;
}

bool vuSampleRay::attachToGrid(const DatGrid& grid)
{
    int m_Side, m_InSide;
    steps = 0;
    
    float t = grid.clipRay(*this,true,m_InSide);

    // if m_inside == 6, we are rendering from within the volume
    if ( m_InSide != 6 && ((t<=0.0) || (m_InSide==-1)))
      return false;

    // Find the point
    m_Position += t*m_Direction;
    vuSampleRay r1(*this);
    r1.m_Position += r1.m_Direction;	// one step into the volume
    t = grid.clipRay(r1,false,m_Side);	// only look at sides in front of ray
    if(t <= 0 || m_Side==-1) 
      {
	steps = 0;
	m_Grid = 0;
	return false;
      } else {
	steps = (int)t+1;
	m_Grid = &grid;
	return true;
      }
}
