// Author:	Steven Bergner
// Created:	Nov01

#ifndef _VUVOLUMEREGULARUNIMODAL3D1B1VALRAYCAST_H_
#define _VUVOLUMEREGULARUNIMODAL3D1B1VALRAYCAST_H_

#include "../intensity.h"
#include "vuTFunc/vuTFPreintegrated.h"

#include "vuSimpleTypes.h"
#include "vuVector.h"
#include "vuThread.h"

#include "vuSampleRay.h"
#include "vuImage.h"

enum {
    TFUNC_SMP = 0,
    TFUNC_PI,
    TFUNC_PIAW,
};

using namespace ns_vu1112113;

class vu1112113;
typedef vu1112113 vuVolumeRegularUnimodal3d1B1ValRaycast;
/** This class performs raycasting using a spectral colour model.
    This allows modelling special physical properties of light like 
    spectral absorption and metamerism.
    long name: vuVolumeRegularUnimodal3d1B1ValRaycast
    \author Steven Bergner
*/
class vu1112113 : public vu111211, public vuThread
{
  friend class DatGrid;
  friend class DatPnt;
public:

    //!Default constructor creating an empty instance.
    vu1112113();
    //!Copy constructor which does a deep copy.
    vu1112113(const vu1112113& inst);
    //!Destructor.
    virtual ~vu1112113();

    //!Assignment operator which does a deep copy.
    vu1112113& operator=(const vu1112113& rhs);

    //!Sets the camera viewing position for the render method.
    void setViewVectors(const vuVector& pos,const vuVector& vrp,const vuVector& up);

    //!Initializes open gl for rendering the volume data.
    void initOpenGL(void);
    //!Implements the abstract render() method of the vuVolume base class.
    void render();

    //!Reimplements the read() method to do some extra volume data processing.
    virtual bool read();

    //!This method reads volume data stored using the .raw data type.
    virtual bool readRaw(void);

    //! set size of the canvas to render to
    void setImageSize(dword sx, dword sy);
    //! set size of the canvas to render to
    void getImageSize(dword &sx, dword &sy);
    //!tell that the data should be rerendered on next glRender
    /*!default behaviour is to just blit the intermediate framebuffer */
    void doRefresh();
    //!returns the center of the dataset
    vuVector getCenter() const;

    //!The parameter enables/disables drawing during calculation of new image.
    void doPreviewDraw(bool predraw = true) { m_PreDraw = predraw; };
    //!Toogles the sampling distance
    void setSamplingDistance(float smpdist) {m_SamplingDistance = smpdist; };
    //! perform specular shading
    void doSpecular(bool dospec) {m_DoSpecular = dospec; };
    /** sets a method for evaluating the transfer function
	0 - evaluate integer density per sample
	1 - evaluate preintegrated tfunc per segment (front and back density)
	2 - like '1' but integral is weighted with alpha function */
    void setTFuncMethod(int which) {
	m_TFuncMethod = which;
	m_TFuncPI.useAlphaWeighting(which == TFUNC_PIAW);
	m_TFuncPI.preintegrate();
    };

	vuImage* getImage ();

	void displayFromImage ();

private:
    /** Overrides virtual vuThread::run().
	Calls shootRays to distribute rendering to two different threads. */
    void run(int whatsup, void* data);
    /** Renders the scene by casting the rays from the camera.
	The parameters can be used to render only specific pixels. That can be used
	for progressive rendering or rendering distributed to different processes.
	The default values setup for full rendering.
	\param xofs pixel offset in each line
	\param xinc increment determining the space between two pixels on one line
	\param yofs line to start with
	\param yinc step size to next line */
    void shootRays(int xofs=0, int xinc=1, int yofs=0, int yinc=1);

    //!Preprocesses volume data for rendering once it's been read.
    void preprocess(void);
    //!Cast a ray through the scene used by render()
    vuColourRGBa Cast(vuSampleRay& Vray);

private:
    vuTFPreintegrated   m_TFuncPI;	//!< preintegrated transfer function
    vuVector		center;		//!< center to rotate about
    DatGrid		m_Grid;		//!< array for storing information for each point

    vuImage		m_Image;	//!< image (just an RGB byte framebuffer)
    vuVector		lightdir;	//!< direction of light vector
    float		brightness;	//!< scalar to toogle brightness of light source
    float		diffuse;	//!< weight for diffuse gradient shading
    vuColourRGBa	m_Specular;	//!< specular colour
    vuColourRGBa	m_Background;	//!< background colour
    bool		pp_gradients;	//!< gradients already preprocessed?
    bool		m_PreDraw;	//!< draw while calculating new image
    float		m_SamplingDistance; //!< the distance used for sampling
    vuMutex		m_Mutex[2];
    bool		m_DoSpecular;	//!< flag for performing specular shading
    int			m_TFuncMethod;  //!< modified by setTFuncMethod()
};

#endif
