#include <dirent.h>
#include <string.h>
#include <sys/stat.h>
#include <stdio.h>
#include <iostream.h>
#include "SimpleDefs.h"
#include "FVR.h"
#include "vuMatrix.h"
#include "Image_io.h"
#include "Transform.h"
#include "SHarmonics.h"

#include <assert.h>
#include <fftw.h>
#include <unistd.h>

#include "vuTFunc/vuTFDesign.h"

using namespace FVR_NS;

#define ODD(x)	((x)&1)

#define TEMP_PATH "/tmp/vuVolume/"

// #define NO_THREADS

float diff_intensity = .8;
float cueCoeff = 15.0f;//.9;
// the harmonic transform coefficients for
// diffuse BRDF with max function
float my_dc[]={
  .1, // ambient coefficient
  3.14198,
  2.09440,2.09440,2.09440,
  0.78520,0.78520,0.78520,0.78520,0.78520};

float* vu1112117::getScale()
{
  return &m_Scale;
}

struct vuFVRarg {
  dword x_stop;
  dword y_stop;
  dword x_pass;
  dword y_pass;
};

//----------------------------------------------------------------------------
//------------------------- The default constructor --------------------------
//----------------------------------------------------------------------------

vu1112117::vu1112117() : m_currLight(1.0f, 0.0f, 0.0f),
			 m_AmbientColour(1.0f, 1.0f, 1.0f),
                         m_DiffuseColour(0.0f, 0.0f, 1.0f),
			 m_XStep(1.0f, 0.0f, 0.0f),
			 m_YStep(0.0f, 1.0f, 0.0f),
			 m_XAxis(1.0f, 0.0f, 0.0f),
			 m_YAxis(0.0f, 1.0f, 0.0f),
			 m_ZAxis(0.0f, 0.0f, 1.0f)
{
  m_volumeNumber = VOLUME_MAX_NUM;
  m_Scale        = 1.5f;
  m_Bias         = 0.0f;
  m_Wrap         = 0;
  m_depthCue     = true;
  m_shLight[0]   = 1.0f;

  for(int i = 0; i < VOLUME_MAX_NUM; i++) m_Yarray[i] = NULL;
  setLightPosition(vuVector(-1.0, 0.0, 0.0));
  initTransforms();

  // slice
  for (int i=0; i<VOLUME_MAX_NUM; i++) m__SliceArray[i] = NULL;
  m_Slice = NULL;

  vuTFDesign myTrans;

  myTrans.addOpacity(1,0.01f);
  myTrans.addOpacity(255,1.0f);
  myTrans.setOpacitySmoothing(0);
  myTrans.setColourSmoothing(0);

  myTrans.generateFunction();
  setTransferFunc(myTrans);
}

//----------------------------------------------------------------------------
//------------------------- The copy constructor -----------------------------
//----------------------------------------------------------------------------

#if 0
vu1112117::vu1112117(const vu1112117& inst) : vu111211(inst)
{
  assert("This has to be implemented: "
	 "vu1112117::vu1112117(const vu1112117& inst)");
}
#endif

//----------------------------------------------------------------------------
//------------------------- The destructor -----------------------------------
//----------------------------------------------------------------------------

vu1112117::~vu1112117()
{
  cerr << "destroying\n";
  destroyTransform2D();
  destroyTransforms();

  deleteVolumes();
  destroySlices();
  if (m_Data) delete m_Data;
}

void vu1112117::setIsDepthCueing(bool flag)
{
  if (m_depthCue != flag) {
    m_depthCue = flag;
  }
}

bool vu1112117::IsDepthCueing(void)
{
  return m_depthCue;
}

void vu1112117::deleteVolumes()
{
  for(int i = 0; i < VOLUME_MAX_NUM; i++)
    if(m_Yarray[i] != NULL) {
      delete m_Yarray[i];
      m_Yarray[i] = NULL;
    }
}

void vu1112117::setIsDiffuseShading(bool flag) {
  if (IsDiffuseShading() == flag)
    return;
  else if (flag)
    m_volumeNumber = VOLUME_MAX_NUM;
  else
    m_volumeNumber = 1;
  destroySlices();
}

bool vu1112117::IsDiffuseShading() {
  if (this->m_volumeNumber == VOLUME_MAX_NUM)
    return true;
  else
    return false;
}

void vu1112117::setAmbientColour(const vuColourRGBa _colour) {
  m_AmbientColour = _colour;
}
vuColourRGBa vu1112117::getAmbientColour(void) {
  return m_AmbientColour;
}

void vu1112117::setDiffuseColour(const vuColourRGBa _colour) {
  m_DiffuseColour = _colour;
}
vuColourRGBa vu1112117::getDiffuseColour(void) {
  return m_DiffuseColour;
}

//----------------------------------------------------------------------------
//------------------------- public setViewVectors() --------------------------
//----------------------------------------------------------------------------

void vu1112117::setViewVectors(const vuVector& view,const vuVector& up,const vuVector& right)
{
  m_XAxis  = right;
  m_YAxis  = up;
  m_ZAxis  = view;

  m_XStep  = right;
  m_YStep  = up;
  
  m_XStep.makeUnit();
  m_YStep.makeUnit();

  m_XStep *= (float)(m_Image.getWidth()  / m_SliceXSize);
  m_YStep *= (float)(m_Image.getHeight() / m_SliceYSize);

  updateShLight();
  m_Origin  = m_XAxis*(m_SliceXSize*-0.5f);
  m_Origin += m_YAxis*(m_SliceYSize*-0.5f);
}

//----------------------------------------------------------------------------
//------------------------- The assignment operator --------------------------
//----------------------------------------------------------------------------

vu1112117& vu1112117::operator=(const vu1112117& rhs)
{
  if (this != &rhs) {
  }
  return *this;
}

//----------------------------------------------------------------------------
//------------------------- public read() ------------------------------------
//----------------------------------------------------------------------------

bool vu1112117::doTempFilesExist(string fileName) {
  FILE *fileStream;

  fileStream = fopen(fileName.c_str(), "r");
  if (fileStream != NULL) {
    fclose(fileStream);
    return true;
  }
  else 
    return false;
}

void vu1112117::ensureTempDirectory() {
  DIR *tmpDir;

  tmpDir = opendir(TEMP_PATH);
  if (tmpDir == NULL) {
#ifdef WIN32
    mkdir(TEMP_PATH);
#else
    mkdir(TEMP_PATH,
	  S_IRUSR | S_IWUSR | S_IXUSR |
	  S_IRGRP | S_IWGRP | S_IXGRP |
	  S_IROTH | S_IWOTH | S_IXOTH);
#endif
  }
  else
    closedir(tmpDir);
}

bool vu1112117::read()
{
  string myFile(this->getFileName());

  unsigned int len = myFile.length();
  unsigned int pos = myFile.rfind("/", len);

  if ((pos != string::npos) && (pos < len)) {
    myFile = myFile.substr(pos+1, len);
  }
  len = myFile.length();

  if (len >4) { // myFile must contain at least the file extension
    myFile.replace(len-4, len, "0.fvr");
    myFile = string(TEMP_PATH) + myFile;

    if (!this->doTempFilesExist(myFile)) {
      bool success = vu111211::read();
      if (!success) return false;

      this->ensureTempDirectory();

      preprocess();

      write_fvr((char *)myFile.c_str());
    }
    { // read preprocessed files
      setWrap(m_Filter->getWidth()/ 2);
      read_shfvr((char *)myFile.c_str());
      _init();
      return true;
    }
  }
  return false;
}

void vu1112117::_init(void) {
  for(int yl = 0; yl < VOLUME_MAX_NUM; yl++) {
    wrapVolume(m_Yarray[yl]);
  }

  m_Image.init(m_Dim1Size, m_Dim2Size);

  m_Origin[0]  = (t_data)(m_Dim1Size / 2) * -1.0f;
  m_Origin[1]  = (t_data)(m_Dim2Size / 2) * -1.0f;
  m_Origin[2]  = 0.0f;

  setOversampling(1.0);

  initTransform2D(m_SliceXSize, m_SliceYSize);
}

//----------------------------------------------------------------------------
//------------------------- public readRaw() ---------------------------------
//----------------------------------------------------------------------------

bool vu1112117::readRaw(void)
{
  return false;
}

//----------------------------------------------------------------------------
//------------------------- public render() ----------------------------------
//----------------------------------------------------------------------------

void vu1112117::setIsPostProcessing(bool flag) {
  m_isPostProcessing = flag;
}
bool vu1112117::IsPostProcessing(void) {
  return m_isPostProcessing;
}

void vu1112117::render(void) {
  getBuffer()->blit();
}

vuImage* vu1112117::getBuffer ()

{
  if (IsReRendering()) {
    computeSlice();
    setIsReRendering(false);
  }
  if (IsPostProcessing()) {
    drawImageFromSlices();
    setIsPostProcessing(false);
  }
  return &m_Image;
}

//----------------------------------------------------------------------------
//------------------------- public initOpenGL() ------------------------------
//----------------------------------------------------------------------------

void vu1112117::initOpenGL(void)
{
  glClearColor(0.0f, 0.0f, 0.0f, 0.0f);
}

//----------------------------------------------------------------------------
//---------------------------- public methods --------------------------------
//----------------------------------------------------------------------------

void vu1112117::setLightPosition(const vuVector &_light)
{
  t_data lv[3];
  lv[0] = _light[0]; lv[1] = _light[1]; lv[2] = _light[2];

  t_data llv = sqrt(lv[0] * lv[0] + lv[1] * lv[1] + lv[2] * lv[2]);
  if(llv > 0) {
    lv[0] = lv[0] / llv;
    lv[1] = lv[1] / llv;
    lv[2] = lv[2] / llv;
  }
  else
    printf("Woooo %f\n", llv);
  
  m_currLight[0] = lv[0];
  m_currLight[1] = lv[1];
  m_currLight[2] = lv[2];

  updateShLight();
  //realLight(&m_shLight[1], lv);
}

#if 0
vuVector &vu1112117::getLightPosition(void) {
  
}
#endif

void vu1112117::write_fvr(char* fileName)// const
{
  cerr << "Writing transformed spherical harmonics basis to files...\n";

  int len = strlen(fileName) + 1;
  char *out = new char[len];
  memset(out, 0, len);
  strcpy(out, fileName);

  for(int yl = 0; yl < VOLUME_MAX_NUM; yl++) {
    ofstream fout;
    out[len - 6] = char('0' + yl);
    cerr << out << endl;
    fout.open(out, ios::out|ios::binary);
    write_fvr_head(fout, m_Dim1Size, m_Dim2Size, m_Dim3Size, sizeof(t_data));
    fout.write((char*)&m_Yarray[yl][0], m_Dim1Size * m_Dim2Size * m_Dim3Size * 2 * sizeof(t_data));
    fout.flush();
    fout.close();
  }
  delete out;
}

int vu1112117::computeDimension(void)
{
  dword dim = (m_Dim1Size > m_Dim2Size) ? m_Dim1Size : m_Dim2Size;
  dim = (dim > m_Dim3Size) ? dim : m_Dim3Size;
  dim = (ODD(dim)) ? dim + 1 : dim;

  cerr << "dimension = " << dim << endl;

  return dim;
}

void vu1112117::preprocess(void)
{
  int    dim    = computeDimension();
  int    sizeX  = m_Dim1Size;
  int    sizeY  = m_Dim2Size;
  int    sizeZ  = m_Dim3Size;
  t_data *myVol = new t_data[dim * dim * dim * 2];

  m_Dim1Size = m_Dim2Size = m_Dim3Size = dim;

  // copy m_Data to myVol and make dimensions equal
  read_raw(m_Data, myVol, m_Dim1Size, m_Dim2Size, m_Dim3Size,
	   sizeX, sizeY, sizeZ, sizeof(byte));

  // applying transfer function
  t_data *v_ptr = myVol;
  for (dword i = dim*dim*dim; i > 0; i--) {
    *v_ptr = (t_data)(m_TFunc.getOpacityAtPos((dword)(*v_ptr))*255);
    v_ptr+=2;
  }
  delete m_Data; m_Data = NULL;


  cerr << "Transforming to Real Spherical Harmonics...\n" << flush;
  realTrans(myVol, (int&)m_Dim1Size, (int&)m_Dim2Size, (int&)m_Dim3Size, m_Yarray);
  delete myVol;
  cerr << "Done.\n" << flush;

  cerr << "Transforming the Spherical Harmonics basis to Frequency domain...";
  cerr << endl;
  
  for(int yl = 0; yl < VOLUME_MAX_NUM; yl++) {
    if(yl == yl) { //0 
      cerr << "shifting spherical basis " << yl << " ... " << flush;
      shift3D(m_Yarray[yl], m_Dim1Size, m_Dim2Size, m_Dim3Size);
      cerr << "done\ntransforming spherical basis " << yl << " ... " << flush;
    }
    initTransform3D(m_Dim1Size, m_Dim2Size, m_Dim3Size);
    transform3D(m_Yarray[yl]);
    destroyTransform3D();

    if(yl == yl) {
      cerr << "done\nshifting spherical basis " << yl << " ... " << flush;
      shift3D(m_Yarray[yl], m_Dim1Size, m_Dim2Size, m_Dim3Size);
      cerr << "done\n" << flush;
    }
  }
  cerr << "Done.\n" << flush;
}

bool vu1112117::read_fvr(t_data* &_volume,
			 ifstream& fin,
			 dword XSize,
			 dword YSize,
			 dword ZSize,
			 dword d_size)
{
  if (d_size != sizeof(t_data)) return false;

  m_Dim1Size = XSize + m_Wrap * 2;
  m_Dim2Size = YSize + m_Wrap * 2;
  m_Dim3Size = ZSize + m_Wrap * 2;

  _volume = new t_data[m_Dim1Size * m_Dim2Size * m_Dim3Size * 2];

  read_raw_fast(fin, (byte*)_volume, m_Dim1Size, m_Dim2Size, m_Dim3Size, XSize, YSize, ZSize, d_size * 2);

  return true;
}

bool vu1112117::read_shfvr(char* fileName)
{
  dword XSize, YSize, ZSize, d_size;
  int len = strlen(fileName);
  char *in = new char[len + 1];
  strcpy(in, fileName);

  for(int yl = 0; yl < VOLUME_MAX_NUM; yl++) {
    in[len - 5] = '0' + yl;
    ifstream fin(in, ios::in|ios::binary
#if IS_NOCREATE_NEEDED
|ios::nocreate
#endif
);

    int ret = read_head(fin, XSize, YSize, ZSize, d_size);

    if(ret != 2)
      cerr << "Error reading basis " << yl << " from file " << in << endl;
    cerr << "reading spherical harmonics basis " << yl << " fvr file " << in << " ... " << flush;
    
    if (!read_fvr(m_Yarray[yl], fin, XSize, YSize, ZSize, d_size)) 
      cerr << "Error reading basis " << yl << " from file " << in << endl;
    cerr << "done" << endl;
    fin.close();
  }
  delete in;

  return true;
}

void vu1112117::setOversampling(t_data s) {
  // don't allow undersampling
  if (s < 1.0) s = 1.0;

  m_SliceXSize = (dword) ceil(m_Image.getWidth()  * s);
  m_SliceYSize = (dword) ceil(m_Image.getHeight() * s);
  m_SliceXSize = (ODD(m_SliceXSize)) ? m_SliceXSize + 1 : m_SliceXSize;
  m_SliceYSize = (ODD(m_SliceYSize)) ? m_SliceYSize + 1 : m_SliceYSize;

  m_XStep = m_XAxis * (float)(m_Image.getWidth()  / m_SliceXSize);
  m_YStep = m_YAxis * (float)(m_Image.getHeight() / m_SliceYSize);

  cerr << "Slice size: " << m_SliceXSize << ' ' << m_SliceYSize << endl;

  m_InnerXSize = (dword) floor((t_data) m_SliceXSize / ROOT_TWO);
  m_InnerYSize = (dword) floor((t_data) m_SliceYSize / ROOT_TWO);
  m_InnerXSize = (ODD(m_InnerXSize)) ? m_InnerXSize + 1 : m_InnerXSize;
  m_InnerYSize = (ODD(m_InnerYSize)) ? m_InnerYSize + 1 : m_InnerYSize;

  destroySlices();
}

void vu1112117::setWrap(dword wrap)
{
  m_Wrap = wrap;
}

void vu1112117::setFilter(Filter* filter)
{
  m_Filter = filter;
}

void vu1112117::setSliceScale(t_data scale)
{
  m_Scale = scale;
}

void vu1112117::setSliceBias(t_data bias)
{
  m_Bias = bias;
}

void vu1112117::rotateSliceX(t_data alpha)
{
  vuMatrix rot;

  rot.makeRotate(m_XAxis, alpha);

  m_YAxis  = rot * m_YAxis;
  m_ZAxis  = rot * m_ZAxis;
  m_XStep  = rot * m_XStep;
  m_YStep  = rot * m_YStep;
  m_Origin = rot * m_Origin;

  updateShLight();
}

void vu1112117::rotateSliceY(t_data alpha)
{
  vuMatrix rot;

  rot.makeRotate(m_YAxis, alpha);

  m_XAxis  = rot * m_XAxis;
  m_ZAxis  = rot * m_ZAxis;
  m_XStep  = rot * m_XStep;
  m_YStep  = rot * m_YStep;
  m_Origin = rot * m_Origin;

  updateShLight();
}

void vu1112117::rotateSliceZ(t_data alpha)
{
  vuMatrix rot;

  rot.makeRotate(m_ZAxis, alpha);

  m_XAxis  = rot * m_XAxis;
  m_YAxis  = rot * m_YAxis;
  m_XStep  = rot * m_XStep;
  m_YStep  = rot * m_YStep;
  m_Origin = rot * m_Origin;
  updateShLight();
}

dword vu1112117::getImageWidth(void) const
{
  return m_Image.getWidth();
}

dword vu1112117::getImageHeight(void) const
{
  return m_Image.getHeight();
}

dword vu1112117::getSliceWidth(void) const
{
  return m_SliceXSize;
}

dword vu1112117::getSliceHeight(void) const
{
  return m_SliceYSize;
}

// fill in zeroed wrap-arounds
void vu1112117::wrapVolume(t_data* &_volume)
{
  dword index;
  dword ii, jj ,kk;

  dword hiX = m_Dim1Size - m_Wrap - 1;
  dword hiY = m_Dim2Size - m_Wrap - 1;
  dword hiZ = m_Dim3Size - m_Wrap - 1;

  dword XSize = m_Dim1Size - m_Wrap * 2;
  dword YSize = m_Dim2Size - m_Wrap * 2;
  dword ZSize = m_Dim3Size - m_Wrap * 2;

  index = 0;
  for(dword k = 0; k < m_Dim3Size; ++k) {
    for(dword j = 0; j < m_Dim2Size; ++j) {
      for(dword i = 0; i < m_Dim1Size; ++i) {
	if (i < m_Wrap)
	  ii = i + XSize;
	else if (i > hiX)
	  ii = i - XSize;
	else {
	  index += 2;
	  continue;
	}

	if (j < m_Wrap)
	  jj = j + YSize;
	else if (j > hiY)
	  jj = j - YSize;
	else {
	  index += 2;
	  continue;
	}

	if (k < m_Wrap)
	  kk = k + ZSize;
	else if (k > hiZ)
	  kk = k - ZSize;
	else {
	  index += 2;
	  continue;
	}

	ii = vcoord(ii, jj, kk);

	_volume[index++] = _volume[ii];
	_volume[index++] = _volume[ii+1];
      }
    }
  }
}

void vu1112117::destroySlices(void)
{
  for (int i=0; i< VOLUME_MAX_NUM; i++) {
    if (m__SliceArray[i]) {
      delete [] m__SliceArray[i];
      m__SliceArray[i] = NULL;
    }
  }
  if (m_Slice != NULL) {
    delete [] m_Slice;
    m_Slice = NULL;
  }
}

void vu1112117::ensureSlices(void)
{
  if (m__SliceArray[0] == NULL) {
    dword length = m_SliceXSize * m_SliceYSize * 2;
    for (int i=0; i<m_volumeNumber; i++) m__SliceArray[i] = new t_data[length];
    _clearSlices();
  }
}

void vu1112117::clearSlices(void)
{
  ensureSlices();
  _clearSlices();
}

void vu1112117::_clearSlices(void)
{
  dword cnt = m_SliceXSize * m_SliceYSize * 2;
  for (int num = 0; num < m_volumeNumber; num++) {
    t_data* s_ptr = m__SliceArray[num];
    for (dword i = 0; i < cnt; i++) *(s_ptr++) = 0.0f;
  }
}

void vu1112117::_ensureSliceDummy(void) {
  dword length = m_SliceXSize * m_SliceYSize * 2;
  if (m_Slice == NULL)
    m_Slice = new t_data[length];
        
  t_data* s_ptr = m_Slice;
  for (dword i = 0; i < length; i++) *(s_ptr++) = 0.0f;
}

// and may the gods of good-style forgive me ...
// inner convolution loop, should be fast!
#define CONVOLVE_NO_MOD(_volume_)                                           \
{                                                                           \
  int x = (int) p[0];          	                                            \
  int y = (int) p[1];          	                                            \
  int z = (int) p[2];          	                                            \
                                                                            \
  t[0] = p[0] - x;          	                                            \
  t[1] = p[1] - y;          	                                            \
  t[2] = p[2] - z;          	                                            \
                                                                            \
  x = x - XSize + m_Wrap;                                                   \
  y = y - YSize + m_Wrap;                                                   \
  z = z - ZSize + m_Wrap;                                                   \
                                                                            \
  register t_data* fweight = m_Filter->getWeights(t);                       \
  register vuVector* fgrad = m_Filter->getGrad(t);                          \
                                                                            \
  register t_data real = 0.0f;                                              \
  register t_data imag = 0.0f;          	                            \
                                                                            \
  register t_data* v_ptr = &_volume_[vcoord(x + f2, y + f2, z + f2)];	    \
                                                                            \
  for (dword l = 0; l < f_len; l++) {                                       \
    for (dword m = 0; m < f_len; m++) {                                     \
      for (dword n = 0; n < f_len; n++) {    	                            \
        t_data tmp = *(fweight++);                                          \
        t_data tmpi = 0.0;                                                  \
        /* vuVector *vFGrad = (fgrad++);  */                                \
        if(m_depthCue)tmpi = (fgrad++)->dot(m_ZAxis) * cueCoeff/(2.0*M_PI); \
        real += (*(v_ptr) * tmp - *(v_ptr + 1) * tmpi);                     \
        imag += (*(v_ptr + 1) * tmp + *(v_ptr) * tmpi);                     \
        v_ptr -= 2;                                                         \
      }            	                                                    \
      v_ptr += fXStep;        	                                            \
    }                                                                       \
    v_ptr += fYStep;                                                        \
  }              	                                                    \
  *(s_ptr++) += light_brdf * real;          	                            \
  *(s_ptr++) += light_brdf * imag;          	                            \
  p += m_XStep;                                                             \
}

#define CONVOLVE(_volume_)                                                  \
{                                                                           \
  int x = (int) p[0];          	                                            \
  int y = (int) p[1];          	                                            \
  int z = (int) p[2];          	                                            \
                                                                            \
  t[0] = p[0] - x;          	                                            \
  t[1] = p[1] - y;          	                                            \
  t[2] = p[2] - z;          	                                            \
                                                                            \
  x = x % XSize + m_Wrap;                                                   \
  y = y % YSize + m_Wrap;                                                   \
  z = z % ZSize + m_Wrap;                                                   \
                                                                            \
  register t_data* fweight = m_Filter->getWeights(t);    	            \
  register vuVector* fgrad = m_Filter->getGrad(t);                          \
                                                                            \
  register t_data real = 0.0;          	                                    \
  register t_data imag = 0.0;          	                                    \
                                                                            \
  register t_data* v_ptr = &_volume_[vcoord(x + f2, y + f2, z + f2)];	    \
                                                                            \
  for (dword l = 0; l < f_len; l++) {      	                            \
    for (dword m = 0; m < f_len; m++) {                                     \
      for (dword n = 0; n < f_len; n++) {    	                            \
        t_data tmp = *(fweight++);                                          \
        t_data tmpi = 0.0;                                                  \
        /* vuVector *vFGrad = (fgrad++);*/                                  \
        if(m_depthCue)tmpi = (fgrad++)->dot(m_ZAxis) * cueCoeff/(2.0*M_PI); \
        real += (*(v_ptr) * tmp - *(v_ptr + 1) * tmpi);                     \
        imag += (*(v_ptr + 1) * tmp + *(v_ptr) * tmpi);                     \
        v_ptr -= 2;                                                         \
      }            	                                                    \
      v_ptr += fXStep;        	                                            \
    }                                                                       \
    v_ptr += fYStep;                                                        \
  }              	                                                    \
  *(s_ptr++) += light_brdf * real;          	                            \
  *(s_ptr++) += light_brdf * imag;          	                            \
  p += m_XStep;                                                             \
}

void vu1112117::interpolateSlice(t_data* &_volume, t_data* &_slice,
				 t_data &_shLight, t_data &_currDC,
				 dword x_stop, dword y_stop,
				 dword x_pass, dword y_pass)
{
  vuVector t;
  dword f_len = m_Filter->getWidth();
  int f2 = f_len / 2;
  x_stop *= 2;
  y_stop *= 2;
  dword stop_l = (m_SliceXSize - x_stop - 2 * x_pass) / 2;
  dword stop_r =  m_SliceXSize - x_stop - 2 * x_pass - stop_l;
  dword stop_b = (m_SliceYSize - y_stop - 2 * y_pass) / 2;
  dword x_hi = x_stop + 2 * x_pass;
  dword s_step = stop_l + stop_r;

  dword XSize = m_Dim1Size - m_Wrap * 2;
  dword YSize = m_Dim2Size - m_Wrap * 2;
  dword ZSize = m_Dim3Size - m_Wrap * 2;

  dword fXStep = (f_len - m_Dim1Size) * 2;
  dword fYStep = (f_len - m_Dim2Size) * m_Dim1Size * 2;

  vuVector q = m_Origin;
  q[0] += XSize / 2 + XSize + stop_l * m_XStep[0] + stop_b * m_YStep[0];
  q[1] += YSize / 2 + YSize + stop_l * m_XStep[1] + stop_b * m_YStep[1];
  q[2] += ZSize / 2 + ZSize + stop_l * m_XStep[2] + stop_b * m_YStep[2];

  t_data* s_ptr = &_slice[scoord(stop_l, stop_b)];

  t_data light_brdf = _shLight * _currDC;
	
  for (dword j = 0; j < y_pass; ++j) {
    vuVector p = q;
    for (dword i = 0; i < x_hi; ++i) CONVOLVE(_volume);
    s_ptr += s_step * 2;
    q += m_YStep;
  }

  for (dword j = 0; j < y_stop; ++j) {
    vuVector p = q;
    for (dword i = 0; i < x_pass; ++i) CONVOLVE(_volume);

    s_ptr += x_stop * 2;
    p[0] += x_stop * m_XStep[0];
    p[1] += x_stop * m_XStep[1];
    p[2] += x_stop * m_XStep[2];

    for (dword i = 0; i < x_pass; ++i) CONVOLVE(_volume);
    s_ptr += s_step * 2;
    q += m_YStep;
  }

  for (dword j = 0; j < y_pass; ++j) {
    vuVector p = q;
    for (dword i = 0; i < x_hi; ++i) CONVOLVE(_volume);
    s_ptr += s_step * 2;
    q += m_YStep;
  }
}

void vu1112117::interpolateSlice(t_data* &_volume, t_data* &_slice,
				 t_data &_shLight, t_data &_currDC)
{
  dword i, j;
  vuVector t;
  dword f_len = m_Filter->getWidth();
  int f2 = f_len / 2;

  dword x_pass = (m_SliceXSize - m_InnerXSize) / 2;
  dword y_pass = (m_SliceYSize - m_InnerYSize) / 2;

  dword XSize = m_Dim1Size - m_Wrap * 2;
  dword YSize = m_Dim2Size - m_Wrap * 2;
  dword ZSize = m_Dim3Size - m_Wrap * 2;

  dword fXStep = (f_len - m_Dim1Size) * 2;
  dword fYStep = (f_len - m_Dim2Size) * m_Dim1Size * 2;

  vuVector q = m_Origin;
  q[0] += XSize / 2 + XSize;
  q[1] += YSize / 2 + YSize;
  q[2] += ZSize / 2 + ZSize;

  t_data* s_ptr = _slice;

  t_data light_brdf = _shLight * _currDC;

  for (j = 0; j < y_pass; ++j) {
    vuVector p = q;
    for (i = 0; i < m_SliceXSize; ++i) CONVOLVE(_volume);
    q += m_YStep;
  }

  for (j = 0; j < m_InnerYSize; ++j) {
    vuVector p = q;
    for (i = 0; i < x_pass; ++i) CONVOLVE(_volume);
    for (i = 0; i < m_InnerXSize; ++i) CONVOLVE_NO_MOD(_volume);
    for (i = 0; i < x_pass; ++i)CONVOLVE(_volume);
    q += m_YStep;
  }

  for (j = 0; j < y_pass; ++j) {
    vuVector p = q;
    for (i = 0; i < m_SliceXSize; ++i) CONVOLVE(_volume);
    q += m_YStep;
  }
}

void vu1112117::accumulateSlices(t_data* _slice, int _from, int _to) {
  dword cnt =  m_SliceXSize * m_SliceYSize * 2;
  for (int num = _from; num < _to; num++) {
    t_data* src_ptr  = m__SliceArray[num];
    t_data* dest_ptr = _slice;
    for (dword i = 0; i < cnt; i++) *(dest_ptr++) += *(src_ptr++);
  }
}

void vu1112117::run(int whatsup, void* data) {
  vuFVRarg *arg = (vuFVRarg*)data;
  if (data == NULL)
    computeSlice(whatsup);
  else
    refineSlice(whatsup, arg->x_stop, arg->y_stop, arg->x_pass, arg->y_pass);
  m_Mutex[whatsup].unlock();
}

void vu1112117::computeSlice(int num)
{
  t_data currDC = (num==0?1.0:diff_intensity) * my_dc[num];
  interpolateSlice(m_Yarray[num], m__SliceArray[num], m_shLight[num], currDC);
}

void vu1112117::refineSlice(int num,
			    dword x_stop, dword y_stop,
			    dword x_pass, dword y_pass)
{
  t_data currDC = (num==0?1.0:diff_intensity) * my_dc[num];
  interpolateSlice(m_Yarray[num], m__SliceArray[num],
		   m_shLight[num], currDC,
		   x_stop, y_stop, x_pass, y_pass);
}

bool vu1112117::computeSlicesUsingThreads(void) {
#ifdef NO_THREADS
  return false;
#endif
  for (int i=0; i < m_volumeNumber; i++) m_Mutex[i].lock();
  if (startThread(0)) {
    for (int i=1; i < m_volumeNumber; i++) {
      startThread(i); // unlocks m_Mutex[i] if ready
    }
  }
  else {
    return false;
  }
  for (int i=0; i<m_volumeNumber; i++) m_Mutex[i].lock();
  for (int i=0; i<m_volumeNumber; i++) m_Mutex[i].unlock();
  return true;
}

bool vu1112117::refineSlicesUsingThreads(dword x_stop, dword y_stop,
					 dword x_pass, dword y_pass)
{
  vuFVRarg arg;
  arg.x_stop = x_stop; arg.y_stop = y_stop;
  arg.x_pass = x_pass; arg.y_pass = y_pass;

#ifdef NO_THREADS
  return false;
#endif
  for (int i=0; i < m_volumeNumber; i++) m_Mutex[i].lock();
  if (startThread(0, &arg)) {
    for (int i=1; i < m_volumeNumber; i++) {
      startThread(i, &arg); // unlocks m_Mutex[i] if ready
    }
  }
  else {
    return false;
  }
  for (int i=0; i<m_volumeNumber; i++) m_Mutex[i].lock();
  for (int i=0; i<m_volumeNumber; i++) m_Mutex[i].unlock();
  return true;
}

void vu1112117::drawImageFromSlices(void) {
  t_data *slice = NULL;

  ensureSlices();

  m_Image.clear();
#if 1
  // compute ambient light
  _ensureSliceDummy();
  slice = m_Slice;
  accumulateSlices(slice, 0, 1);
  shift2D(slice, m_SliceXSize, m_SliceYSize);
  transform2D(slice);
  shift2D(slice, m_SliceXSize, m_SliceYSize);
  scaleAndBias(slice, m_AmbientColour);
#endif

  _ensureSliceDummy();
  if (IsDiffuseShading()) {
    // compute diffuse light
    slice = m_Slice;
    accumulateSlices(slice, 1, m_volumeNumber);
    shift2D(slice, m_SliceXSize, m_SliceYSize);
    transform2D(slice);
    shift2D(slice, m_SliceXSize, m_SliceYSize);
    scaleAndBias(slice, m_DiffuseColour);
  }

}

void vu1112117::computeSlice(void)
{
  clearSlices();

  if (!computeSlicesUsingThreads()) {
    for(int yl = 0; yl < this->m_volumeNumber; yl++) computeSlice(yl);
  }
  setIsPostProcessing(true);
}

void vu1112117::refineSlice(dword x_stop, dword y_stop,
			    dword x_pass, dword y_pass)
{
  ensureSlices();

  if (!refineSlicesUsingThreads(x_stop, y_stop, x_pass, y_pass)) {
    for(int yl = 0; yl < this->m_volumeNumber; yl++) {
      refineSlice(yl, x_stop, y_stop, x_pass, y_pass);
    }
  }
  drawImageFromSlices();
}

void vu1112117::scaleAndBias(t_data* _slice, const vuColourRGBa &color) {
#define clip(a) (a<0?0:a>1?1:a)
#define clip255(a) (a>255?255:a)
  dword imgWidth  = m_Image.getWidth();
  dword imgHeight = m_Image.getHeight();
  dword imgStep   = (m_SliceXSize - imgWidth) * 2;
  byte* i_ptr     = m_Image.get_buffer();
  t_data* s_ptr   = _slice;

  for (dword j = 0; j < imgHeight; j++) {
    for (dword i = 0; i < imgWidth; i++) {
      register dword tmp;

      tmp = (dword)(clip(*s_ptr * m_Scale + m_Bias) * color[0] * 255);
      *(i_ptr++) = (byte)clip255(*i_ptr+tmp);
      tmp = (dword)(clip(*s_ptr * m_Scale + m_Bias) * color[1] * 255);
      *(i_ptr++) = (byte)clip255(*i_ptr+tmp);
      tmp = (dword)(clip(*s_ptr * m_Scale + m_Bias) * color[2] * 255);
      *(i_ptr++) = (byte)clip255(*i_ptr+tmp);
      s_ptr += 2;
    }
    s_ptr += imgStep;
  }
}

void vu1112117::updateShLight()
{
  vuVector newLight(0,0,0);
  vuVector vx(0,0,0), vy(0,0,0), vz(0,0,0);
  vx = m_currLight[0] * m_XAxis;
  vy = m_currLight[1] * m_YAxis;
  vz = m_currLight[2] * m_ZAxis;
  newLight = vx + vy + vz;
  realLight(&m_shLight[1], newLight.getData());
}


/****************************************************************************
 ***                            not used                                  ***
 ***************************************************************************/

#if 0
int indexOf(int x, int xMax, int y, int yMax)
{
  return 2 * (y * xMax + x);
}

void refine(t_data *fSlice, int M, int L)
#define clip(a) (a<0 ? 0 : a) // remove neg values due to approx in image
{
  t_data phaser; // undo the shift
  t_data maxval = -HUGE;
  int l, m;
  for(l=0;l<L;l++) {
    for(m=0;m<M;m++) {
      phaser=(l+m) % 2 ==0 ? 1:-1;
      if(clip(fSlice[indexOf(m, M, l, L)]) > maxval)
	maxval = clip(fSlice[indexOf(m, M, l, L)]);
    }
  }
  cerr << "maxval is " << maxval << endl;
  if(maxval >0) maxval = 1.0/maxval;
  for(l=0;l<L;l++) {
    for(m=0;m<M;m++) {
      fSlice[indexOf(m, M, l, L)] = (clip(fSlice[indexOf(m, M, l, L)])*maxval);
    }
  }
}
#endif
