/*
  Copyright (c) 2002 Tai Meng (tmeng@sfu.ca). All Rights Reserved
*/ 

/*============================================================*/

#include <GL/glut.h> //OpenGL / GLUT
#include <iostream> //iostream.h is an older library
#include <string> //for debugging and unit testing; string.h is c-style

#include "vuVector.h"

#include "vuGrid.h"

/*============================================================*/

using namespace std;

/*============================================================*/
/* CONSTRUCTORS */
/*============================================================*/

vuGrid::vuGrid()
  //All components will be initialized to 0
{
}

/*============================================================*/

vuGrid::vuGrid(const vuGrid& v)
  : xWidth(v.xWidth), yHeight(v.yHeight), zDepth(v.zDepth), 
    isDrawGrid(v.isDrawGrid)
{
}

/*============================================================*/

vuGrid::vuGrid(float width, float height, float depth)
  : xWidth(width), yHeight(height), zDepth(depth), isDrawGrid(true)
{
}

/*============================================================*/

vuGrid::vuGrid(float width, float height, float depth, bool isGrid)
  : xWidth(width), yHeight(height), zDepth(depth), isDrawGrid(isGrid)
{
}

/*============================================================*/

vuGrid::~vuGrid()
{
}

/*============================================================*/
/* MUTATORS */
/*============================================================*/
  
vuGrid& vuGrid::operator=(const vuGrid& v)
{
 //if LHS is not referencing the same object as the right hand side
 if(this != &v)
   {
     //do the copying
     xWidth = v.xWidth;
     yHeight = v.yHeight;
     zDepth = v.zDepth;
     isDrawGrid = v.isDrawGrid;
   }
 return *this;
}

/*============================================================*/

void vuGrid::disable()
{
  isDrawGrid = false;
}

/*============================================================*/

void vuGrid::enable()
{
  isDrawGrid = true;
}

/*============================================================*/

void vuGrid::toggle()
{
  isDrawGrid = !isDrawGrid;
}

/*============================================================*/
/* ACCESSORS */
/*============================================================*/

float vuGrid::getWidth() const
{
  return xWidth;
}

/*============================================================*/

float vuGrid::getHeight() const
{
  return yHeight;
}

/*============================================================*/

float vuGrid::getDepth() const
{
  return zDepth;
}

/*============================================================*/

bool vuGrid::getStatus() const
{
  return isDrawGrid;
}

/*============================================================*/

float vuGrid::getCenterX() const
{
  return xWidth / 2.0f;
}

/*============================================================*/

float vuGrid::getCenterY() const
{
  return yHeight / 2.0f;
}
/*============================================================*/

float vuGrid::getCenterZ() const
{
  return zDepth / 2.0f;
}
/*============================================================*/

bool vuGrid::operator==(const vuGrid& v) const
{ 
  return (xWidth == v.xWidth &&
	  yHeight == v.yHeight &&
	  zDepth == v.zDepth);
}

/*============================================================*/

bool vuGrid::operator!=(const vuGrid& v) const
{
  return !vuGrid::operator==(v);
}

/*============================================================*/

void vuGrid::drawLine(vuVector& start, vuVector& end) const
{
  //the line has to be opaque else it won't draw in blend mode
  float c1 = 0.5f;
  float c2 = 0.75f;
  
  glColor4f(c1, c2, 0.0f, 1.0f);
  glVertex3fv(start.getData());
  glColor4f(0.0f, c1, c2, 1.0f);
  glVertex3fv(end.getData());
}

/*============================================================*/

void vuGrid::labelAxes() const
{
  //static so that the variables need not be reinitilized every
  //time this function is called

  //but the variables here are not member attribs since they are
  //only used here

  static const float GRID_ORIGIN_X = 0.0f;
  static const float GRID_ORIGIN_Y = 0.0f;
  static const float GRID_ORIGIN_Z = 0.0f;

  const float GRID_WIDTH = xWidth;
  const float GRID_HEIGHT = yHeight;
  const float GRID_DEPTH = zDepth;

  float offset = (xWidth + yHeight + zDepth) / 30.0f;

  glBegin(GL_LINES);

  //the 'X'
  vuVector x1(GRID_WIDTH, GRID_ORIGIN_Y+offset, GRID_ORIGIN_Z);
  vuVector x2(GRID_WIDTH+offset, GRID_ORIGIN_Y-offset, GRID_ORIGIN_Z);
  drawLine(x1, x2);
  vuVector x3(GRID_WIDTH+offset, GRID_ORIGIN_Y+offset, GRID_ORIGIN_Z);
  vuVector x4(GRID_WIDTH, GRID_ORIGIN_Y-offset, GRID_ORIGIN_Z);
  drawLine(x3, x4);
 
  //the 'Y'
  vuVector y1(GRID_ORIGIN_X-(offset/2.0f), GRID_HEIGHT+(offset*2.0f), GRID_ORIGIN_Z);
  vuVector y2(GRID_ORIGIN_X, GRID_HEIGHT+offset, GRID_ORIGIN_Z);
  drawLine(y1, y2);
  vuVector y3(GRID_ORIGIN_X+(offset/2.0f), GRID_HEIGHT+(offset*2.0f), GRID_ORIGIN_Z);
  vuVector y4(GRID_ORIGIN_X-(offset/2.0f), GRID_HEIGHT, GRID_ORIGIN_Z);
  drawLine(y3, y4);

  //the 'Z'
  vuVector z1(GRID_ORIGIN_X, GRID_ORIGIN_Y+offset, GRID_DEPTH+offset);
  vuVector z2(GRID_ORIGIN_X, GRID_ORIGIN_Y+offset, GRID_DEPTH);
  drawLine(z1, z2);
  vuVector z3(GRID_ORIGIN_X, GRID_ORIGIN_Y-offset, GRID_DEPTH+offset);
  vuVector z4(GRID_ORIGIN_X, GRID_ORIGIN_Y-offset, GRID_DEPTH);
  drawLine(z3, z4);
  vuVector z5(GRID_ORIGIN_X, GRID_ORIGIN_Y+offset, GRID_DEPTH);
  vuVector z6(GRID_ORIGIN_X, GRID_ORIGIN_Y-offset, GRID_DEPTH+offset);
  drawLine(z5, z6);

  glEnd();
}

/*============================================================*/

void vuGrid::drawAxes() const
{
  const float GRID_ORIGIN_X = 0.0f;
  const float GRID_ORIGIN_Y = 0.0f;
  const float GRID_ORIGIN_Z = 0.0f;

  vuVector origin(GRID_ORIGIN_X, GRID_ORIGIN_Y, GRID_ORIGIN_Z);
  vuVector yAxis(GRID_ORIGIN_X, yHeight, GRID_ORIGIN_Z);
  vuVector xAxis(xWidth, GRID_ORIGIN_Y, GRID_ORIGIN_Z);
  vuVector zAxis(GRID_ORIGIN_X, GRID_ORIGIN_Y, zDepth);

  glBegin(GL_LINES);
  //first draw 3 axes, regardless of whether or not grid is enabled
  drawLine(origin, xAxis);
  drawLine(origin, yAxis);
  drawLine(origin, zAxis);
  glEnd();
}

/*============================================================*/

void vuGrid::drawInOpenGL() const
{
  if (isDrawGrid)
    {
      labelAxes();
      drawAxes();

      //could also draw grid walls, but that is optional
    }
}

/*============================================================*/
/* Friend Functions */
/*============================================================*/

ostream& operator<<(ostream& out, const vuGrid& v)
{
  out<<"Width: "<<v.xWidth<<endl
     <<"Height: "<<v.yHeight<<endl
     <<"Depth: "<<v.zDepth<<endl
     <<"Is grid enabled? "<<v.isDrawGrid<<endl;
  return out;
}
