/***************************************************************************
                          CellProjection.h  -  description
                             -------------------
    begin                : Thu May 1 2003
    copyright            : (C) 2003 by tmeng
    email                : tmeng@sfu.ca
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef _vu111211a_H_ //SimpleFVR was vu1112119, so 'a' is used for CellProjection
#define _vu111211a_H_ //as specified by the coding conventions.

//----------------------------------------------------------------------------

#include "../intensity.h" //for inheritance.
#include "vuImage.h" //for m_ImgBuffer
#include "vuVector.h" //for m_View
#include "vuGrid.h"

//----------------------------------------------------------------------------

//!The enums are equal to 0, 1, 2, ... etc
/*! 
  This enum is used by both CellProjector and vuCellProjector. Thus
  it should be globally accessible.
*/
enum DebugEnum { NORMAL, MANUAL_TET, ANIMATE_TET_ORDER, ANIMATE_VOXEL_ORDER }; 

//! This enum describes the various subdivision techniques used by CellProjetor
enum SubdivEnum { FIVE_FOLD, SIX_FOLD };

//----------------------------------------------------------------------------

class vu111211a;
//forward declaration for friend, can be omitted, but then next line has to go to end of file.
typedef vu111211a vuVolumeRegularUnimodal3d1B1CellProjector;

//! This class models a cell projector.
/*!
  A cell projector breaks a data set into tetrahedra, sorts, displays, and composits
  the tetrahedra from back to front order.
*/

class vuCellProjector; //forward declaration for friend so no need to include.
class vu111211a : public vu111211
{
friend class vuCellProjector;

 public:
  
  //!Default constructor creating an empty instance.
  vu111211a();
  
  //!Copy constructor which does a deep copy.
  vu111211a(const vu111211a& inst);
  
  //!Destructor.
  /*virtual*/ ~vu111211a();

  //!Assignment operator which does a deep copy.
  vu111211a& operator=(const vu111211a& rhs);

 public:
   
  //!Sets the camera viewing position for the render method.
  /*virtual*/ void setViewVectors(const vuVector& view,const vuVector& up,const vuVector& right);

  //!Initializes open gl for rendering the volume data.  
  /* 
     This doesn't seem like a virtual function (by searching in kdevelop).
     If it is a virtual function or will become one, make sure the "const"
     matches the parent function.
  */
  void initOpenGL() const;
  
  //!Implements the abstract render() method of the vuVolume base class.
  /*!
    Not const becuase render() calls writeImgBuffer() which modifies
    attribute m_ImgBuffer.
  */
  /*virtual*/ void render();

  /** Draws on the screen the image contained in the image buffer.
      Basically, it's a flush() function. */
  /* 
     This doesn't seem like a virtual function (by searching in kdevelop).
     If it is a virtual function or will become one, make sure the "const"
     matches the parent function.
  */
  void drawPic() const;

  //! Returns the currently rendered image.  
  /* 
     This doesn't seem like a virtual function (by searching in kdevelop).
     If it is a virtual function or will become one, make sure the the scope
     and qualifiers of this function matches the parent function.
  */
  vuImage* getBuffer();

  //!Reimplements the read() method to do some extra volume data processing.
  /*virtual*/ bool read();

  //!This method reads volume data stored using the .raw data type.
  /*
    I did a search in Kdevelop, and this function seems not to have been
    used. When it is used, it should be implemented as a virtual function
    in a level above and the virtual keyword here should be commented out.
   */
  virtual bool readRaw();

 private:
  
  //!Draws the voxels sorted.
  void drawVoxelsSorted() const;

  //!Draws the voxels unsorted.
  void drawVoxelsUnsorted() const;
  
  //!Draws a voxel based on the location of its origin (bottom lower left corner).
  void drawVoxel(int x, int y, int z) const;

  //!Draws a tet based on the voxel subdivision info.
  /*!
    The relative indices are indices of the vertices.
   */
  void drawTet(const int TET_RELATIVE_INDICES[4], const int VOXEL_ORIGIN[3]) const;

  //!Draws a tet by the vertices given.
  /*!
    First calculate the intensities based on the tet locations.

    Then, draw the four triangles as follows:
    
    v2
    /|\
    /  |  \
    /    |    \    
    /      |      \
    /        |        \
    /__________|__________\	
    v0	      v1	  v3

    The triangles are defined as follows:

    t0 = v0, v1, v2
    t1 = v1, v3, v2
    t2 = v0, v2, v3
    t3 = v0, v3, v1
   */
  void drawTet(int tetVertexLocations[4][3]) const;

  //! Draws a normal given a triangle.
  /*
    A triangle has 3 vertices, each with 3 locations.
   */
  void drawTetNormals(int tetVertexLocations[4][3]) const;
 
  //! Gets the orientation of a tet.
  void getTetOrientation(int tetVertexLocations[4][3], 
			 char orientation[4] /* out */) const;

  //! Gets the four normals for a tetrahedron one per face.
  void getTetNormals(int tetVertexLocations[4][3], 
		     vuVector normals[4] /* out */) const;
 
  //! gets the normal of the triangle passed in.
  vuVector getNormal(vuVector* v[3]) const;

  //! gets the intensity at a particular vertex of a tetrahedron.
  byte getIntensity(int vertex[3]) const;

  //! Initializes the grid to match the size of the data.
  void initGrid();

  /** Writes the current image buffer into m_ImgBuffer so that
      it will be accessible from outside this module.
  */
  /*!
    Although this function is not virtual, it could conceivably
    become virtual if more algorithsm implement it.
   */
  void writeImgBuffer();

 private: //for vuVolume

  //! The current image buffer.
  vuImage m_ImgBuffer;
   
  /** The view direction, which is updated as the user rotates
      the data set. */
  vuVector m_View;

 private: //for CellProjector

  //! Depending on the kind of data read in, different options become avail.
  /*! 
    Data sets of 8 or 12 bytes will enable manual display of tets for
    debugging purposes.
  */
  DebugEnum m_state;

  //! Subdivision scheme to be used choosable by user.
  SubdivEnum m_subdivScheme;

  //! Whether or not to perform sorting and triangulation.
  bool m_isSort;

  //! Whether to render, only used when data size is large.
  bool m_isRender;

  //! Indexes which tet in a voxel is to be rendered.
  /*! 
    Only used when state = MANUAL_TET
  */
  int m_tetIndex;

  //! A temporary grid used as visual aid.
  vuGrid m_grid;

 private: //class constants

  //! 5-fold subdivision.
  static const int TETS_DEF_5_FOLD[10][4];

  //! 6-fold subdivision.
  static const int TETS_DEF_6_FOLD[6][4];

  //! Triangle definition.
  static const int TRIANGLE_DEF[4][3];

  //! Voxel definition.
  static const int VOXEL_VERTEX_LOCATIONS[8][3];
};

#endif
