/***************************************************************************
                          CellProjector.cpp  -  description
                             -------------------
    begin                : Thu May 1 2003
    copyright            : (C) 2003 by tmeng
    email                : tmeng@sfu.ca
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <string.h> //for strcmp

#include "CellProjector.h"

//----------------------------------------------------------------------------
//------------------------- Class Variables Declaration ----------------------
//----------------------------------------------------------------------------

const int vu111211a::TETS_DEF_5_FOLD[10][4] = //checked by inspection
  {{4,0,1,2}, {7,1,3,2}, {4,2,1,7}, {4,7,6,2}, {4,5,7,1}, //first scheme
   {6,0,3,2}, {5,0,1,3}, {5,6,0,3}, {4,5,6,0}, {5,7,6,3}}; //second scheme

const int vu111211a::TETS_DEF_6_FOLD[6][4] = //checked by inspection
    {{5,1,7,6}, {5,1,6,4}, {4,1,6,0}, {0,1,6,2}, {6,1,3,2}, {6,1,7,3}};

//The voxel at origin has 8 vertices, each with x,y,z component
const int vu111211a::VOXEL_VERTEX_LOCATIONS[8][3] = 
  {{0,0,0}, {1,0,0}, {0,1,0}, {1,1,0},
   {0,0,1}, {1,0,1}, {0,1,1}, {1,1,1}};

const int vu111211a::TRIANGLE_DEF[4][3] = //checked by inspecting code
  {{0,1,2}, {1,3,2}, {0,2,3}, {0,3,1}};

//----------------------------------------------------------------------------
//------------------------- The default constructor --------------------------
//----------------------------------------------------------------------------

vu111211a::vu111211a()
  : m_state(NORMAL), m_subdivScheme(SIX_FOLD), m_isSort(false),
    m_isRender(false), m_tetIndex(0), m_grid()
  //m_View is set elsewhere, m_ImgBuffer set later.
{
}

//----------------------------------------------------------------------------
//------------------------- The copy constructor -----------------------------
//----------------------------------------------------------------------------

vu111211a::vu111211a(const vu111211a& inst) 
{
  *this = inst; //invokes operator=, avoids redundent code
}

//----------------------------------------------------------------------------
//------------------------- The destructor -----------------------------------
//----------------------------------------------------------------------------

vu111211a::~vu111211a()
{
}

//----------------------------------------------------------------------------
//------------------------- The assignment operator --------------------------
//----------------------------------------------------------------------------

vu111211a& vu111211a::operator=(const vu111211a& rhs)
{
  if (this != &rhs)
    {
      vu111211::operator=(rhs); //chain to parent.
      m_View = rhs.m_View;
      m_ImgBuffer = rhs.m_ImgBuffer;
      m_state = rhs.m_state;
      m_subdivScheme = rhs.m_subdivScheme;
      m_isSort = rhs.m_isSort;
      m_isRender = rhs.m_isRender;
      m_tetIndex = rhs.m_tetIndex;
      m_grid = rhs.m_grid;
    }
    return *this;
}

//----------------------------------------------------------------------------
//------------------------- public setViewVectors() --------------------------
//----------------------------------------------------------------------------

void vu111211a::setViewVectors(const vuVector& view,const vuVector& up,const vuVector& right)
{
  m_View = view; 
  //m_View is the offset between the camera and the origin
  //so it is the opposite of the viewRay.
}

//----------------------------------------------------------------------------
//------------------------- public initOpenGL() ------------------------------
//----------------------------------------------------------------------------

void vu111211a::initOpenGL() const
  //I've looked in CellProjector's initGlut function, everything that's
  //there seems to be here, plus some more. Will need to clean up later.
{
  GLfloat shininess = 50.0f; //very shiny material.
  GLfloat specular[4] = {1.0f, 1.0f, 1.0f, 1.0f};

  glMaterialf(GL_FRONT_AND_BACK, GL_SHININESS, shininess);
  glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, specular);
  glEnable(GL_COLOR_MATERIAL);
  glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
  //after this line, OpenGL pays attention to glColor
  //and not glMaterial. Without this line, it would
  //be the other way around.

  glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
  //GL_FRONT only should NOT work, coz lots of tets will need to have their backfaces
  //toward the viewer and they are needed for compositing.

  glShadeModel(GL_SMOOTH);
  glDisable(GL_DEPTH_TEST);
  glDisable(GL_ALPHA_TEST);
  glDisable(GL_CULL_FACE);

  //For blending and compositing calculations.
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

  //scaling scales normals, this normalizes them. KEEP.
  glEnable(GL_NORMALIZE); //does not have to normalize gradients later on.
  //the gradients are just like normals and are passed into glNormal()
  //should compare to see how slow this is. If software normalization is
  //faster, then should use that.

  glClearColor(0.0f, 0.0f, 0.0f, 0.0f); //background color;
  glClearDepth(1.0f); //default depth
  //value clamps to between 0 and 1. If an incoming pixel has a depth value
  //less than 1, that pixel will replace the current one. Else, it will be 
  //discarded.
}

//----------------------------------------------------------------------------
//------------------------- public render() ----------------------------------
//----------------------------------------------------------------------------

void vu111211a::render() 
{
  if(IsReRendering()) //only if redraw is desired do we render.
  {
    glClear(GL_COLOR_BUFFER_BIT); //erase color buffer

    //Be careful about the following values. Off by 1 -> very distorted images.
    //So if image looks correct, couldn't have made a mistake here.

    //quick hack to center data set, should be removed later
    //don't understand why the offset is needed, but it has
    //been working for every data set that I've been reading in.
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glTranslatef(0.5f, 0.5f, 0.0f);

    //Set the OpenGL light info. Replaces the 0th light that
    //may already be shining from somewhere. Let the light be 
    //fixed to the camera so the data set is always visible.
    glLightfv(GL_LIGHT0, GL_POSITION, m_View.getData());

    //Drawing the grid without lighting so it shows up properly.
    glDisable(GL_LIGHTING);
    m_grid.drawInOpenGL();
    glEnable(GL_LIGHTING);
    
    if ((m_DataSize <= 4096) || (m_isRender)) //16 cubed
      { //draw in real time only if data set is small.
	m_isRender = false; //only draw once for large data sets.
	if (m_isSort)
	  drawVoxelsSorted();
	else
	  drawVoxelsUnsorted();
      }
    glPopMatrix();
  }

  this->writeImgBuffer(); //writes current frame into m_ImgBuffer.
  //this line hardly takes up any time (about 6 ms) to execute. 
  //So leave it in, even if there is no need to access the image 
  //buffer from the outside.
}

//----------------------------------------------------------------------------
//------------------------- public drawPic() ---------------------------------
//----------------------------------------------------------------------------

void vu111211a::drawPic() const
  /** Draws on the screen the image contained in the image buffer.
      Basically, it's a flush() function. */
  //called by vuCellProjector::DrawFromImage().
{
  glDrawBuffer (GL_FRONT);
  glFinish ();
  
  glDisable (GL_BLEND);

  glDrawPixels (m_ImgBuffer.getWidth () - 1, m_ImgBuffer.getHeight () - 1, GL_RGB, GL_UNSIGNED_BYTE, (unsigned char *) (m_ImgBuffer.get_rgb ()));
  glDrawBuffer (GL_BACK);
  glDrawPixels (m_ImgBuffer.getWidth () - 1, m_ImgBuffer.getHeight () - 1, GL_RGB, GL_UNSIGNED_BYTE, (unsigned char *) (m_ImgBuffer.get_rgb ()));

  glFinish ();

  glDrawBuffer (GL_BACK);

  glEnable (GL_BLEND);
}

//----------------------------------------------------------------------------
//------------------------- public getBuffer() -------------------------------
//----------------------------------------------------------------------------

vuImage* vu111211a::getBuffer ()
{
  return &m_ImgBuffer;
}

//----------------------------------------------------------------------------
//------------------------- public read() ------------------------------------
//----------------------------------------------------------------------------

bool vu111211a::read()
{
  bool success = vu111211::read();
  if (!success) return false;

  //don't preprocess the gradients, cause quantized normals
  //are not entirely accurate, and float* normals take 12 times
  //as much space as the data set and will slow computations
  //down or even halt the program.

  this->initGrid();

  return true;
}

//----------------------------------------------------------------------------
//------------------------- public readRaw() ---------------------------------
//----------------------------------------------------------------------------

bool vu111211a::readRaw(void)
{
  dword len;
  ifstream in;

#ifdef IS_NOCREATE_NEEDED
  in.open(m_FileName, ios::in|ios::binary|ios::nocreate);
#else
  // The nocreate is not available on the IRIX boxes that we were compiling
  // this code on, so we had to remove this part from
  in.open(m_FileName, ios::in|ios::binary);
#endif
  if (!in.is_open()) return false;

  in.seekg(0, ios::end);
  len = in.tellg();
  in.seekg(0, ios::beg);

  in >> m_Dim1Size >> m_Dim2Size >> m_Dim3Size;
  if (in.fail()) return false;
  m_DataSize = m_Dim1Size*m_Dim2Size*m_Dim3Size; //volume::m_DataSize

  m_Data = new byte[m_DataSize]; //volume::m_Data
  in.read((char *) (m_Data), int (m_DataSize));
  if (in.fail()) return false;

  in.close();

  //see read() above as to why not preprocess the gradients.

  return true;
}

//----------------------------------------------------------------------------
//------------------------- private drawVoxelsSorted() -----------------------
//----------------------------------------------------------------------------

void vu111211a::drawVoxelsSorted() const
{
}

//----------------------------------------------------------------------------
//------------------------- private drawVoxelsUnSorted() ---------------------
//----------------------------------------------------------------------------

void vu111211a::drawVoxelsUnsorted() const
{
  //CanNOT use the static keyword, else when opening a data set with
  //different dimentions, incorrect dimensions will be used.
  const int NUM_X_VOXELS = m_Dim1Size - 1; //one less than num of intensities.
  const int NUM_Y_VOXELS = m_Dim2Size - 1;
  const int NUM_Z_VOXELS = m_Dim3Size - 1;

  int x, y, z; //indices for the three respective axes.
  for (x = 0; x < NUM_X_VOXELS; x++)
      for (y = 0; y < NUM_Y_VOXELS; y++)
	  for(z = 0; z < NUM_Z_VOXELS; z++)
	      drawVoxel(x, y, z);
}

//----------------------------------------------------------------------------
//------------------------- private drawVoxel() ------------------------------
//----------------------------------------------------------------------------

void vu111211a::drawVoxel(int x, int y, int z) const
{  
  int i;
  const int VOXEL_ORIGIN[3] = {x,y,z};
  if (m_state == MANUAL_TET)
    {
      //Need to clear canvas for redrawing.
      glClear(GL_DEPTH_BUFFER_BIT); 
      //can only draw one tet at once with above clearing call.

      glDisable(GL_LIGHTING);
      glDisable(GL_BLEND);
      glEnable(GL_DEPTH_TEST);
      
      switch (m_subdivScheme)
	{
	case FIVE_FOLD:
	  drawTet(TETS_DEF_5_FOLD[m_tetIndex], VOXEL_ORIGIN);
	  break;
	case SIX_FOLD:
	  drawTet(TETS_DEF_6_FOLD[m_tetIndex], VOXEL_ORIGIN);
	  break;
	}

      glEnable(GL_LIGHTING);
      glEnable(GL_BLEND);
      glDisable(GL_DEPTH_TEST);
    }
  else if (m_state == NORMAL)
    {
      switch (m_subdivScheme)
	{
	case FIVE_FOLD:
	  if((x+y+z)%2) //scheme 1
	    { //could merge this with else by being smart with for statement.
	      //int* tetOrder = getTetOrder();
	      for (i = 0; i < 5; i++)
		drawTet(TETS_DEF_5_FOLD[i], VOXEL_ORIGIN);
	    }
	  else //scheme 2, alternate with scheme 1 in all 3 axes
	    {
	      //int* tetOrder = getTetOrder();
	      for (i = 5; i < 10; i++)
		drawTet(TETS_DEF_5_FOLD[i], VOXEL_ORIGIN);
	    }
	  break;
	case SIX_FOLD:
	  for (i = 0; i < 6; i++)
	    drawTet(TETS_DEF_6_FOLD[i], VOXEL_ORIGIN);
	};
    }
}

//----------------------------------------------------------------------------
//------------------------- private drawTet() --------------------------------
//----------------------------------------------------------------------------

void vu111211a::drawTet(const int TET_RELATIVE_INDICES[4], const int VOXEL_ORIGIN[3]) const
{
  int tet[4][3]; //each tet has 4 vertices of 3 components each.
  int i, j;
  int vertexIndex;
  for (i = 0; i < 4; i++) //for each vertex of tet
    //compute the ith vertex of tet.
    {
      //find the vertex index on the voxel.
      vertexIndex = TET_RELATIVE_INDICES[i];
      for (j = 0; j < 3; j++) //for each of x,y,z component
	{
	  //add the xyz offset to the voxel at the origin.
	  tet[i][j] = VOXEL_VERTEX_LOCATIONS[vertexIndex][j] + VOXEL_ORIGIN[j];
	}
    }
  //seems right so far by inspecting cout statements
  drawTet(tet);
}

//----------------------------------------------------------------------------
//------------------------- private drawTet() --------------------------------
//----------------------------------------------------------------------------

void vu111211a::drawTet(int tetVertexLocations[4][3]) const
{

  int i, j; //loop indices
  int vertexIndex;
  byte intensity;

  if (m_state == MANUAL_TET)
    {
      drawTetNormals(tetVertexLocations); //draw normals for debugging
      char orientation[5];
      orientation[4] = '\0'; //nul terminated
      getTetOrientation(tetVertexLocations, orientation);
    }
   
  glBegin(GL_TRIANGLES); //each triple is a triangle.

  for (i = 0; i < 4; i++) //for each triangle
    {
      for (j = 0; j < 3; j++) //for each of its vertices
	{
	  vertexIndex = TRIANGLE_DEF[i][j]; //draw this vertex in this iteration.
	  if (m_state == NORMAL)
	    {
	      //compute the intensity at the vertex
	      intensity = this->getIntensity(tetVertexLocations[vertexIndex]);

	      glColor4f(m_TFunc[intensity][0], 
			//first [] returns RGBA color array, second [] returns float
			//in the range of 0 to 1.
			m_TFunc[intensity][1], 
			m_TFunc[intensity][2],
			m_TFunc[intensity][3]);
	    }
	  else if (m_state == MANUAL_TET)
	    {
	      glColor4f(((i+j)/7.0f+0.5f)/1.5f,
			(i/7.0f+0.5f)/1.5f,
			(j/7.0f+0.5f)/1.5f,
			1.0f);
	    }
	  glVertex3iv(tetVertexLocations[vertexIndex]);	  
	}
    }
  glEnd();
}

//----------------------------------------------------------------------------
//------------------------- private drawTetNormals() -------------------------
//----------------------------------------------------------------------------

void vu111211a::drawTetNormals(int tetVertexLocations[4][3]) const
{
  int i, j; //loop indices.
  vuVector v[4]; //vertex locations in float instead of int

  //compute the four triangle faces of the tet.
  for (i = 0; i < 4; i++) //for each triangle.
    {
      for (j = 0; j < 3; j++) //for each vertex of the triangle.
	{
	  v[i][j] = (float)(tetVertexLocations[i][j]);
	}
    }

  //Calculating a point inside the tetrahedron
  float m = 0.5f;
  vuVector p = v[0] + m*(v[1] - v[0]);
  p = p + m*(v[2] - p);
  p = p + m*(v[3] - p);

  //drawing the normals centered at p.
  glBegin(GL_LINES); //each new pair is a new line.
  vuVector normals[4]; //one for each triangle
  this->getTetNormals(tetVertexLocations, normals);
  for (i = 0; i < 4; i++)
    {
      glColor4f(i/6.0f, (i+4)/ 7.0f, i/ 4.0f, 1.0f);
      glVertex3fv(p.getData());
      glColor4f(i/4.0f, (i+3)/ 6.0f, i/ 7.0f, 1.0f);
      glVertex3fv((p+normals[i]).getData());
    }
  glEnd(); //make sure the glBegin and glEnd are not nested
}

//----------------------------------------------------------------------------
//------------------------- private getTetOrientation() ----------------------
//----------------------------------------------------------------------------

void vu111211a::getTetOrientation(int tetVertexLocations[4][3], 
				  char orientation[4] /* out */) const
{
  vuVector normals[4];
  this->getTetNormals(tetVertexLocations, normals);
  
  int i;
  for (i = 0; i < 4; i++) //for each triangle
    {
      if (normals[i].dot(m_View) > 0.0f)
	//m_View is the offset between the camera and the origin
	//so it is the opposite of the viewRay.
	{ //triangle is pointing TOWARD the viewer (opp dir as view ray)
	  orientation[i] = '+';
	}
      else orientation[i] = '-'; //pointing AWAY from viewer
      //note the if else is different from old code, since m_View
      //is the view ray negated.
    }
}

//----------------------------------------------------------------------------
//------------------------- private getTetNormals() --------------------------
//----------------------------------------------------------------------------

void vu111211a::getTetNormals(int tetVertexLocations[4][3], 
			      vuVector normals[4] /* out */) const
{
  /*static*/ const int TRIANGLE_DEF[4][3] = //checked by inspecting code
    {{0,1,2}, {1,3,2}, {0,2,3}, {0,3,1}};

  int i, j, k; //loop indices.
  int vertexIndex;
  vuVector v[4]; //vertex locations in float instead of int
  vuVector* triangles[4][3]; //4 triangles, 3 vertices each

  //compute the four triangle faces of the tet.
  for (i = 0; i < 4; i++) //for each triangle.
    {
      for (j = 0; j < 3; j++) //for each vertex of the triangle.
	{
	  v[i][j] = (float)(tetVertexLocations[i][j]); //update ith vertex
	  for (k = 0; k < 3; k++) //for each xyz component of vertex.
	    {
	      vertexIndex = TRIANGLE_DEF[i][j];
	      triangles[i][j] = &(v[vertexIndex]); //update ith triangle
	    }
	} //ith vertex and ith triangle computed, ok since triangle 
      //is composed of pointers and other vertices will be changed
      //in subsequence iteration of loop.
    }
  for (i = 0; i < 4; i++)  //calculate normal for each triangle.
    normals[i] = getNormal(triangles[i]);
}

//----------------------------------------------------------------------------
//------------------------- private getNormal() ------------------------------
//----------------------------------------------------------------------------

vuVector vu111211a::getNormal(vuVector* v[3]) const
{
  vuVector normal;
  normal = (*v[2] - *v[1]).cross(*v[0] - *v[1]);
  return normal;
}

//----------------------------------------------------------------------------
//------------------------- private getTetIntensity() -------------------------
//----------------------------------------------------------------------------

byte vu111211a::getIntensity(int vertex[3]) const
{
  //Cannot use "static" because data sets may be of different sizes
  //as the user opens and closes data set files.
  const int NUM_X_SAMPLES = m_Dim1Size;
  const int NUM_Y_SAMPLES = m_Dim2Size;

  int x = vertex[0];
  int y = vertex[1];
  int z = vertex[2];
  int intensityIndex = z * (NUM_X_SAMPLES * NUM_Y_SAMPLES) + y * NUM_X_SAMPLES + x; 

  byte intensity = this->m_Data[intensityIndex];  

  return intensity;
}

//----------------------------------------------------------------------------
//------------------------- private initGrid() -------------------------------
//----------------------------------------------------------------------------

void vu111211a::initGrid()
{
  float scale = 1.25f;
  float width = scale * (m_Dim1Size-1); //x
  float height = scale * (m_Dim2Size-1); //y
  float depth = scale * (m_Dim3Size-1); //z

  m_grid = vuGrid(width, height, depth);
}

//----------------------------------------------------------------------------
//------------------------- private writeImgBuffer() -------------------------
//----------------------------------------------------------------------------

void vu111211a::writeImgBuffer()
  /** Writes the current image buffer into m_ImgBuffer so that
      it will be accessible from outside this module.
  */
  //To be used by i.e. a parallel coordinates interface that displays
  //renderings from different renderers side by side.

  //This function needs to be re-examined to make sure it works.
{
  int retval [4];
  glGetIntegerv (GL_VIEWPORT, retval);
  m_ImgBuffer.init (retval [2] + 1, retval [3] + 1);
  glReadBuffer (GL_FRONT);
  glReadPixels (0, 0, m_ImgBuffer.getWidth () - 1, m_ImgBuffer.getHeight () - 1, GL_RGB, GL_UNSIGNED_BYTE, (unsigned char *) (m_ImgBuffer.get_rgb ()));

  //testing to see if the dimensions are updating with window resizing.
  //cout << "dim1:" << retval [2] << "dim2:" << retval [3] << endl;
}
