#include "spheric.h"
#include "vuCamera/vuParallelCamera.h"
#include "vuLightfield/vuSphericView.h"
#include "vuLinAlg/vuSpherical.h"
#include "vuLinAlg/vuMatrix.h"
#include <math.h>

template <int SIZE, class TYPE> vu1611<SIZE,TYPE>::vu1611()
{
  m_width         = 0;
  m_height        = 0;
  m_numberOfViews = 0;
  m_views         = NULL;
  m_Camera        = new vuParallelCamera();
}

//!The copy constructor.
template <int SIZE, class TYPE>
vu1611<SIZE,TYPE>::vu1611(const vu1611& inst)
{
  operator=(inst);
}

//!The assignment operator.
template <int SIZE, class TYPE>
vu1611<SIZE,TYPE>& vu1611<SIZE,TYPE>::operator=(const vu1611& rhs)
{
  if (this != &rhs) {
    vu161::operator=(rhs);
    m_width         = rhs.m_width;
    m_height        = rhs.m_height;
    m_numberOfViews = rhs.m_numberOfViews;
    // read and initialize views from m_Data
    _initViews();
  }
  return *this;
}

template <int SIZE, class TYPE>
vu1611<SIZE,TYPE>::~vu1611()
{
  if (m_views != NULL) {
    delete [] m_views;
    m_views         = NULL;
    m_numberOfViews = 0;
  }
}

template <int SIZE, class TYPE>
dword vu1611<SIZE,TYPE>::getWidth(void) const
{
  return m_width;
}

template <int SIZE, class TYPE>
dword vu1611<SIZE,TYPE>::getHeight(void) const
{
  return m_height;
}

template <int SIZE, class TYPE>
dword vu1611<SIZE,TYPE>::getNumberOfViews(void) const
{
  return m_numberOfViews;
}

template <int SIZE, class TYPE>
vuSphericView<SIZE,TYPE> *vu1611<SIZE,TYPE>::getView(dword i)
{
  return &m_views[i];
}

template <int SIZE, class TYPE>
int vu1611<SIZE,TYPE>::getIndexOfView(vuSphericView<SIZE,TYPE> *view)
{
  for (dword i=0; i<m_numberOfViews; i++) {
    if (view == &m_views[i]) return i;
  }
  return -1;
}

template <int SIZE, class TYPE>
vuSphericView<SIZE,TYPE>* &vu1611<SIZE,TYPE>::getViews()
{
  return m_views;
}

template <int SIZE, class TYPE>
const char *vu1611<SIZE,TYPE>::_typeName()
{
  if (sizeof(TYPE) == 1) // assuming byte
    return "byte";
  else // assuming float
    return "float";
}

template <int SIZE, class TYPE>
void vu1611<SIZE,TYPE>::setViewVectors(const vuVector&,const vuVector&,const vuVector&)
{
  cerr << "vu1611.setViewVectors(): is depricated and should not be";
  cerr << " supported anymore";
  throw "vu1611.setViewVectors(): is depricated and should not be"
    " supported anymore";
}

template <int SIZE, class TYPE>
bool vu1611<SIZE,TYPE>::read(FILE *file)
{
  int   ret  = 0;
  dword size = 0;
  int   len  = 0;

  if (m_views != NULL) {
    delete [] m_views;
    m_views         = NULL;
    m_numberOfViews = 0;
  }
  if (m_Data != NULL) {
    delete [] m_Data;
    m_Data     = NULL;
    m_DataSize = 0;
  }

  //Read in the base data
  bool success = vu161::read(file);
  if (!success) return false;

  //Read in the dimensions of the data
  ret = fscanf(file,"SPHERIC %lu %lu %lu ", &m_width, &m_height,
	       &m_numberOfViews);
  if (ret != 3)
    return setError("Could not read width, height or numberOfViews");

  //Make sure that this is spherical view data and that it's valid.
  if (m_width  <= 1)       return setError("Could not read width");
  if (m_height <= 1)       return setError("Could not read height");
  if (m_numberOfViews <=1) return setError("Could not read numberOfViews");

  //Read in the data size in bytes stored in the file.
  ret = fscanf(file,"DATA_SIZE %lu ", &m_DataSize);
  if (ret != 1)
    return setError("Could not read DataSize");
  if (!_isDataSizeValid(m_DataSize))
    return setError("DataSize is not valid");

  //Read in the name of the data and the type
  int typeSize;
  char dataName[64];
  char typeName[32];
  ret = fscanf(file,"FIELDS %s %d %s", dataName, &typeSize, typeName);
  if (ret  != 3)
    return setError("Could not read dataName, typeSize or typeName");
  if (typeSize != SIZE)
    return setError("typeSize is not valid");
  if (strcmp(typeName, _typeName()) != 0)
    return setError("typeName is not valid");
  //Store the name of the data
  m_DataName = dataName;

  //Read in the colour lookup table.
  fscanf(file,"\nLOOKUP_TABLE default%n",&len);
  if ((len < 21) || (fgetc(file) != '\n'))
    return setError("colour lookup table is not valid");

  //Allocate memory for the volume data
  m_Data = new byte[m_DataSize];

  //Read in the volume data according to the format of the file
  if (m_Binary)
    size = fread(m_Data,sizeof(byte),m_DataSize,file);
  else {
    ret = 1;
    dword i = 0;
    while ((ret > 0) && (i < m_DataSize))
      ret = fscanf(file," %c",&m_Data[i++]);
    size = i;
  }

  //Make sure that the right amount of data was read in
  if (size != m_DataSize) return setInvalidFormatError();

  // init all views and read their data from m_Data
  _initViews();

  m_Camera->setPosition(m_Camera->getLookAtVector() * -1);
  m_Camera->init();

  return true;
}

template <int SIZE, class TYPE>
bool vu1611<SIZE,TYPE>::write(FILE *file)
{
  int ret;
  dword size = 0;

  m_Binary = true;

  bool success = vu161::write(file);
  if (!success) return setWriteError();

  //Write the dimensions of the data
  ret = fprintf(file,"SPHERIC %lu %lu %lu\n",m_width,m_height,
		m_numberOfViews);
  if (ret <3) return setWriteError();

  //Write the total number of data points.
  ret = fprintf(file,"DATA_SIZE %lu\n",m_DataSize);
  if (ret == 0) return setWriteError();

  fprintf(file,"FIELDS ");

  //Write the name of the data
  if (!m_DataName.isEmpty())
    fprintf(file,"%s ",m_DataName.c_str());
  else
    fprintf(file,"data ");

  fprintf(file,"%d %s\nLOOKUP_TABLE default\n", SIZE, _typeName());

  _syncViewsToBuffer();

  //Write the volume data according to it's format
  if (m_Binary)
    size = fwrite(m_Data,sizeof(byte),m_DataSize,file);
  else
    {
      ret = 1;
      dword i = 0;
      while ((ret > 0) && (i < m_DataSize))
	ret = fprintf(file," %d",m_Data[i++]);
      size = i;
    }
    
  if (size != m_DataSize)
    return setWriteError();
  return true;
}

template <int SIZE, class TYPE>
void vu1611<SIZE,TYPE>::glResize(dword width, dword height)
{
  if (m_Camera != NULL) {
    m_Camera->setWidth(width);
    m_Camera->setHeight(height);
    m_Camera->init();
  }
  else {
    cerr << "Warning: vu1611::glResize() no camera set!" << endl;
  }
}

/* ------------------------------------------------------------------------ */
/* ---- Private Methods --------------------------------------------------- */
/* ------------------------------------------------------------------------ */

template <int SIZE, class TYPE>
void vu1611<SIZE,TYPE>::_initViews()
{
  if (m_Data != NULL) {
    if (m_views != NULL)         delete [] m_views;
    m_views         = new vuSphericView<SIZE,TYPE>[m_numberOfViews];

    // read views
    byte  *ptr = m_Data;
    dword step = vuSphericView<SIZE,TYPE>::getSizeInByte(m_width, m_height);

    for (dword i=0; i<m_numberOfViews; i++) {
      m_views[i].readFromBuffer(ptr, m_width, m_height);
      ptr += step;
    }
  }
}

template <int SIZE, class TYPE>
bool vu1611<SIZE,TYPE>::_isDataSizeValid(dword dataSize)
{
  dword size = vuSphericView<SIZE,TYPE>::getSizeInByte(m_width, m_height);
  return (dataSize == (size * m_numberOfViews));
}

template <int SIZE, class TYPE>
void vu1611<SIZE,TYPE>::_syncViewsToBuffer()
{
  byte  *ptr = m_Data;
  dword step = vuSphericView<SIZE,TYPE>::getSizeInByte(m_width, m_height);

  for (dword i=0; i<m_numberOfViews; i++) {
    m_views[i].writeIntoBuffer(ptr, m_width, m_height);
    ptr += step;
  }
}

template <int SIZE, class TYPE>
vuVector vu1611<SIZE,TYPE>::getCenter() const { return vuVector(0,0,0); }


template <int SIZE, class TYPE>
bool vu1611<SIZE,TYPE>::read(void)
{
  if (m_FileName.isEmpty()) return setError("No file name specified.");

  FILE *file = fopen(m_FileName,"rb");
  if (file != NULL) {
    bool success = read(file);
    fclose(file);
    return success;
  }
  else
    return setError("Could not open the specified file.");
}

template <int SIZE, class TYPE>
bool vu1611<SIZE,TYPE>::write(void)
{
  if (m_FileName.isEmpty()) return setError("No file name specified.");

  FILE *file = fopen(m_FileName,"wb");
  if (file != NULL)
    {
      bool success = write(file);
      fclose(file);
      return success;
    }
  else
    return setError("Could not open the specified file.");
}

template <int SIZE, class TYPE>
bool vu1611<SIZE,TYPE>::write(const vuString &fileName)
{
  vuString tmp    = m_FileName;
  bool     result = false;
  m_FileName = fileName;
  result = write();
  m_FileName = tmp;
  return result;
}
