#include "3d.h"
#include <fstream.h>
#include <stdio.h>

//----------------------------------------------------------------------------
//------------------------- The default constructor --------------------------
//----------------------------------------------------------------------------

template <int S>
vu1712<S>::vu1712()
{
    m_Dim1Size = 0;
    m_Dim2Size = 0;
    m_Dim3Size = 0;

    m_Spacing = 0;

    m_Dim1Origin = 0;
    m_Dim2Origin = 0;
    m_Dim3Origin = 0;
}

//----------------------------------------------------------------------------
//------------------------- The copy constructor -----------------------------
//----------------------------------------------------------------------------

template <int S>
vu1712<S>::vu1712(const vu1712& inst) : vu171(inst)
{
    m_Dim1Size = inst.m_Dim1Size;
    m_Dim2Size = inst.m_Dim2Size;
    m_Dim3Size = inst.m_Dim3Size;

    m_Spacing = inst.m_Spacing;

    m_Dim1Origin = inst.m_Dim1Origin;
    m_Dim2Origin = inst.m_Dim2Origin;
    m_Dim3Origin = inst.m_Dim3Origin;
}

//----------------------------------------------------------------------------
//------------------------- The assignment operator --------------------------
//----------------------------------------------------------------------------

template <int S>
vu1712<S>& vu1712<S>::operator=(const vu1712<S>& rhs)
{
    if (this != &rhs)
    {
        m_Dim1Size = rhs.m_Dim1Size;
        m_Dim2Size = rhs.m_Dim2Size;
        m_Dim3Size = rhs.m_Dim3Size;

        m_Spacing = rhs.m_Spacing;

        m_Dim1Origin = rhs.m_Dim1Origin;
        m_Dim2Origin = rhs.m_Dim2Origin;
        m_Dim3Origin = rhs.m_Dim3Origin;
        vu171::operator=(rhs);
    }
    return *this;
}

//----------------------------------------------------------------------------
//------------------------- The set/get methods ------------------------------
//----------------------------------------------------------------------------

template <int S>
dword vu1712<S>::getDim1Size(void) const
{
    return m_Dim1Size;
}

template <int S>
dword vu1712<S>::getDim2Size(void) const
{
    return m_Dim2Size;
}

template <int S>
dword vu1712<S>::getDim3Size(void) const
{
    return m_Dim3Size;
}

template <int S>
dword vu1712<S>::getSpacing(void) const
{
    return m_Spacing;
}

template <int S>
int vu1712<S>::getDim1Origin(void) const
{
    return m_Dim1Origin;
}

template <int S>
int vu1712<S>::getDim2Origin(void) const
{
    return m_Dim2Origin;
}

template <int S>
int vu1712<S>::getDim3Origin(void) const
{
    return m_Dim3Origin;
}


//----------------------------------------------------------------------------
//------------------------- protected readHeader() ---------------------------
//----------------------------------------------------------------------------

template <int S>
bool vu1712<S>::readHeader(FILE *file)
{
    int ret = 0;

    bool success = vu171::read(file);
    if (!success) return false;

    //Read in the dimensions of the data
    dword d4=0;
    ret = fscanf(file,"DIMENSIONS %lu %lu %lu %lu ",&m_Dim1Size,&m_Dim2Size,
          &m_Dim3Size,&d4);
    if (ret != 4) return setInvalidFormatError();

    //Read in the origin of the data
    int do4=0;
    ret = fscanf(file,"ORIGIN %i %i %i %i ",&m_Dim1Origin,&m_Dim2Origin,
		 &m_Dim3Origin,&do4);
    if (ret != 4) return setInvalidFormatError();
    //Read in the spacing of the data
    dword ds1,ds2,ds3,ds4;
    ret = fscanf(file,"SPACING %lu %lu %lu %lu ",&ds1,&ds2,&ds3,&ds4);
    if (ret != 4) return setInvalidFormatError();
    m_Spacing = ds1;
    //Read in the number of data points stored in the file.
    ret = fscanf(file,"DATA_SIZE %lu ",&m_DataSize);
    if (ret != 1) return setInvalidFormatError();
    //Make sure that this is 3d data and that it's valid.
    if ( (m_Dim1Size<=1) || (m_Dim2Size<=1) || (m_Dim3Size<=1) || (d4!=1) )
        return setInvalidFormatError();
    else if ( (ds1 != ds2) || (ds2 != ds3)) // has to be equally spaced
        return setInvalidFormatError();

    // Read name, number of channels and base type
    char dataName[64];
    char typeName[16];
    int  channels;
    ret = fscanf(file,"FIELDS %s %d %s ", dataName, &channels, typeName);
    if (ret != 3)
      return setError("Could not read dataName or number of channels");
    if (channels != S)
      return setError("wrong number of channels");
    if (strcmp(typeName, "float") != 0)
      return setError("typeName is not valid");
    //Store the name of the data
    m_DataName = dataName;

    // --- read volume ----------------------------------------------------

    return true;
}

template <int S>
bool vu1712<S>::readData(FILE *file, float* vol,
			 dword XX, dword YY, dword ZZ,
			 dword XXsmall, dword YYsmall, dword ZZsmall)
{
  if (file == NULL) return false;

  float* v_ptr = vol;

  dword lowX = (XX - XXsmall) / 2;
  dword lowY = (YY - YYsmall) / 2;
  dword lowZ = (ZZ - ZZsmall) / 2;

  dword hiX = XX - lowX - XXsmall;
  dword hiY = YY - lowY - YYsmall;
  dword hiZ = ZZ - lowZ - ZZsmall;

  dword bufSize = XXsmall * 2 * S;
  dword size    = 0;

  v_ptr = pad(v_ptr, lowZ * XX * YY * 2 * S);
  for (dword k = 0; k < ZZsmall; ++k) {

    v_ptr = pad(v_ptr, lowY * XX * 2 * S);
    for (dword j = 0; j < YYsmall; ++j) {

      v_ptr = pad(v_ptr, lowX * 2 * S);

      size = fread(v_ptr, sizeof(float), bufSize, file);
      if (size != bufSize) return false;

      v_ptr += bufSize;
      v_ptr = pad(v_ptr, hiX * 2 * S);

    }
    v_ptr = pad(v_ptr, hiY * XX * 2 * S);

  }
  v_ptr = pad(v_ptr, hiZ * XX * YY * 2 * S);

  return true;
}

//----------------------------------------------------------------------------
//------------------------- protected write() --------------------------------
//----------------------------------------------------------------------------

template <int S>
bool vu1712<S>::writeHeader(FILE *file)
{
    int ret = 0;

    bool success = vu171::write(file);
    if (!success) return false;

    //Write the dimensions of the data
    ret = fprintf(file,"DIMENSIONS %lu %lu %lu %lu\n",m_Dim1Size,m_Dim2Size,
          m_Dim3Size,(dword)1);

    //Write the origin of the data
    ret = fprintf(file,"ORIGIN %i %i %i %i\n",m_Dim1Origin,m_Dim2Origin,
          m_Dim3Origin,0);
    
    //Write the spacing of the data
    ret = fprintf(file,"SPACING %lu %lu %lu %lu\n",m_Spacing,m_Spacing,
          m_Spacing,(dword)0);

    //Write the total number of data points.
    ret = fprintf(file,"DATA_SIZE %lu\n",m_DataSize);

    if (m_DataName.isEmpty()) m_DataName = "data";
    //Write dataName, number of channels and base type
    ret = fprintf(file,"FIELDS %s %d float\n", m_DataName.c_str(), S);
    
    if (ret > 0)
        return true;
    else
        return setWriteError();
}

template <int S>
bool vu1712<S>::writeData(FILE *file, float* vol,
			  dword XX, dword YY, dword ZZ,
			  dword XXsmall, dword YYsmall, dword ZZsmall)
{
  if (file == NULL) return false;

  float* v_ptr = vol;

  dword lowX = (XX - XXsmall) / 2;
  dword lowY = (YY - YYsmall) / 2;
  dword lowZ = (ZZ - ZZsmall) / 2;

  dword hiX = XX - lowX - XXsmall;
  dword hiY = YY - lowY - YYsmall;

  dword bufSize = XXsmall * 2 * S;
  dword size    = 0;

  v_ptr += lowZ * XX * YY * 2 * S;
  for (dword k = 0; k < ZZsmall; ++k) {

    v_ptr += lowY * XX * 2 * S;
    for (dword j = 0; j < YYsmall; ++j) {

      v_ptr += lowX * 2 * S;

      size = fwrite(v_ptr, sizeof(float), bufSize, file);
      if (size != bufSize) return false;

      v_ptr += bufSize;
      v_ptr += hiX * 2 * S;
    }
    v_ptr += hiY * XX * 2 * S;
  }

  return true;
}

template <int S>
bool vu1712<S>::writeHeader(FILE *file, dword XSize, dword YSize, dword ZSize)
{
  if (file == NULL) return false;

  fprintf(file, "# vu DataFile Version 1.0\n"
	        "Volume Data\n"
                "BINARY\n"
                "DATASET FOURIER\n"
                "UNIMODAL\n"
                "DIMENSIONS %lu %lu %lu 1\n", XSize, YSize, ZSize);

  fprintf(file, "ORIGIN 0 0 0 0\n"
	        "SPACING 0 0 0 0\n"
	        "DATA_SIZE %lu\n", XSize * YSize * ZSize * S);

  fprintf(file, "FIELDS dataName %d float\n", S);
  return true;
}

template <int S>
bool vu1712<S>::readHeader(FILE *file,dword &XSize,dword &YSize,dword &ZSize)
{
  if (file == NULL) return false;

  int   i_tmp = 0;
  dword d_tmp = 0;
  int   ret   = 0;
  int   len   = 0;

  //Read in the standard vuVolume header to assert the file type.
  ret = fscanf(file,"# vu DataFile Version 1.0\n%n",&len);
  if ((len != 26)) {
    cerr << "no standard vuVolume header" << endl;
    return false;
  }

  //Volume Data?
  ret = fscanf(file,"Volume Data\n%n",&len);
  if ((len != 12)) {
    cerr << "no standard vuVolume header" << endl;
    return false;
  }
		
  //Read in the format of the file.
  ret = fscanf(file,"BINARY\n%n",&len);
  if (len != 7) {
    cerr << "no BINARAY file!" << endl;
    return false;
  }

  //Read in the Dataset type, checking if it's fourier
  ret = fscanf(file,"DATASET FOURIER\n%n",&len);
  if (len != 16) {
    cerr << "no fourier file" << endl;
    return false;
  }

  //Read in the modality of the data
  ret = fscanf(file,"UNIMODAL\n%n",&len);
  if (len != 9) {
    cerr << "no unimodal file" << endl;
    return false;
  }

  //Read in the dimensions of the data
  ret = fscanf(file,"DIMENSIONS %lu %lu %lu %lu ",&XSize,&YSize,&ZSize,&d_tmp);
  if (ret != 4) {
    cerr << " XSize=" << d_tmp  << endl;
    cerr << "wrong number of dimensions" << endl;
    return false;
  }

  //Read in the origin of the data
  ret = fscanf(file,"ORIGIN %i %i %i %i ", &i_tmp, &i_tmp, &i_tmp, &i_tmp);
  if (ret != 4) {
    cerr << "origin not set" << endl;
    return false;
  }

  //Read in the spacing of the data
  ret = fscanf(file,"SPACING %lu %lu %lu %lu ", &d_tmp,&d_tmp,&d_tmp,&d_tmp);
  if (ret != 4) {
    cerr << "spacing not set properly" << endl;
    return false;
  }

  //Read in the number of data points stored in the file.
  ret = fscanf(file,"DATA_SIZE %lu ",&d_tmp);
  if (ret != 1) {
    cerr << "no DATA_SIZE given" << endl;
    return false;
  }

  // Read name, number of channels and base type
  char dataName[64];
  char typeName[16];
  int  channels;
  ret = fscanf(file,"FIELDS %s %d %s ", dataName, &channels, typeName);
  if (ret != 3) {
    cerr << "Could not read dataName or number of channels" << endl;
    return false;
  }
  if (channels != S) {
    cerr << "wrong number of channels" << endl;
    return false;
  }

  if (strcmp(typeName, "float") != 0) {
    cerr << "typeName is not valid" << endl;
    return false;
  }

  return true;
}

// pad n spaces
template <int S>
float* vu1712<S>::pad(float* v, dword n)
{
  for (dword i = 0; i < n; ++i) {
    *(v++) = 0;
  }
  return v;
}
