#include "3d.h"
#include <fstream.h>
#include <stdio.h>
#include <GL/gl.h>

//----------------------------------------------------------------------------
//------------------------- The default constructor --------------------------
//----------------------------------------------------------------------------

vu1512::vu1512()
{
    m_Dim1Size = 0;
    m_Dim2Size = 0;
    m_Dim3Size = 0;

    m_Spacing = 0;

    m_Dim1Origin = 0;
    m_Dim2Origin = 0;
    m_Dim3Origin = 0;
}

//----------------------------------------------------------------------------
//------------------------- The copy constructor -----------------------------
//----------------------------------------------------------------------------

vu1512::vu1512(const vu1512& inst) : vu151(inst)
{
    m_Dim1Size = inst.m_Dim1Size;
    m_Dim2Size = inst.m_Dim2Size;
    m_Dim3Size = inst.m_Dim3Size;

    m_Spacing = inst.m_Spacing;

    m_Dim1Origin = inst.m_Dim1Origin;
    m_Dim2Origin = inst.m_Dim2Origin;
    m_Dim3Origin = inst.m_Dim3Origin;
}

//----------------------------------------------------------------------------
//------------------------- The assignment operator --------------------------
//----------------------------------------------------------------------------

vu1512& vu1512::operator=(const vu1512& rhs)
{
    if (this != &rhs)
    {
        m_Dim1Size = rhs.m_Dim1Size;
        m_Dim2Size = rhs.m_Dim2Size;
        m_Dim3Size = rhs.m_Dim3Size;

        m_Spacing = rhs.m_Spacing;

        m_Dim1Origin = rhs.m_Dim1Origin;
        m_Dim2Origin = rhs.m_Dim2Origin;
        m_Dim3Origin = rhs.m_Dim3Origin;
        vu151::operator=(rhs);
    }
    return *this;
}

//----------------------------------------------------------------------------
//------------------------- The set/get methods ------------------------------
//----------------------------------------------------------------------------

dword vu1512::getDim1Size(void) const
{
    return m_Dim1Size;
}

dword vu1512::getDim2Size(void) const
{
    return m_Dim2Size;
}

dword vu1512::getDim3Size(void) const
{
    return m_Dim3Size;
}

dword vu1512::getSpacing(void) const
{
    return m_Spacing;
}

int vu1512::getDim1Origin(void) const
{
    return m_Dim1Origin;
}

int vu1512::getDim2Origin(void) const
{
    return m_Dim2Origin;
}

int vu1512::getDim3Origin(void) const
{
    return m_Dim3Origin;
}

//----------------------------------------------------------------------------
//------------------------- protected read() ---------------------------------
//----------------------------------------------------------------------------

bool vu1512::read(FILE *file)
{
    int ret = 0;

    bool success = vu151::read(file);
    if (!success) return false;

    //Read in the dimensions of the data
    dword d4=0;
    ret = fscanf(file,"DIMENSIONS %lu %lu %lu %lu ",&m_Dim1Size,&m_Dim2Size,
          &m_Dim3Size,&d4);
    if (ret != 4) return setInvalidFormatError();

    //Read in the origin of the data
    int do4=0;
    ret = fscanf(file,"ORIGIN %i %i %i %i ",&m_Dim1Origin,&m_Dim2Origin,
		 &m_Dim3Origin,&do4);
    if (ret != 4) return setInvalidFormatError();

    //Read in the spacing of the data
    dword ds1,ds2,ds3,ds4;
    ret = fscanf(file,"SPACING %lu %lu %lu %lu ",&ds1,&ds2,&ds3,&ds4);
    if (ret != 4) return setInvalidFormatError();
    m_Spacing = ds1;

    //Read in the number of data points stored in the file.
    ret = fscanf(file,"POINT_DATA %lu ",&m_DataSize);
    if (ret != 1) return setInvalidFormatError();

    //Make sure that this is 3d data and that it's valid.
    if ( (m_Dim1Size<=1) || (m_Dim2Size<=1) || (m_Dim3Size<=1) || (d4!=1) )
        return setInvalidFormatError();
    else if ( (ds1 != ds2) || (ds2 != ds3))
        return setInvalidFormatError();
    else if (m_DataSize != (m_Dim1Size * m_Dim2Size * m_Dim3Size))
        return setInvalidFormatError();

    return true;
}

//----------------------------------------------------------------------------
//------------------------- protected write() --------------------------------
//----------------------------------------------------------------------------

bool vu1512::write(FILE *file)
{
    int ret = 0;
    
    bool success = vu151::write(file);
    if (!success) return false;

    //Write the dimensions of the data
    ret = fprintf(file,"DIMENSIONS %lu %lu %lu %lu\n",m_Dim1Size,m_Dim2Size,
          m_Dim3Size,(dword)1);
    
    //Write the origin of the data
    ret = fprintf(file,"ORIGIN %i %i %i %i\n",m_Dim1Origin,m_Dim2Origin,
          m_Dim3Origin,0);
    
    //Write the spacing of the data
    ret = fprintf(file,"SPACING %lu %lu %lu %lu\n",m_Spacing,m_Spacing,
          m_Spacing,(dword)0);

    //Write the total number of data points.
    ret = fprintf(file,"POINT_DATA %lu\n",m_DataSize);
    
    if (ret > 0)
        return true;
    else
        return setWriteError();
}


//----------------------------------------------------------------------------
vuVector vu1512::getVoxelPosition(int x, int y, int z) const
{
  vuVector pos;
  if(z&0x01){
    //secondary grid
    pos[0] = (float(x)+0.5)*T;
    pos[1] = (float(y)+0.5)*T;
    pos[2] = (float(z))*(T*0.5);
  } else {
    //primary grid
    pos[0] = float(x)*T;
    pos[1] = float(y)*T;
    pos[2] = float(z)*(T*0.5);
  }

  return pos;
}

void vu1512::preview(int hint)
{
  vuCamera *camera = getCameraPtr();

  if (camera != NULL) {
    glEnable(GL_DEPTH_TEST);
    glDepthFunc(GL_LESS);
    glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);

    //Set the viewport.
    camera->glViewport();

    //Set the opengl projection matrix.
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();

    camera->glInit();
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    camera->gluLookAt();

    // min. image and max. volume extensions
    int   minImg = camera->getWidth();
    vuVector volsiz(getVoxelPosition(m_Dim1Size, m_Dim2Size, m_Dim3Size));
    dword maxVol = (dword)((volsiz[0] > volsiz[1]) ? volsiz[0] : volsiz[1]);

    minImg = (minImg < camera->getHeight()) ? minImg : camera->getHeight();
    maxVol = (dword)((maxVol > volsiz[2])          ? maxVol : volsiz[2]);

    float fac = (float)maxVol/(float)minImg;
    fac*=1.5; // make it 50% bigger

    // draw the box
    drawBox((float)volsiz[0]*fac,(float)volsiz[1]*fac,(float)volsiz[2]*fac);
  }
  else {
    cerr << "Warning: Could not draw preview. No Camera set." << endl;
  }
}

void vu1512::drawBox(float sx, float sy, float sz)
{
  static GLfloat n[6][3] = {  /* Normals for the 6 faces of a cube. */
    {-1.0, 0.0, 0.0}, {0.0, 1.0, 0.0}, {1.0, 0.0, 0.0},
    {0.0, -1.0, 0.0}, {0.0, 0.0, -1.0}, {0.0, 0.0, 1.0} };
  static GLint faces[6][4] = {  /* Vertex indices for the 6 faces of a cube. */
    {0, 1, 2, 3}, {3, 2, 6, 7}, {7, 6, 5, 4},
    {4, 5, 1, 0}, {5, 6, 2, 1}, {7, 4, 0, 3} };
  static GLfloat v[8][3];  /* Will be filled in with X,Y,Z vertexes. */

  sx = sx/2;
  sy = sy/2;
  sz = sz/2;

  vuVector center(getCenter());
  glTranslatef(center[0], center[1], center[2]);

  /* Setup cube vertex data. */
  v[0][0] = v[1][0] = v[2][0] = v[3][0] = -sx;
  v[4][0] = v[5][0] = v[6][0] = v[7][0] = sx;
  v[0][1] = v[1][1] = v[4][1] = v[5][1] = -sy;
  v[2][1] = v[3][1] = v[6][1] = v[7][1] = sy;
  v[0][2] = v[3][2] = v[4][2] = v[7][2] = sz;
  v[1][2] = v[2][2] = v[5][2] = v[6][2] = -sz;

  int i;
  for (i = 0; i < 6; i++) {
    glBegin(GL_LINE_LOOP);
    glNormal3fv(&n[i][0]);
    glColor3fv(v[faces[i][0]]);
    glVertex3fv(&v[faces[i][0]][0]);
    glColor3fv(v[faces[i][1]]);
    glVertex3fv(&v[faces[i][1]][0]);
    glColor3fv(v[faces[i][2]]);
    glVertex3fv(&v[faces[i][2]][0]);
    glColor3fv(v[faces[i][3]]);
    glVertex3fv(&v[faces[i][3]][0]);
    glEnd();
  }
}
