#include <fstream.h>
#include <math.h>

#include "marchingtetrahedra.h"


//----------------------------------------------------------------------------
//------------------------- The default constructor --------------------------
//----------------------------------------------------------------------------

vu1512121::vu1512121()
{
  shading = 1;
  nTriangles = 0;
  printN = 1;
}

//----------------------------------------------------------------------------
//------------------------- The copy constructor -----------------------------
//----------------------------------------------------------------------------

vu1512121::vu1512121(const vu1512121& inst) : vu151212(inst)
{
}

//----------------------------------------------------------------------------
//------------------------- The destructor -----------------------------------
//----------------------------------------------------------------------------

vu1512121::~vu1512121()
{
}

//----------------------------------------------------------------------------
//------------------------- The assignment operator --------------------------
//----------------------------------------------------------------------------

vu1512121& vu1512121::operator=(const vu1512121& rhs)
{
    if (this != &rhs)
    {
        vu151212::operator=(rhs);
    }
    return *this;
}

//----------------------------------------------------------------------------
//------------------------- public setViewVectors() --------------------------
//----------------------------------------------------------------------------

void vu1512121::setViewVectors(const vuVector& view,const vuVector& up,const vuVector& right)
{
    m_View = view;
    m_Shift1 = right*3.2f;
    m_Shift2 = up*3.2f;
    m_Shift0 = (m_Shift1+m_Shift2)*(-0.5f);
}


//----------------------------------------------------------------------------
//------------------------- public read() ------------------------------------
//----------------------------------------------------------------------------

bool vu1512121::read()
{
    bool success = vu151212::read();
    if (!success) return false;

    return true;
}

//----------------------------------------------------------------------------
//------------------------- public readRaw() ---------------------------------
//----------------------------------------------------------------------------

bool vu1512121::readRaw(void)
{
    dword len;
    ifstream in;

#ifdef IS_NOCREATE_NEEDED
    in.open(m_FileName, ios::in|ios::binary|ios::nocreate);
#else
	// The nocreate is not available on the IRIX boxes that we were compiling
	// this code on, so we had to remove this part from
    in.open(m_FileName, ios::in|ios::binary);
#endif
    if (!in.is_open()) return false;

    in.seekg(0, ios::end);
    len = in.tellg();
    in.seekg(0, ios::beg);

    in >> m_Dim1Size >> m_Dim2Size >> m_Dim3Size;
    if (in.fail()) return false;
    m_DataSize = m_Dim1Size*m_Dim2Size*m_Dim3Size;

    m_Data = new byte[m_DataSize];
    in.read((char *) (m_Data), int (m_DataSize));
    if (in.fail()) return false;

    in.close();

    return true;
}

//----------------------------------------------------------------------------
//------------------------- public render() ----------------------------------
//----------------------------------------------------------------------------

void vu1512121::render(void)
{
  int p[4][3];
  //  float rp[4][3];

  // bcc grid
  static const int index_offsets[12][4][3] = {{{0, 0, 1}, {1, 0, 1}, {1, 0, 0}, {1, 0, 2}}, // x-direction
						{{0, 0, 1}, {1, 0, 1}, {1, 0, 2}, {1, 1, 2}},
						{{0, 0, 1}, {1, 0, 1}, {1, 1, 2}, {1, 1, 0}},
						{{0, 0, 1}, {1, 0, 1}, {1, 1, 0}, {1, 0, 0}},
						{{0, 0, 1}, {0, 1, 1}, {0, 1, 0}, {0, 1, 2}}, // y-direction
						{{0, 0, 1}, {0, 1, 1}, {0, 1, 2}, {1, 1, 2}},
						{{0, 0, 1}, {0, 1, 1}, {1, 1, 2}, {1, 1, 0}},
						{{0, 0, 1}, {0, 1, 1}, {1, 1, 0}, {0, 1, 0}},
						{{0, 0, 1}, {0, 0, 3}, {0, 0, 2}, {0, 1, 2}}, // z-direction
						{{0, 0, 1}, {0, 0, 3}, {0, 1, 2}, {1, 1, 2}},
						{{0, 0, 1}, {0, 0, 3}, {1, 1, 2}, {1, 0, 2}},
						{{0, 0, 1}, {0, 0, 3}, {1, 0, 2}, {0, 0, 2}}};

  /*
  // face divided
  static const int index_offsets[12][4][3] = {{{0, 0, 0}, {1, 0, 0}, {0, 1, 0}, {0, 0, 1}},
					      {{1, 0, 0}, {0, 1, 0}, {1, 1, 0}, {0, 0, 1}},
					      {{0, 0, 0}, {1, 0, 0}, {1, 0, 2}, {0, 0, 1}},
					      {{0, 0, 0}, {0, 0, 2}, {1, 0, 2}, {0, 0, 1}},
					      {{1, 0, 0}, {1, 1, 0}, {1, 1, 2}, {0, 0, 1}},
					      {{1, 0, 0}, {1, 0, 2}, {1, 1, 2}, {0, 0, 1}},
					      {{0, 1, 0}, {1, 1, 0}, {1, 1, 2}, {0, 0, 1}},
					      {{0, 1, 0}, {0, 1, 2}, {1, 1, 2}, {0, 0, 1}},
					      {{0, 0, 0}, {0, 1, 0}, {0, 1, 2}, {0, 0, 1}},
					      {{0, 0, 0}, {0, 0, 2}, {0, 1, 2}, {0, 0, 1}},
					      {{0, 0, 2}, {1, 0, 2}, {1, 1, 2}, {0, 0, 1}},
					      {{0, 0, 2}, {0, 1, 2}, {1, 1, 2}, {0, 0, 1}}};
  */

  for (unsigned int x = 0; x < m_Dim1Size - 1; x++)	      //	loop through cells
    for (unsigned int y = 0; y < m_Dim2Size - 1; y++)
	for (unsigned int z = 0; z < m_Dim3Size - 3; z+=2)
	{
	  for (int s=0; s < 12; s++)  // 12 simplices
	    {
	      for (int i=0; i < 4; i++)  // calculate actual indices
		{
		  p[i][0] = index_offsets[s][i][0] + x;
		  p[i][1] = index_offsets[s][i][1] + y;
		  p[i][2] = index_offsets[s][i][2] + z;
		}
	      
	      DrawSurfaceInSimplex(p, threshold);	              //	call routine to draw the triangles
	    }
	}
}

//----------------------------------------------------------------------------
//------------------------- public DrawSurfaceInSimplex() ----------------------------------
//----------------------------------------------------------------------------

void vu1512121::DrawSurfaceInSimplex(int indices[4][3], float ht)	//	draws intersection with a given simplex
{
  //SB: the changes I made in this file are just to prevent warnings...
  float vt[4][3],nt[4][3] /*, t[4][3] */;				//	local array for the vertices
  float /*normal[3],*/ g1[3], g2[3];			//	normal vector for triangles, gradients at vertices
  int v1, v2;						//	local copies of vertex ID
  float values[4];

  static const int mtFacetData[16][4] =
    {{0, 0, 0, 0}, {1, 3, 4, 0}, {1, 2, 5, 0}, {2, 3, 4, 5},
     {3, 2, 6, 0}, {1, 2, 6, 4}, {1, 5, 6, 3}, {4, 5, 6, 0},
     {4, 5, 6, 0}, {1, 3, 6, 5}, {1, 4, 6, 2}, {3, 6, 2, 0},
     {2, 5, 4, 3}, {1, 2, 5, 0}, {1, 3, 4, 0}, {0, 0, 0, 0}};

  static const int mtEdges[7][2] = {{0, 0}, {0, 1}, {1, 2}, {0, 2}, {0, 3}, {1, 3}, {2, 3}};
  
  for (int i = 0; i < 4; i++)
    values[i] = getDataValue(indices[i][0], indices[i][1], indices[i][2]);
 
  int whichSimplex = 0;					//	ID of type of cube for MC table

  for (int m = 0; m < 4; m++)				//	loop through corners, computing facet lookup
    if (ht < values[m])					//	if the corner is above desired height
      whichSimplex |= (1 << m);				//	set bit flag

  if ((whichSimplex == 0) || (whichSimplex == 15)) return;		//	bail out if no intersection
  
  int verts;

  if (mtFacetData[whichSimplex][3] == 0)
    verts = 3;
  else
    verts = 4;

  for (int i=0; i<verts; i++)
    { // for each vertex
      int edge = mtFacetData[whichSimplex][i];	//	local copies of edge ID

      v1 = mtEdges[edge][0];
      v2 = mtEdges[edge][1];
       
      float x1 = T*float(indices[v1][0]),
	y1 =     T*float(indices[v1][1]),
	z1 = 0.5*T*float(indices[v1][2]),
	x2 =     T*float(indices[v2][0]),
	y2 =     T*float(indices[v2][1]),
	z2 = 0.5*T*float(indices[v2][2]);

      if ((indices[v1][2] % 2) == 1)
	{
	  x1 += 0.5*T;
	  y1 += 0.5*T;
	}

      if ((indices[v2][2] % 2) == 1)
	{
	  x2 += 0.5*T;
	  y2 += 0.5*T;
	}

      InterpolatePoint(x1, y1, z1, values[v1], x2, y2, z2, values[v2], ht, vt[i]);
      
      switch(shading)  // compute normals
	{
	case WIRE:
	case FLAT:
	  break;     // nothing to do
	case SMOOTH:
	  computeGradient(indices[v1][0],
			  indices[v1][1],
			  indices[v1][2], g1, 1);

	  computeGradient(indices[v2][0],
			  indices[v2][1],
			  indices[v2][2], g2, 1);

	  InterpolatePoint(g1[0], g1[1], g1[2], values[v1], g2[0], g2[1], g2[2], values[v2], ht, nt[i]);

	  double nNorm = nt[i][0]*nt[i][0] + nt[i][1]*nt[i][1] + nt[i][2]*nt[i][2];	//	compute the norm
      
	  if (nNorm != 0.0)
	    {
	      nNorm = 1.0 / sqrt(nNorm);
	      nt[i][0] *= nNorm; nt[i][1] *= nNorm; nt[i][2] *= nNorm;	//	and normalize the vector
	    }
	  break;
	}
    }
  

  // lets try to determine the order in one more way...

  int edge = mtFacetData[whichSimplex][0];

  v1 = mtEdges[edge][0];
  v2 = mtEdges[edge][1];

  float x1 = T*float(indices[v1][0]),
    y1 =     T*float(indices[v1][1]),
    z1 = 0.5*T*float(indices[v1][2]),
    x2 =     T*float(indices[v2][0]),
    y2 =     T*float(indices[v2][1]),
    z2 = 0.5*T*float(indices[v2][2]);

  if ((indices[v1][2] % 2) == 1)
    {
      x1 += 0.5*T;
      y1 += 0.5*T;
    }
  
  if ((indices[v2][2] % 2) == 1)
    {
      x2 += 0.5*T;
      y2 += 0.5*T;
    }
  
  float v[3], n[3];

  if (values[v1] < values[v2])
    {
      v[0] = x2 - x1; v[1] = y2 - y1; v[2] = z2 - z1;
    }
  else
    {
      v[0] = x1 - x2; v[1] = y1 - y2; v[2] = z1 - z2;
    }

  ComputeNormal(vt[0], vt[1], vt[2], n);

  float dot = v[0]*n[0] + v[1]*n[1] + v[2]*n[2];

  drawTriangle(vt[0], vt[1], vt[2], nt[0], nt[1], nt[2], (dot > 0.0));

  if (verts == 4)  // we have a quadrangle, so lets draw another triangle...
    {
      drawTriangle(vt[0], vt[2], vt[3], nt[0], nt[2], nt[3], (dot > 0.0));
    }
} // DrawSimplex()

//----------------------------------------------------------------------------
//------------------------- public drawTriangle() ----------------------------------
//----------------------------------------------------------------------------

void vu1512121::drawTriangle(float v1[3], float v2[3], float v3[3],
			       float n1[3], float n2[3], float n3[3], bool order)
{
  nTriangles++;        // count triangles

  switch(shading)
    {
    case WIRE:
      glBegin(GL_LINES);
      glVertex3fv(v1);
      glVertex3fv(v2);
	  
      glVertex3fv(v2);
      glVertex3fv(v3);
	  
      glVertex3fv(v3);
      glVertex3fv(v1);
      glEnd();
      break;
    case FLAT:
      if (order)
	{
	  ComputeNormal(v1, v2, v3, n1);
	  
	  glBegin(GL_TRIANGLES);
	  glNormal3fv(n1);			//	set normal
	  
	  glVertex3fv(v1);			//	set the first vertex
	  glVertex3fv(v2);			//	set the second vertex
	  glVertex3fv(v3);			//	set the third vertex
	  glEnd();
	}
      else
	{
	  ComputeNormal(v1, v3, v2, n1);
	  
	  glBegin(GL_TRIANGLES);
	  glNormal3fv(n1);			//	set normal
	  
	  glVertex3fv(v1);			//	set the first vertex
	  glVertex3fv(v3);			//	set the second vertex
	  glVertex3fv(v2);			//	set the third vertex
	  glEnd();
	}
      break;
    case SMOOTH:
      if (order)
	{
	  glBegin(GL_TRIANGLES);
	  glNormal3fv(n1);			//	set normal
	  glVertex3fv(v1);			//	set the first vertex
	  
	  glNormal3fv(n2);			//	and set it
	  glVertex3fv(v2);			//	set the second vertex
	  
	  glNormal3fv(n3);			//	and set it
	  glVertex3fv(v3);			//	set the third vertex
	  glEnd();
	}
      else
	{
	  glBegin(GL_TRIANGLES);
	  glNormal3fv(n1);			//	set normal
	  glVertex3fv(v1);			//	set the first vertex
	  
	  glNormal3fv(n3);			//	and set it
	  glVertex3fv(v3);			//	set the third vertex
	  
	  glNormal3fv(n2);			//	and set it
	  glVertex3fv(v2);			//	set the second vertex
	  glEnd();
	}
      break;
    }
}

//----------------------------------------------------------------------------
//------------------------- public computeGradient() ------------------------------
//----------------------------------------------------------------------------

void vu1512121::computeGradient(int i, int j, int k, float *g, int mode)
{
  switch(mode)
    {
    case 0:
      g[0] = -0.5*(getDataValue(i+1, j  , k  ) - getDataValue(i-1, j  , k  ));
      g[1] = -0.5*(getDataValue(i	, j+1, k  ) - getDataValue(i  , j-1, k  ));
      g[2] = -0.5*(getDataValue(i	, j  , k+2) - getDataValue(i  , j  , k-2));
      break;
    case 1:
      i += (k % 2);
      j += (k % 2);

      g[0] = ((getDataValue(i,j,k-1) - getDataValue(i-1, j, k-1)) +
	(getDataValue(i,j,k+1) - getDataValue(i-1, j, k+1)) +
	(getDataValue(i,j-1,k-1) - getDataValue(i-1, j-1, k-1)) +
	(getDataValue(i,j-1,k+1) - getDataValue(i-1, j-1, k+1)))/(4.0*T);
      g[1] = ((getDataValue(i,j,k-1) - getDataValue(i, j-1, k-1)) +
	(getDataValue(i,j,k+1) - getDataValue(i, j, k+1)) +
	(getDataValue(i-1,j,k-1) - getDataValue(i-1, j-1, k-1)) +
	(getDataValue(i-1,j,k+1) - getDataValue(i-1, j-1, k+1)))/(4.0*T);
      g[2] = ((getDataValue(i,j,k+1) - getDataValue(i, j, k-1)) +
	(getDataValue(i,j-1,k+1) - getDataValue(i, j-1, k-1)) +
	(getDataValue(i-1,j,k+1) - getDataValue(i-1, j, k-1)) +
	(getDataValue(i-1,j-1,k+1) - getDataValue(i-1, j-1, k-1)))/(4.0*T);
      break;
    }
}

//----------------------------------------------------------------------------
//------------------------- public ComputeNormal() ------------------------------
//----------------------------------------------------------------------------

void vu1512121::ComputeNormal(float *p, float *q, float *r, float *n)		//	computes normal for triangle pqr, puts it in n
{
  float pq[3], pr[3];																	//	local variables for edge vectors
  float nNorm;																		//	norm of this vector (I like unit vectors)
  
  for (int i = 0; i < 3; i++)															//	loop to compute pq, pr
    {
      pq[i] = q[i] - p[i];
      pr[i] = r[i] - p[i];		//	just subtract . . .
    } // end of loop to compute vectors 
  
  n[0] = pq[1]*pr[2] - pr[1]*pq[2];													//	compute cross-product
  n[1] = pq[2]*pr[0] - pr[2]*pq[0];
  n[2] = pq[0]*pr[1] - pr[0]*pq[1];

  nNorm = n[0]*n[0] + n[1]*n[1] + n[2]*n[2];											//	compute the norm

  if (nNorm > 0.0)
    {
      nNorm = 1.0 / sqrt(nNorm);												
      
      n[0] *= nNorm;
      n[1] *= nNorm;
      n[2] *= nNorm;										//	and adjust the vector
    }
} // end of ComputeNormal()

//----------------------------------------------------------------------------
//------------------------- public InterpolatePoint() ------------------------------
//----------------------------------------------------------------------------

void vu1512121::InterpolatePoint(float x1, float y1, float z1, float h1, 	//	linear interpolation along an edge
				float x2, float y2, float z2, float h2, 
				float ht, float* result)					
{ // InterpolatePoint()
  float delta, one_delta;				//	parameters of interpolation

  if (h1 == h2)
    delta = 0.0;
  else
    delta = (h1 - ht) / (h1 - h2);	        //	compute parameter from p1 to p2

  one_delta = 1.0 - delta;														//	and inverse parameter

  result[0] = one_delta*x1 + delta*x2;
  result[1] = one_delta*y1 + delta*y2;	//	compute resultant vertex
  result[2] = one_delta*z1 + delta*z2;				
} // InterpolatePoint() 

//----------------------------------------------------------------------------
//------------------------- public initOpenGL() ------------------------------
//----------------------------------------------------------------------------

void vu1512121::initOpenGL(void)
{
}
