#include "splat.h"
#include "vuTFunc/simpleFuncs.h"
#include <fstream.h>
#include <math.h>

//----------------------------------------------------------------------------
//------------------------- Static texture variables -------------------------
//----------------------------------------------------------------------------

static float texcoord0[2] = {0.0f, 0.0f};
static float texcoord1[2] = {0.0f, 1.0f};
static float texcoord2[2] = {1.0f, 1.0f};
static float texcoord3[2] = {1.0f, 0.0f};

//----------------------------------------------------------------------------
//------------------------- The default constructor --------------------------
//----------------------------------------------------------------------------

vu1512111::vu1512111()
{
    m_Normals = 0;

    m_FPSize    = 256;
    m_Footprint = 0;
    m_GLSplat   = 0;

    computeSplat();
}

//----------------------------------------------------------------------------
//------------------------- The copy constructor -----------------------------
//----------------------------------------------------------------------------

vu1512111::vu1512111(const vu1512111& inst) : vu151211(inst)
{
    dword i;

    m_NTable = inst.m_NTable;

#if defined(QUANTIZE_NORMALS)
    m_Normals = new byte[m_DataSize];
#else
    m_Normals = new float[m_DataSize*3];
#endif
    for(i=0;i<m_DataSize;++i)
        m_Normals[i] = inst.m_Normals[i];

    m_FPSize = inst.m_FPSize;
    computeSplat();
}

//----------------------------------------------------------------------------
//------------------------- The destructor -----------------------------------
//----------------------------------------------------------------------------

vu1512111::~vu1512111()
{
    if (m_Normals)
        delete [] m_Normals;
    if (m_Footprint)
    {
        delete [] m_Footprint;
        glDeleteTextures(1, &m_GLSplat);
    }
}

//----------------------------------------------------------------------------
//------------------------- The assignment operator --------------------------
//----------------------------------------------------------------------------

vu1512111& vu1512111::operator=(const vu1512111& rhs)
{
    if (this != &rhs)
    {
        vu151211::operator=(rhs);

        dword i;

        m_NTable = rhs.m_NTable;

        if (m_Normals)
            delete [] m_Normals;

#if defined(QUANTIZE_NORMALS)
        m_Normals = new byte[m_DataSize];
#else
        m_Normals = new float[m_DataSize*3];
#endif
        for(i=0;i<m_DataSize;++i)
            m_Normals[i] = rhs.m_Normals[i];

        m_FPSize = rhs.m_FPSize;

        computeSplat();
    }
    return *this;
}

//----------------------------------------------------------------------------
//------------------------- public setViewVectors() --------------------------
//----------------------------------------------------------------------------

void vu1512111::setViewVectors(const vuVector& view,const vuVector& up,const vuVector& right)
{
    m_View = view;
    m_Shift1 = right*3.2f;
    m_Shift2 = up*3.2f;
    m_Shift0 = (m_Shift1+m_Shift2)*(-0.5f);
}

//----------------------------------------------------------------------------
//------------------------- public setFootprintSize() ------------------------
//----------------------------------------------------------------------------

void vu1512111::setFootprintSize(dword size)
{
    m_FPSize = size;
}

//----------------------------------------------------------------------------
//------------------------- public getFootprintSize() ------------------------
//----------------------------------------------------------------------------

dword vu1512111::getFootprintSize() const
{
    return m_FPSize;
}

//----------------------------------------------------------------------------
//------------------------- public read() ------------------------------------
//----------------------------------------------------------------------------

bool vu1512111::read()
{
    bool success = vu151211::read();
    if (!success) return false;

    //Allocate memory for the normals.
    if (m_Normals != 0) delete [] m_Normals;
#if defined(QUANTIZE_NORMALS)
    m_Normals = new byte[m_DataSize];
#else
    m_Normals = new float[m_DataSize*3];
#endif
    preprocess();
    return true;
}

//----------------------------------------------------------------------------
//------------------------- public readRaw() ---------------------------------
//----------------------------------------------------------------------------

bool vu1512111::readRaw(void)
{
    dword len;
    ifstream in;

#ifdef IS_NOCREATE_NEEDED
    in.open(m_FileName, ios::in|ios::binary|ios::nocreate);
#else
	// The nocreate is not available on the IRIX boxes that we were compiling
	// this code on, so we had to remove this part from
    in.open(m_FileName, ios::in|ios::binary);
#endif

    if (!in.is_open()) return false;

    in.seekg(0, ios::end);
    len = in.tellg();
    in.seekg(0, ios::beg);

    in >> m_Dim1Size >> m_Dim2Size >> m_Dim3Size;
    if (in.fail()) return false;
    m_DataSize = m_Dim1Size*m_Dim2Size*m_Dim3Size;

    m_Data = new byte[m_DataSize];
    in.read((char *) (m_Data), int (m_DataSize));
    if (in.fail()) return false;

    in.close();

    //Allocate memory for the normals.
    if (m_Normals != 0) delete [] m_Normals;
#if defined(QUANTIZE_NORMALS)
    m_Normals = new byte[m_DataSize];
#else
    m_Normals = new float[m_DataSize*3];
#endif
    preprocess();

    return true;
}

//----------------------------------------------------------------------------
//------------------------- private preprocess() -----------------------------
//----------------------------------------------------------------------------

void vu1512111::preprocess(void)
{
    float norm[3];
    float len;
    dword w, wh;
    dword i, j, k, index;

    w = m_Dim1Size;
    wh = m_Dim1Size*m_Dim2Size;

    //
    // Compute all of the normals and create a quantization
    // table with 256 entries.
    //
    index = 0;
#if defined(QUANTIZE_NORMALS)
    m_NTable.initCollection(m_DataSize);
#endif
    for(k=0;k<m_Dim3Size;++k)
    {
        for(j=0;j<m_Dim2Size;++j)
        {
            for(i=0;i<m_Dim1Size;++i)
            {
                if (i == 0)
                    norm[0] = m_Data[index+1]-m_Data[index];
                else if (i == m_Dim1Size-1)
                    norm[0] = m_Data[index]-m_Data[index-1];
                else
                    norm[0] = (m_Data[index+1]-m_Data[index-1])*0.5;

                if (j == 0)
                    norm[1] = m_Data[index+w]-m_Data[index];
                else if (j == m_Dim2Size-1)
                    norm[1] = m_Data[index]-m_Data[index-w];
                else
                    norm[1] = (m_Data[index+w]-m_Data[index-w])*0.5;

                if ((k == 0) || (k == 1))
                    norm[2] = m_Data[index+2*wh]-m_Data[index];
                else if ((k == m_Dim3Size-1) || (k == m_Dim3Size-2))
                    norm[2] = m_Data[index]-m_Data[index-2*wh];
                else
                    norm[2] = (m_Data[index+2*wh]-m_Data[index-2*wh])*0.5;

                len = (float)sqrt((double)((norm[0]*norm[0])+
                                           (norm[1]*norm[1])+
                                           (norm[2]*norm[2])));
                if (len > 0.0f)
                {
                    norm[0] /= len;
                    norm[1] /= len;
                    norm[2] /= len;
                }
#if defined(QUANTIZE_NORMALS)
                m_NTable.addToCollection(norm);
#else
                m_Normals[index*3] = norm[0];
                m_Normals[index*3+1] = norm[1];
                m_Normals[index*3+2] = norm[2];
#endif
                ++index;
            }
        }
    }

#if defined(QUANTIZE_NORMALS)
    m_NTable.computeTable();

    //
    // Create a map from each voxel into the table
    // of quantized normals.
    //
    index = 0;
    for(k=0;k<m_Dim3Size;++k)
    {
        for(j=0;j<m_Dim2Size;++j)
        {
            for(i=0;i<m_Dim1Size;++i)
            {
                if (i == 0)
                    norm[0] = m_Data[index+1]-m_Data[index];
                else if (i == m_Dim1Size-1)
                    norm[0] = m_Data[index]-m_Data[index-1];
                else
                    norm[0] = (m_Data[index+1]-m_Data[index-1])*0.5;

                if (j == 0)
                    norm[1] = m_Data[index+w]-m_Data[index];
                else if (j == m_Dim2Size-1)
                    norm[1] = m_Data[index]-m_Data[index-w];
                else
                    norm[1] = (m_Data[index+w]-m_Data[index-w])*0.5;

                if ((k == 0) || (k == 1))
                    norm[2] = m_Data[index+2*wh]-m_Data[index];
                else if ((k == m_Dim3Size-1) || (k == m_Dim3Size-2))
                    norm[2] = m_Data[index]-m_Data[index-2*wh];
                else
                    norm[2] = (m_Data[index+2*wh]-m_Data[index-2*wh])*0.5;

                len = (float)sqrt((double)((norm[0]*norm[0])+
                                           (norm[1]*norm[1])+
                                           (norm[2]*norm[2])));
                if (len > 0.0f)
                {
                    norm[0] /= len;
                    norm[1] /= len;
                    norm[2] /= len;
                }
                m_Normals[index] = (byte)m_NTable.findNearest(norm);

                ++index;
            }
        }
    }
#endif
}

//----------------------------------------------------------------------------
//------------------------- public render() ----------------------------------
//----------------------------------------------------------------------------

void vu1512111::render(void)
{
    int S_index, S_step, S_size;
    int M_index, M_step, M_size;
    int F_index, F_step, F_size;
    float dp[3];
    float vals[3];
    float *F, *M, *S;
    float dF, dM, dS;
    float rF, rM;
    float *n;
    int   index;
    dword density;

    dp[0] = (m_View[0] > 0.0f)?m_View[0]:m_View[0]*-1.0f;
    dp[1] = (m_View[1] > 0.0f)?m_View[1]:m_View[1]*-1.0f;
    dp[2] = (m_View[2] > 0.0f)?m_View[2]:m_View[2]*-1.0f;

    //
    // Compute indices, ranges, and values so that the rendering
    // loop can be as efficient as possible.
    //
    if (dp[0] > dp[1])
    {
        if (dp[0] > dp[2])
        {
            if (dp[1] > dp[2])
            {
                // 2, 1, 0  (i.e. z=fast, y=medium, x=slow)
                S = &vals[0];
                M = &vals[1];
                F = &vals[2];

                S_size = m_Dim1Size;
                if (m_View[0] >= 0.0f){
                    S_step  = 1;
                    *S = 0.0f;
                    dS = 1.0f;
                    index = 0;
                }else{
                    S_step  = -1;
                    *S = (float)m_Dim1Size-1.0f;
                    dS = -1.0f;
                    index = m_Dim1Size-1;
                }

                M_size = m_Dim2Size;
                if (m_View[1] >= 0.0f){
                    M_step  = m_Dim1Size;
                    *M = 0.0f;
                    dM = 1.0f;
                    rM = -1.0f * (float)m_Dim2Size;
                }else{
                    M_step  = -1*m_Dim1Size;
                    *M = (float)m_Dim2Size-1.0f;
                    dM = -1.0f;
                    rM = (float)m_Dim2Size;
                    index += (1-m_Dim2Size)*M_step;
                }

                F_size = m_Dim3Size;
                if (m_View[2] >= 0.0f){
                    F_step  = m_Dim1Size*m_Dim2Size;
                    *F = 0.0f;
                    dF = 1.0f;
                    rF = -1.0f * (float)m_Dim3Size;
                }else{
                    F_step  = -1*m_Dim1Size*m_Dim2Size;
                    *F = (float)m_Dim3Size-1.0f;
                    dF = -1.0f;
                    rF = (float)m_Dim3Size;
                    index += (1-m_Dim3Size)*F_step;
                }
            }
            else
            {
                // 1, 2, 0  (i.e. y=fast, z=medium, x=slow)
                S = &vals[0];
                F = &vals[1];
                M = &vals[2];

                S_size = m_Dim1Size;
                if (m_View[0] >= 0.0f){
                    S_step  = 1;
                    *S = 0.0f;
                    dS = 1.0f;
                    index = 0;
                }else{
                    S_step  = -1;
                    *S = (float)m_Dim1Size-1.0f;
                    dS = -1.0f;
                    index = m_Dim1Size-1;
                }

                F_size = m_Dim2Size;
                if (m_View[1] >= 0.0f){
                    F_step  = m_Dim1Size;
                    *F = 0.0f;
                    dF = 1.0f;
                    rF = -1.0f * (float)m_Dim2Size;
                }else{
                    F_step  = -1*m_Dim1Size;
                    *F = (float)m_Dim2Size-1.0f;
                    dF = -1.0f;
                    rF = (float)m_Dim2Size;
                    index += (1-m_Dim2Size)*F_step;
                }

                M_size = m_Dim3Size;
                if (m_View[2] >= 0.0f){
                    M_step  = m_Dim1Size*m_Dim2Size;
                    *M = 0.0f;
                    dM = 1.0f;
                    rM = -1.0f * (float)m_Dim3Size;
                }else{
                    M_step  = -1*m_Dim1Size*m_Dim2Size;
                    *M = (float)m_Dim3Size-1.0f;
                    dM = -1.0f;
                    rM = (float)m_Dim3Size;
                    index += (1-m_Dim3Size)*M_step;
                }
            }
        }
        else
        {
            // 1, 0, 2  (i.e. y=fast, x=medium, z=slow)
            M = &vals[0];
            F = &vals[1];
            S = &vals[2];

            M_size = m_Dim1Size;
            if (m_View[0] >= 0.0f){
                M_step  = 1;
                *M = 0.0f;
                dM = 1.0f;
                rM = -1.0f* (float)m_Dim1Size;
                index = 0;
            }else{
                M_step  = -1;
                *M = (float)m_Dim1Size-1.0f;
                dM = -1.0f;
                rM = (float)m_Dim1Size;
                index = m_Dim1Size-1;
            }

            F_size = m_Dim2Size;
            if (m_View[1] >= 0.0f){
                F_step  = m_Dim1Size;
                *F = 0.0f;
                dF = 1.0f;
                rF = -1.0f * (float)m_Dim2Size;
            }else{
                F_step  = -1*m_Dim1Size;
                *F = (float)m_Dim2Size-1.0f;
                dF = -1.0f;
                rF = (float)m_Dim2Size;
                index += (1-m_Dim2Size)*F_step;
            }

            S_size = m_Dim3Size;
            if (m_View[2] >= 0.0f){
                S_step  = m_Dim1Size*m_Dim2Size;
                *S = 0.0f;
                dS = 1.0f;
            }else{
                S_step  = -1*m_Dim1Size*m_Dim2Size;
                *S = (float)m_Dim3Size-1.0f;
                dS = -1.0f;
                index += (1-m_Dim3Size)*S_step;
            }
        }
    }
    else
    {
        if (dp[1] > dp[2])
        {
            if (dp[0] > dp[2])
            {
                // 2, 0, 1  (i.e. z=fast, x=medium, y=slow)
                M = &vals[0];
                S = &vals[1];
                F = &vals[2];

                M_size = m_Dim1Size;
                if (m_View[0] >= 0.0f){
                    M_step  = 1;
                    *M = 0.0f;
                    dM = 1.0f;
                    rM = -1.0f* (float)m_Dim1Size;
                    index = 0;
                }else{
                    M_step  = -1;
                    *M = (float)m_Dim1Size-1.0f;
                    dM = -1.0f;
                    rM = (float)m_Dim1Size;
                    index = m_Dim1Size-1;
                }

                S_size = m_Dim2Size;
                if (m_View[1] >= 0.0f){
                    S_step  = m_Dim1Size;
                    *S = 0.0f;
                    dS = 1.0f;
                }else{
                    S_step  = -1*m_Dim1Size;
                    *S = (float)m_Dim2Size-1.0f;
                    dS = -1.0f;
                    index += (1-m_Dim2Size)*S_step;
                }

                F_size = m_Dim3Size;
                if (m_View[2] >= 0.0f){
                    F_step  = m_Dim1Size*m_Dim2Size;
                    *F = 0.0f;
                    dF = 1.0f;
                    rF = -1.0f * (float)m_Dim3Size;
                }else{
                    F_step  = -1*m_Dim1Size*m_Dim2Size;
                    *F = (float)m_Dim3Size-1.0f;
                    dF = -1.0f;
                    rF = (float)m_Dim3Size;
                    index += (1-m_Dim3Size)*F_step;
                }
            }
            else
            {
                // 0, 2, 1  (i.e. x=fast, z=medium, y=slow)
                F = &vals[0];
                S = &vals[1];
                M = &vals[2];

                F_size = m_Dim1Size;
                if (m_View[0] >= 0.0f){
                    F_step  = 1;
                    *F = 0.0f;
                    dF = 1.0f;
                    rF = -1.0f* (float)m_Dim1Size;
                    index = 0;
                }else{
                    F_step  = -1;
                    *F = (float)m_Dim1Size-1.0f;
                    dF = -1.0f;
                    rF = (float)m_Dim1Size;
                    index = m_Dim1Size-1;
                }

                S_size = m_Dim2Size;
                if (m_View[1] >= 0.0f){
                    S_step  = m_Dim1Size;
                    *S = 0.0f;
                    dS = 1.0f;
                }else{
                    S_step  = -1*m_Dim1Size;
                    *S = (float)m_Dim2Size-1.0f;
                    dS = -1.0f;
                    index += (1-m_Dim2Size)*S_step;
                }

                M_size = m_Dim3Size;
                if (m_View[2] >= 0.0f){
                    M_step  = m_Dim1Size*m_Dim2Size;
                    *M = 0.0f;
                    dM = 1.0f;
                    rM = -1.0f * (float)m_Dim3Size;
                }else{
                    M_step  = -1*m_Dim1Size*m_Dim2Size;
                    *M = (float)m_Dim3Size-1.0f;
                    dM = -1.0f;
                    rM = (float)m_Dim3Size;
                    index += (1-m_Dim3Size)*M_step;
                }
            }
        }
        else
        {
            // 0, 1, 2  (i.e. x=fast, y=medium, z=slow)
            F = &vals[0];
            M = &vals[1];
            S = &vals[2];

            F_size = m_Dim1Size;
            if (m_View[0] >= 0.0f){
                F_step  = 1;
                *F = 0.0f;
                dF = 1.0f;
                rF = -1.0f* (float)m_Dim1Size;
                index = 0;
            }else{
                F_step  = -1;
                *F = (float)m_Dim1Size-1.0f;
                dF = -1.0f;
                rF = (float)m_Dim1Size;
                index = m_Dim1Size-1;
            }

            M_size = m_Dim2Size;
            if (m_View[1] >= 0.0f){
                M_step  = m_Dim1Size;
                *M = 0.0f;
                dM = 1.0f;
                rM = -1.0f * (float)m_Dim2Size;
            }else{
                M_step  = -1*m_Dim1Size;
                *M = (float)m_Dim2Size-1.0f;
                dM = -1.0f;
                rM = (float)m_Dim2Size;
                index += (1-m_Dim2Size)*M_step;;
            }

            S_size = m_Dim3Size;
            if (m_View[2] >= 0.0f){
                S_step  = m_Dim1Size*m_Dim2Size;
                *S = 0.0f;
                dS = 1.0f;
            }else{
                S_step  = -1*m_Dim1Size*m_Dim2Size;
                *S = (float)m_Dim3Size-1.0f;
                dS = -1.0f;
                index += (1-m_Dim3Size)*S_step;
            }
        }
    }

    S_step -= M_step*M_size;
    M_step -= F_step*F_size;

    //
    // The following loop is highly optimized for speed, given
    // the values and indices computed in the above conditionals.
    //

    float vals2[3];

    glBegin(GL_QUADS);
    for(S_index=0; S_index<S_size; ++S_index)
    {
        for(M_index=0; M_index<M_size; ++M_index)
        {
            for(F_index=0; F_index<F_size; ++F_index)
            {
                density = m_Data[index];
                if (m_TFunc[density][3] > 0.0f)
                {
                    glColor4fv(m_TFunc[density]);
#if defined(QUANTIZE_NORMALS)
                    n = m_NTable[m_Normals[index]];
#else
                    n = &m_Normals[index*3];
#endif

#if defined(ONE_SIDED_LIGHTING)
                    if (n[0]*m_View[0] + n[1]*m_View[1] + n[2]*m_View[2] > 0.0f)
                    {
                        n[0] *= -1.0f;
                        n[1] *= -1.0f;
                        n[2] *= -1.0f;
                    }
#endif
                    glNormal3fv(n);

		    vals2[0] =     T*vals[0];
		    vals2[1] =     T*vals[1];
		    vals2[2] = 0.5*T*vals[2];

		      if ((((int)vals[2]) % 2) == 1)
			{
			  vals2[0] += 0.5*T;
			  vals2[1] += 0.5*T;
			}

                    drawSplatOrtho(vals2);
                }

                index += F_step;
                *F += dF;
            }
            *F += rF;
            *M += dM;
            index += M_step;
        }
        *M += rM;
        *S += dS;
        index += S_step;
    }

    glEnd();
}

//----------------------------------------------------------------------------
//------------------------- private computeSplat() ---------------------------
//----------------------------------------------------------------------------

void vu1512111::computeSplat(void)
{
    dword index;
    dword i;
    dword j;
    double x;
    double y;
    double d;

    if (m_Footprint)
    {
        delete [] m_Footprint;
        glDeleteTextures(1, &m_GLSplat);
    }
#if defined(COMBINERS)
    m_Footprint = new GLubyte[m_FPSize*m_FPSize];
#else
    m_Footprint = new GLubyte[m_FPSize*m_FPSize*2];
#endif

    index = 0;
    for(j=0;j<m_FPSize;++j)
    {
        for(i=0;i<m_FPSize;++i)
        {
            x = 0.5 - (double)i/(double)(m_FPSize-1);
            y = 0.5 - (double)j/(double)(m_FPSize-1);
            d = sqrt(x*x+y*y);
#if !defined(COMBINERS)
            m_Footprint[index++] = 255;
#endif
            m_Footprint[index++] = (GLubyte)(255.0*gaussSplat(2.0*d));
        }
    }
}

double vu1512111::gaussSplat(double r)
{
  return 0.466*exp(-2*r*r);
}
//----------------------------------------------------------------------------
//------------------------- private nelsonSplat() ----------------------------
//----------------------------------------------------------------------------

double vu1512111::nelsonSplat(double r)
{
    double s = 0.8893919243498159;
    double t = 1.556227804798694;
    double a = 0.5575267703530060;
    double b = -1.157744128784905;
    double c = 0.6710333014812285;
    double d = 0.067598983313541278;
    double e = 0.2824741750452964;
    double f = t*t*(d+e*t);
    double g = -t*(2*d+3*e*t);
    double h = d+3*e*t;
    double v, u, rs, y, z, p, q;

    if(r == 0)
    {
        v = s*((a-f) + s*((-g)/2.0 + s*((b-h)/3.0 + s*(c+e)/4.0)))
          + t*(f + t*(g/2.0 + t*(h/3.0 + t*(-e)/4.0)));
    }
    else if(r < s)
    {
        rs = r*r;
        y = sqrt(s*s-rs);
        z = sqrt(t*t-rs);
        p = log(y+s);
        q = log(z+t);
        u = log(r);
        v = a*y + b*y*y*y/3.0 + b*y*rs + 0.5*c*rs*(y*s+rs*(p-u))
          + c*(y*s*s*s/4.0 - rs*0.125*(y*s+rs*(p-u)))
          + f*(z-y) + 0.5*(g-e*rs)*(z*t-y*s+rs*(q-p))
          + h*rs*(z-y) + (h/3.0)*(z*z*z-y*y*y)
          - e*(0.25*(z*t*t*t-y*s*s*s) - rs*0.125*(z*t-y*s+rs*(q-p)));
    }
    else if(r < t)
    {
        rs = r*r;
        z = sqrt(t*t-rs);
        q = log(z+t);
        u = log(r);
        v = f*z + 0.5*(g-e*rs)*(z*t+rs*(q-u)) + h*rs*z + (h/3.0)*z*z*z
          - e*(0.25*z*t*t*t-rs*0.125*(z*t+rs*(q-u)));
    }
    else
    {
        v = 0.0;
    }

    return 2.0*v;
}

//----------------------------------------------------------------------------
//------------------------- public initOpenGL() ------------------------------
//----------------------------------------------------------------------------

void vu1512111::initOpenGL(void)
{
    GLfloat shininess = 50.0f;
    GLfloat specular[4] = {1.0f, 1.0f, 1.0f, 1.0f};

#if defined(ONE_SIDED_LIGHTING)
    glMaterialf(GL_FRONT, GL_SHININESS, shininess);
    glMaterialfv(GL_FRONT, GL_SPECULAR, specular);
    glEnable(GL_COLOR_MATERIAL);
    glColorMaterial(GL_FRONT, GL_AMBIENT_AND_DIFFUSE);

    glPolygonMode(GL_FRONT, GL_FILL);
#else
    glMaterialf(GL_FRONT_AND_BACK, GL_SHININESS, shininess);
    glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, specular);
    glEnable(GL_COLOR_MATERIAL);
    glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);

    glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
#endif

    glShadeModel(GL_SMOOTH);
    glDisable(GL_DEPTH_TEST);
    glDisable(GL_ALPHA_TEST);
    glDisable(GL_CULL_FACE);

    glGenTextures(1, &m_GLSplat);
    glBindTexture(GL_TEXTURE_2D, m_GLSplat);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
    glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);

#if defined(COMBINERS)

    gluBuild2DMipmaps(GL_TEXTURE_2D, 1, m_FPSize, m_FPSize,
                      GL_LUMINANCE, GL_UNSIGNED_BYTE,
                      (void*)m_Footprint);
    glEnable(GL_TEXTURE_2D);

	if (!glh_init_extensions("GL_NV_register_combiners "))
        cout << "Could not initialize extensions.\n" << flush;

    glEnable(GL_REGISTER_COMBINERS_NV);
	glCombinerParameteriNV(GL_NUM_GENERAL_COMBINERS_NV, 1);

    // Setup combiner 0 - RGB
    glCombinerInputNV(GL_COMBINER0_NV, GL_RGB, GL_VARIABLE_A_NV, GL_PRIMARY_COLOR_NV,   GL_UNSIGNED_IDENTITY_NV, GL_RGB);
    glCombinerInputNV(GL_COMBINER0_NV, GL_RGB, GL_VARIABLE_B_NV, GL_ZERO,               GL_UNSIGNED_INVERT_NV,   GL_RGB);
    glCombinerInputNV(GL_COMBINER0_NV, GL_RGB, GL_VARIABLE_C_NV, GL_SECONDARY_COLOR_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
    glCombinerInputNV(GL_COMBINER0_NV, GL_RGB, GL_VARIABLE_D_NV, GL_ZERO,               GL_UNSIGNED_INVERT_NV,   GL_RGB);
    glCombinerOutputNV(GL_COMBINER0_NV, GL_RGB, GL_DISCARD_NV, GL_DISCARD_NV, GL_SPARE0_NV, GL_NONE, GL_NONE, GL_FALSE, GL_FALSE, GL_FALSE);

    // Setup combiner 0 - alpha
    glCombinerInputNV(GL_COMBINER0_NV, GL_ALPHA, GL_VARIABLE_A_NV, GL_PRIMARY_COLOR_NV, GL_UNSIGNED_IDENTITY_NV, GL_ALPHA);
    glCombinerInputNV(GL_COMBINER0_NV, GL_ALPHA, GL_VARIABLE_B_NV, GL_TEXTURE0_ARB,     GL_UNSIGNED_IDENTITY_NV, GL_BLUE);
    glCombinerOutputNV(GL_COMBINER0_NV, GL_ALPHA, GL_SPARE0_NV, GL_DISCARD_NV, GL_DISCARD_NV, GL_NONE, GL_NONE, GL_FALSE, GL_FALSE, GL_FALSE);

    // Setup final combiner - rgb
    glFinalCombinerInputNV(GL_VARIABLE_A_NV, GL_ZERO,      GL_UNSIGNED_IDENTITY_NV, GL_RGB);
    glFinalCombinerInputNV(GL_VARIABLE_B_NV, GL_ZERO,      GL_UNSIGNED_IDENTITY_NV, GL_RGB);
    glFinalCombinerInputNV(GL_VARIABLE_C_NV, GL_ZERO,      GL_UNSIGNED_IDENTITY_NV, GL_RGB);
    glFinalCombinerInputNV(GL_VARIABLE_D_NV, GL_SPARE0_NV, GL_UNSIGNED_IDENTITY_NV, GL_RGB);
    glFinalCombinerInputNV(GL_VARIABLE_E_NV, GL_ZERO,      GL_UNSIGNED_IDENTITY_NV, GL_RGB);
    glFinalCombinerInputNV(GL_VARIABLE_F_NV, GL_ZERO,      GL_UNSIGNED_IDENTITY_NV, GL_RGB);

    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

#else

    gluBuild2DMipmaps(GL_TEXTURE_2D, 2, m_FPSize, m_FPSize,
                      GL_LUMINANCE_ALPHA, GL_UNSIGNED_BYTE,
                      (void*)m_Footprint);
    glEnable(GL_TEXTURE_2D);

    glEnable(GL_BLEND);
    glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
#endif
}

//----------------------------------------------------------------------------
//------------------------- private drawSplatOrtho() -------------------------
//----------------------------------------------------------------------------

void vu1512111::drawSplatOrtho(float* v)
{
    vuVector pos(v[0], v[1], v[2]);

    pos += m_Shift0;
    glTexCoord2fv(texcoord0);
    glVertex3fv(pos.getData());

    pos += m_Shift1;
    glTexCoord2fv(texcoord1);
    glVertex3fv(pos.getData());

    pos += m_Shift2;
    glTexCoord2fv(texcoord2);
    glVertex3fv(pos.getData());

    pos -= m_Shift1;
    glTexCoord2fv(texcoord3);
    glVertex3fv(pos.getData());
}

//----------------------------------------------------------------------------
//------------------------- private drawSplatPerspective() -------------------
//----------------------------------------------------------------------------

void vu1512111::drawSplatPerspective(float* v)
{
}
