#include <stdio.h>
#include <fstream.h>
#include <math.h>

#include "SheetSplat.h"

#include "vuColour7a.h"
#include "vuColour31a.h"
#include "vuColourXYZa.h"
#include "vuColourRGBa.h"
#include "vuColourSpec.h"

#define SHSP_SPLAT_CROP_RATIO	0.0		// ratio of slice width below which splat is ignored
#define SHSP_SPLATSTDSCALE	1		// scaling for the standard deviation of the splat is scaled
						// the integral is not normalized so values != 1 suck...
#define SHSP_SPLATRADIUS	2.3		// changed from 1.6
						// 1.6 is not enough for BCC grids, a larger texture up
						// to 2.2 hugely increases quality.
						// 2.6 finally makes artifacts disappear.
#define SHSP_SHEETWIDTH		1

/* notes
   least distance on grid with T=sqrt(2) is 1.2247449 (=sqrt(3/2))
   that times 1.6 is 1.9595918
   sqrt(2) * 1.6 is 2.2627417
*/

//----------------------------------------------------------------------------
/** calculates b to the power of e, with e being a positive integral value
  This takes only log e multiplications
*/
float pow(float b, unsigned int e)
{
  float r = 1;
  while(e) {
    if(e&1) r*=b;
    b*=b;
    e=e>>1;
  }

  vuColourXYZa xyz;
  xyz.from(vuColourRGBa(1));
  return r;
}

//----------------------------------------------------------------------------
//------------------------- The default constructor --------------------------
//----------------------------------------------------------------------------

vu1512112::vu1512112()
{
  //choose size of splat depending on size of framebuffer
  //(to avoid resizing)
  m_DataPrepared = false;
  m_SplatPos = NULL;
  m_Slices = NULL;
  m_SliceDist = NULL;
  setSliceWidth(SHSP_SHEETWIDTH);
  setFootprintSize(SHSP_SPLATRADIUS);
  m_Normals = NULL;
  m_Shading = NULL;
  m_LightDir = vuVector(-1,0,0);
  m_LightDiffuse = 0.9;
  m_LightAmbient = 0.1;
  float scol[] = {1,1,1,0};
  m_LightSpecular = vuColourRGBa(scol);
  m_LightShininess = 20;
  setBGColour(0,0,0);
  setImageSize(200,200);

  m_Camera = new vuParallelCamera;
}

//----------------------------------------------------------------------------
//------------------------- The destructor -----------------------------------
//----------------------------------------------------------------------------

vu1512112::~vu1512112()
{
  clearSheets();
  if (m_Camera != NULL) {
    delete m_Camera;
    m_Camera = NULL;
  }
}

//----------------------------------------------------------------------------
//------------------------- The assignment operator --------------------------
//----------------------------------------------------------------------------

vu1512112& vu1512112::operator=(const vu1512112& rhs)
{
    if (this != &rhs)
    {
        vu151211::operator=(rhs);
    }
    return *this;
}

//----------------------------------------------------------------------------
//------------------------- public getFootprintSize() ------------------------
//----------------------------------------------------------------------------

float vu1512112::getFootprintSize() const
{
    return m_SplatRadius;
}

//----------------------------------------------------------------------------
//------------------------- public setFootprintSize() ------------------------
//----------------------------------------------------------------------------

void vu1512112::setFootprintSize(float size)
{
  m_SplatRadius = size;
  m_SpS_2 = int(m_SplatRadius/m_SliceWidth)+1;
}

//----------------------------------------------------------------------------

float vu1512112::getSliceWidth() const
{
    return m_SliceWidth;
}

//----------------------------------------------------------------------------

void vu1512112::setSliceWidth(const float size)
{
  if(size<0.05) m_SliceWidth = 0.05;
  else m_SliceWidth = size;
  m_SpS_2 = int(m_SplatRadius/m_SliceWidth)+1;
  m_SplatCrop = m_SliceWidth*SHSP_SPLAT_CROP_RATIO;
}

//----------------------------------------------------------------------------
//------------------------- public setViewVectors() --------------------------
//----------------------------------------------------------------------------

void vu1512112::setViewVectors(const vuVector& view,const vuVector& up,const vuVector& right)
{
  //we own the camera, so this call is not really important... (?)
}

//----------------------------------------------------------------------------
//------------------------- public read() ------------------------------------
//----------------------------------------------------------------------------

bool vu1512112::read()
{
    bool success = vu151211::read();
    if (!success) return false;

    m_Center = getVoxelPosition(m_Dim1Size, m_Dim2Size, m_Dim3Size);
    m_Center *= 0.5;
    m_Camera->setPosition(m_Center);
    m_Camera->translateXYZ(0,0,-100);
    //m_Camera->setLookAtVector(m_Center-m_Camera->getPosition());
    cout<<"cpos";
    m_Camera->getPosition().print();

    //Allocate memory for the normals.
    if (m_Normals != 0) delete [] m_Normals;
    m_Normals = new vuVector[m_DataSize];
    if (m_Shading != 0) delete [] m_Shading;
    m_Shading = new float[m_DataSize*2];
    //Allocate memory for projected splat information
    if (m_SplatPos != 0) delete m_SplatPos;
    m_SplatPos = new float[m_DataSize*3];
    //showHistogram();

    preprocess();
    return true;
}

void vu1512112::showHistogram()
{
  dword hist[256];
  dword i;
  for(i=0;i<256;i++)
    hist[i]=0;
  byte *dat = m_Data;
  for(i=0;i<m_DataSize;i++,dat++)
    hist[*dat]++;
  for(i=0;i<256;i++)
    printf("[%i]=%i ",int(i),int(hist[i]));
  putchar('n');
}

//----------------------------------------------------------------------------
//------------------------- public readRaw() ---------------------------------
//----------------------------------------------------------------------------

bool vu1512112::readRaw(void)
{
    dword len;
    ifstream in;


#ifdef IS_NOCREATE_NEEDED
    in.open(m_FileName, ios::in|ios::binary|ios::nocreate);
#else
	// The nocreate is not available on the IRIX boxes that we were compiling
	// this code on, so we had to remove this part from
    in.open(m_FileName, ios::in|ios::binary);
#endif
    if (!in.is_open()) return false;

    in.seekg(0, ios::end);
    len = in.tellg();
    in.seekg(0, ios::beg);

    in >> m_Dim1Size >> m_Dim2Size >> m_Dim3Size;
    if (in.fail()) return false;
    m_DataSize = m_Dim1Size*m_Dim2Size*m_Dim3Size;

    m_Data = new byte[m_DataSize];
    in.read((char *)m_Data, (int)m_DataSize);
    if (in.fail()) return false;

    in.close();

    m_Center = getVoxelPosition(m_Dim1Size, m_Dim2Size, m_Dim3Size);
    m_Center *= 0.5;
    m_Camera->setPosition(m_Center);
    m_Camera->translateXYZ(0,0,-100);
    //m_Camera->setLookAtVector(m_Center-m_Camera->getPosition());
    cout<<"cpos";
    m_Camera->getPosition().print();

    //Allocate memory for the normals.
    if (m_Normals != 0) delete [] m_Normals;
    m_Normals = new vuVector[m_DataSize];
    if (m_Shading != 0) delete [] m_Shading;
    m_Shading = new float[m_DataSize*2];
    //Allocate memory for projected splat information
    if (m_SplatPos != 0) delete m_SplatPos;
    m_SplatPos = new float[m_DataSize*3];
    //showHistogram();
    preprocess();

    return true;
}

//----------------------------------------------------------------------------
//------------------------- private preprocess() -----------------------------
//----------------------------------------------------------------------------

void vu1512112::preprocess(void)
{
  
    // calculate and store normals
    dword w, wh;
    dword i, j, k, index;

    w = m_Dim1Size;
    wh = m_Dim1Size*m_Dim2Size;

    //
    // Compute all of the normals and create a quantization
    // table with 256 entries.
    //
    index = 0;
    for(k=0;k<m_Dim3Size;++k)
    {
        for(j=0;j<m_Dim2Size;++j)
        {
            for(i=0;i<m_Dim1Size;++i)
            {
                if (i == 0)
                    m_Normals[index][0] = m_Data[index+1]-m_Data[index];
                else if (i == m_Dim1Size-1)
                    m_Normals[index][0] = m_Data[index]-m_Data[index-1];
                else
                    m_Normals[index][0] = (m_Data[index+1]-m_Data[index-1])*0.5;

                if (j == 0)
                    m_Normals[index][1] = m_Data[index+w]-m_Data[index];
                else if (j == m_Dim2Size-1)
                    m_Normals[index][1] = m_Data[index]-m_Data[index-w];
                else
                    m_Normals[index][1] = (m_Data[index+w]-m_Data[index-w])*0.5;

                if ((k == 0) || (k == 1))
                    m_Normals[index][2] = m_Data[index+2*wh]-m_Data[index];
                else if ((k == m_Dim3Size-1) || (k == m_Dim3Size-2))
                    m_Normals[index][2] = m_Data[index]-m_Data[index-2*wh];
                else
                    m_Normals[index][2] = (m_Data[index+2*wh]-m_Data[index-2*wh])*0.5;

		m_Normals[index].makeUnit();

                ++index;
            }
        }
    }
    
    m_DataPrepared = true;
}


void vu1512112::setBGColour(float r, float g, float b)
{
  m_BGColour[0] = r;
  m_BGColour[1] = g;
  m_BGColour[2] = b;
  m_BGColour[3] = 0;
}

void vu1512112::run(int whatsup, void* data)
{
  drawSlices();
}

//----------------------------------------------------------------------------
//------------------------- public render() ----------------------------------
//----------------------------------------------------------------------------

void vu1512112::drawSlices()
{
  float black[] = {0,0,0,0};
  RGBABuffer sheetBuffer;
  sheetBuffer.setSize(m_FBWidth, m_FBHeight);
  int cs;
  while((cs=m_CurrentSheet++)<m_NSlices) {
    sheetBuffer.clear(vuColourRGBa(black));
    drawSlice(sheetBuffer, cs);
    m_FBMutex.lock();
    m_FrameBuffer.blendUnder(sheetBuffer);
    m_FBMutex.unlock();
    cout<<'.'<<flush;
  }
}

void vu1512112::render(void)
{
  if(!m_Data || !m_DataPrepared) return;

  int i,j;

  if(m_Refresh) {

    // build preintegrated splat lut
    int pixsize = int(ceil(m_Camera->getWidth()/((vuParallelCamera *)m_Camera)->getXRange()*2*m_SplatRadius));
    int piysize = int(ceil(m_Camera->getHeight()/((vuParallelCamera *)m_Camera)->getYRange()*2*m_SplatRadius));
    m_Splat.buildSplat(m_SplatRadius/SHSP_SPLATSTDSCALE,pixsize, piysize,128);

    // decide which parts of the transfer function to exclude from splatting
    for(i=0; i<256; i++) {
      m_Exclude[i] = m_TFunc[i][3] < 0.001;
    }
    
    cout<<"sorting..."<<endl;
    sortByDistance();
    
    cout<<"drawing"<<flush;

    m_FrameBuffer.clear(m_BGColour);
    m_CurrentSheet = 0;

    startThread(0);
    drawSlices();

    cout<<endl;

    /*
    for(i=m_NSlices-1;i>=0;i--) {
      m_SheetBuffer.clear(black);
      drawSlice(i);
      m_FrameBuffer.blendOver(m_SheetBuffer);
      cout<<'.'<<flush;
    }
    */

    m_Refresh = false;
  }

  fbtype *fbuf = m_FrameBuffer.getData();
  byte *sbuf = new byte[3*m_FBHeight*m_FBWidth];
  byte *screen = sbuf;
  if(!sbuf) return;
  for(i=0;i<m_FBHeight;i++)
    for(j=0;j<m_FBWidth;j++) {
#ifdef FB_FLOAT
      if(*fbuf>1) *fbuf = 1;
      else if(*fbuf<0) *fbuf = 0;
      *(screen++) = int(*(fbuf++)*255);
      if(*fbuf>1) *fbuf = 1;
      else if(*fbuf<0) *fbuf = 0;
      *(screen++) = int(*(fbuf++)*255);
      if(*fbuf>1) *fbuf = 1;
      else if(*fbuf<0) *fbuf = 0;
      *(screen++) = int(*(fbuf++)*255);
#else
      *(screen++) = (*(fbuf++))>>8;		//FB_WORD
      *(screen++) = (*(fbuf++))>>8;
      *(screen++) = (*(fbuf++))>>8;
#endif
      fbuf++;
    }
  glRasterPos2f(0,0);
  glPixelStorei(GL_UNPACK_ALIGNMENT,1);
  glDrawPixels(m_FBWidth,m_FBHeight,GL_RGB,GL_UNSIGNED_BYTE,sbuf);
  delete sbuf;
}


bool vu1512112::setImageSize(int width, int height)
{
  if(!m_FrameBuffer.setSize(width,height)) return false;
  m_FrameBuffer.clear(m_BGColour);
  m_Camera->setWidth(width);
  m_Camera->setHeight(height);
  m_FBWidth = width;
  m_FBHeight = height;
  //doRefresh();
  return true;
}

//----------------------------------------------------------------------------
//------------------------- public initOpenGL() ------------------------------
//----------------------------------------------------------------------------

void vu1512112::initOpenGL(void)
{
}

void vu1512112::initSheets(float dist0, float ddist)
{
  if(m_Slices != 0) clearSheets();
  m_Slices = new dword*[m_NSlices];
  m_SliceDist = new float[m_NSlices];
  if(!m_Slices) return;
  int i;
  for(i=0;i<m_NSlices;i++) {
    m_Slices[i] = new dword[1];
    m_Slices[i][0] = 0;
    m_SliceDist[i] = dist0+i*ddist;
  }
}

void vu1512112::prepareSheets()
{
  if(m_Slices == 0) return;
  int i;
  for(i=0;i<m_NSlices;i++) {
    int n = m_Slices[i][0]+1;
    delete [] m_Slices[i];
    m_Slices[i] = new dword[n];
    m_Slices[i][0] = 1;		// first entry is reserved for number of contents
  }
}

void vu1512112::clearSheets()
{
  if(m_Slices == 0) return;
  int i;
  for(i=0;i<m_NSlices;i++)
    delete [] m_Slices[i];
  delete [] m_Slices;
  delete [] m_SliceDist;
  m_SliceDist = NULL;
  m_Slices = NULL;
}

void vu1512112::sortByDistance()
{
  if(m_Dim1Size<2) return;

  m_LightDir = m_Center;
  m_LightDir -= m_Camera->getPosition();
  m_LightDir.makeUnit();

  clearSheets();

  // figure out number of slices
  // find nearest corner
  vuVector ldir = m_Camera->getPosition();
  ldir -= m_Center;

  int nx,ny,nz;
  nx = (ldir[0]>0);
  ny = (ldir[1]>0);
  nz = (ldir[2]>0);

  int splatslices = int(m_SplatRadius*2/m_SliceWidth)+1;
  //if((splatslices*m_SliceWidth)-(m_SplatRadius*2) > 0.1) splatslices++;
  float nearest = ((vuParallelCamera *)m_Camera)->getDistance(getVoxelPosition(m_Dim1Size*nx,
							  m_Dim2Size*ny,
							  m_Dim3Size*nz));
  float farest = ((vuParallelCamera *)m_Camera)->getDistance(getVoxelPosition(m_Dim1Size* (!nx),
						    m_Dim2Size* (!ny),
						    m_Dim3Size* (!nz)));
  nearest -= (splatslices*m_SliceWidth)/2;
  farest += (splatslices*m_SliceWidth)/2;
  m_NSlices = (int)ceil((farest-nearest)/m_SliceWidth) + 2;
  farest = nearest+ m_SliceWidth*m_NSlices;
  printf("near = %4.4f far = %4.4f nslices = %i\n", nearest, farest, m_NSlices);

  initSheets(nearest,m_SliceWidth);

  dword i,j,k;
  int index=0, index3=0;		// set to first voxel
  float sd;
  dword sind;

  for(k=0;k<m_Dim3Size;k++) {
    for(j=0;j<m_Dim2Size;j++) {
      index = 0+(j*m_Dim1Size)+(k*m_Dim1Size*m_Dim2Size);
      index3 = index*3;
      vuVector p0 = getVoxelPosition(0,j,k);
      vuVector pr0=  p0;
      ((vuParallelCamera *)m_Camera)->project(pr0);
      memcpy(&m_SplatPos[index3],pr0.getData(),sizeof(float)*3);
      if(!m_Exclude[m_Data[index]]) {
	sd = m_SplatPos[index3+2]-nearest;
	sind = int(sd/m_SliceWidth);
	dword lslice = sind+m_SpS_2+1;	//assign splat to the near slices
	for(sind = sind-m_SpS_2;sind<lslice;sind++)
	  m_Slices[sind][0]++;
	m_Shading[index<<1] = m_LightDir.dot(m_Normals[index]);
      }

      index = 1+(j*m_Dim1Size)+(k*m_Dim1Size*m_Dim2Size);
      index3 = index*3;
      vuVector p1 = getVoxelPosition(1,j,k);
      vuVector pr1= p1;
      ((vuParallelCamera *)m_Camera)->project(pr1);
      memcpy(&m_SplatPos[index3],pr1.getData(),sizeof(float)*3);
      if(!m_Exclude[m_Data[index]]) {
	sd = m_SplatPos[index3+2]-nearest;
	sind = int(sd/m_SliceWidth);

	dword lslice = sind+m_SpS_2+1;	//assign splat to the near slices
	for(sind = sind-m_SpS_2;sind<lslice;sind++)
	  m_Slices[sind][0]++;

	m_Shading[index<<1] = m_LightDir.dot(m_Normals[index]);
      }

      p0 = p1-p0;	// p0 is now step from one position to the next
      pr0 = pr1-pr0;	// pr0 is now step of xy projection and distance

      for(i=2;i<m_Dim1Size;i++) {
	index = i+(j*m_Dim1Size)+(k*m_Dim1Size*m_Dim2Size);
	index3 = index*3;
	p1 += p0;
	pr1 += pr0;
	
	if(!m_Exclude[m_Data[index]]) {
	  memcpy(&m_SplatPos[index3],pr1.getData(),sizeof(float)*3);
	  sd = m_SplatPos[index3+2]-nearest;
	  //#if(SHSP_SHEETWIDTH == 1)
	  //sind = int(sd);
	  //#else
	  sind = int(sd/m_SliceWidth);
	  //#endif
	  dword lslice = sind+m_SpS_2+1;	//assign splat to the near slices
	  for(sind = sind-m_SpS_2;sind<lslice;sind++)
	    m_Slices[sind][0]++;
	  
	  // do shading calculation
	  vuVector& n = m_Normals[index];
	  float d = m_LightDir.dot(n);			// diffuse
	  m_Shading[index<<1] = (d > 0 ? d : 0)*m_LightDiffuse + m_LightAmbient;
	  vuVector r = p1-m_Camera->getPosition();	// ray from eye to point
	  r.makeUnit();
	  r -= ((2*r.dot(n))*n);				// reflect
	  float s = m_LightDir.dot(r);		// has negative sign
	  s = s<0 ? -s : 0;			// cut?
	  m_Shading[(index<<1)+1] = pow(s,m_LightShininess); //shininess
	}
      }
    }
  }

  //int s;
  //for(s=0;s<m_NSlices;s++)
  //  cout<<s<<":"<<m_Slices[s][0]<<"  ";
  //cout<<endl;

  prepareSheets();

  index = index3 = 0;
  for(k=0;k<m_Dim3Size;k++) {
    for(j=0;j<m_Dim2Size;j++) {
      for(i=0;
	  i<m_Dim1Size;
	  i++) {
	index = i+(j*m_Dim1Size)+(k*m_Dim1Size*m_Dim2Size);
	index3 = index*3;

	if(!m_Exclude[m_Data[index]]) {
	  sd = m_SplatPos[index3+2]-nearest;
	  //#if(SHSP_SHEETWIDTH == 1)
	  //sind = int(sd);
	  //#else
	  sind = int(sd/m_SliceWidth);
	  //#endif
	  if(sind>0 && sind<(dword)m_NSlices-1) {
	    dword lslice = sind+m_SpS_2+1;	//assign splat to the near slices
	    for(sind = sind-m_SpS_2;sind<lslice;sind++)
	      m_Slices[sind][m_Slices[sind][0]++] = index;
	  }
	}
      }
    }
  }
}

void vu1512112::drawSlice(RGBABuffer& sheet, int n)
{
  dword *splat = &m_Slices[n][1];
  byte density;
  int w2 = int (m_Camera->getWidth()/2);
  int h2 = int (m_Camera->getHeight()/2);
  float sdist = m_SliceDist[n];
  for(int SC = m_Slices[n][0]-1; SC!=0; SC--, splat++) {

    dword index = *splat;
    dword index3 = index*3;
    density = m_Data[index];
    float t0=sdist-m_SplatPos[index3+2];
    float t1=t0+m_SliceWidth;

    vuColourRGBa col(m_TFunc[density]);
    col*=m_Shading[index<<1];
    vuColourRGBa spec = m_LightSpecular;
    ////spec.multAlpha();
    ////spec.setAlpha(0);
    spec*=m_Shading[(index<<1)+1];
    col += spec;
    col.clampTo01();

    sheet.addColourWithM1subM2(m_Splat.getSpla(t1), m_Splat.getSpla(t0),
			       col,
			       (int) m_SplatPos[index3]+w2, 
			       (int) (h2-m_SplatPos[index3+1]));	//flip y-coord
  }
}

