#ifndef _FRAMEBUFFER_H_
#define _FRAMEBUFFER_H_

#include <stdio.h>
#include <string.h>

#include "vuColour.h"


#define	FB_FLOAT

//namespace vuVolumeBCCUnimodal3d1BIntensitySheetSplat {

typedef float fbtype;

/** Frame buffer class with basic functionality.
    Based on floats, provides RGBA channels.
    The buffer is based on floats.
*/

template <int N>
class FrameBuffer
{
  friend class vuColour<N>;
public:

    /** default constructor */
    FrameBuffer() {m_Buffer = 0; }

    /** constructor */
    FrameBuffer(int width, int height)
    {
        m_Buffer = 0;
        setSize(width, height);
    }

	/** destructor */
    ~FrameBuffer()
    {
        freeBuffer();
    }

    	/** returns the number of channels */
    int getNumberOfChannels() { return N; }
    	/** returns the height of the frame buffer */
    int getHeight() { return m_Height; }
    	/** returns the width of the frame buffer */
    int getWidth() { return m_Width; }
    	/** returns a ptr to the buffer data */
    float* getData() { return m_Buffer; }

    	/** this will clear the buffer and reset the size of the buffer */
    bool setSize(int width, int height)
    {
        freeBuffer();
        m_Buffer = new float[ width * height * N];
        if(!m_Buffer) return false;
        m_Width = width;
        m_Height = height;
        return true;
    }

	/** this will clear the buffer and set the size of the buffer to 0, 0 */
    bool freeBuffer()
    {
        if(m_Buffer) delete [] m_Buffer;
        m_Buffer = 0;
        m_Width = 0;
        m_Height = 0;
        return true;
    }

    /** copy into current frame buffer
        number of channel has to be equal
        \param fb frame buffer to take data from
        \param x,y position in current buffer to operate on
        */
    bool copy(FrameBuffer<N>& fb, int x, int y)
    {
        if ((!fb.m_Buffer) ||
            (N != fb.getNumberOfChannels()) ||
            (x < 0 || y < 0) ||
            (x + fb.m_Width>m_Width)||
            (y + fb.m_Height>m_Height)) return false;

        if(fb.getNumberOfChannels() == N)
        {
            int h = m_Height;
            int incdst = m_Width * N;
            int incsrc = fb.m_Width * N;
            float* src = fb.m_Buffer;
            float* dst = m_Buffer[y*m_Width + x];
            for(h=m_Height; h!=0; h--)
            {
                memcpy(dst,src,m_Width);
                dst += incdst;
                src += incsrc;
            }
        }
        else
        {
            // don't know
        }
        return true;
    }

    /** add intensities into buffer
        \param fb frame buffer to take data from
        \param x,y position in current buffer to operate on
        */
    bool add(FrameBuffer<N>& fb, int x, int y)
    {
      if ((!fb.m_Buffer) ||
	  (x < 0 || y < 0) ||
	  (x + fb.m_Width>m_Width)||
	  (y + fb.m_Height>m_Height)) return false;

      int h = m_Height;
      int incdst = (m_Width-fb.m_Width) * N;
      register float* src = fb.m_Buffer;
      register float* dst = &m_Buffer[y*m_Width + x];
      for(h=m_Height; h!=0; h--)
	{
	  for(int len = fb.m_Width * N;len!=0;len--)
	    *(dst++) += *(src++);
	  dst += incdst;
	}
        return true;
    }


    /** add intensities into buffer
        \param fb frame buffer to take data from
        \param x,y position in current buffer to operate on
        */
    bool addColourWithMask(FrameBuffer<1>& fb, vuColour<N>& colour, int x, int y)
    {
        if ((!fb.getData()) ||
            (x < 0 || y < 0) ||
            (x + fb.getWidth()>m_Width)||
            (y + fb.getHeight()>m_Height)) return false;

        int cpyh = fb.getHeight();
        int cpyw = fb.getWidth();
        int incdst = (m_Width-fb.getWidth()) * N;
        register float* src = fb.getData();
        register float* dst = &m_Buffer[(y*m_Width + x)*N];
        for(int h=cpyh; h!=0; h--) {
          for(register int len = cpyw; len!=0; len--) {
	    register float *colval = colour.getData();
	    register float alpha = (*src)*colour[3];
            for(register int comp = N-1; comp!=0; comp--, colval++) {
              *(dst++) += alpha*(*colval);		//add weighted colour
            }
            *(dst++) += *(src++)*(*colval);			//add alpha
          }
          dst += incdst;
        }
        return true;
    }

    /** subtract intensities from current frame buffer
        \param fb frame buffer to take data from
        \param x,y position in current buffer to operate on
        */
    bool subColourWithMask(FrameBuffer<1>& fb, vuColour<N>& colour, int x, int y)
    {
        if ((!fb.getData()) ||
            (x < 0 || y < 0) ||
            (x + fb.getWidth()>m_Width)||
            (y + fb.getHeight()>m_Height)) return false;

        int cpyh = fb.getHeight();
        int cpyw = fb.getWidth();
        int incdst = (m_Width-fb.getWidth()) * N;
        register float* src = fb.getData();
        register float* dst = &m_Buffer[(y*m_Width + x)*N];
        for(int h=cpyh; h!=0; h--) {
          for(register int len = cpyw; len!=0; len--) {
	    register float *colval = colour;
            for(register int comp = N-1; comp!=0; comp--, colval++) {
              *(dst++) -= (*src)*(*colval);		//sub weighted colour
            }
            *(dst++) -= *(src++)*(*colval);			//sub alpha
          }
          dst += incdst;
        }
        return true;
    }

    /** add intensities into buffer using the difference of the two alpha masks
        \param fb frame buffer to take data from
        \param x,y position in current buffer to operate on
        */
    bool addColourWithM1subM2(FrameBuffer<1>& m1, FrameBuffer<1>& m2,
			      vuColour<N>& colour, const int x, const int y)
    {
        if ((!m1.getData()) ||
	    (!m2.getData()) ||
	    ( m1.getWidth()!=m2.getWidth()) ||
	    ( m1.getHeight()!=m2.getHeight()) ||
            (x < 0 || y < 0) ||
            (x + m1.getWidth()>m_Width)||
            (y + m1.getHeight()>m_Height)) return false;

        int cpyh = m1.getHeight();
        int cpyw = m1.getWidth();
        int incdst = (m_Width-m1.getWidth()) * N;
	if (N != 4) {
	  register float* datm1 = m1.getData();
	  register float* datm2 = m2.getData();
	  register float* dst = &m_Buffer[(y*m_Width + x)*N];
	  float *col = colour.getData();
	  for(int h=cpyh; h!=0; h--) {
	    for(register int len = cpyw; len!=0; len--) {
	      float a = ((*datm1)-(*datm2))*col[N-1];
	      for(int comp = 0; comp<N-1; comp++) {
		*(dst++) += a*col[comp];		//add weighted colour
	      }
	      *(dst++) += a;			//add alpha
	      if(*(dst-1)>1) {
		//  cout<<"ac"<<flush;
		*(dst-1) = .99;
	      }
	      datm1++; datm2++;
	    }
	    dst += incdst;
	  }
	} else {	// N==4
	  register float* datm1 = m1.getData();
	  register float* datm2 = m2.getData();
	  register float* dst = &m_Buffer[(y*m_Width + x)*N];
	  float *col = colour.getData();
	  for(int h=cpyh; h!=0; h--) {
	    for(register int len = cpyw; len!=0; len--) {
	      register float a = ((*datm1)-(*datm2))*col[N-1];
	      register float *comp = col;
	      *(dst++) += a*(*comp++);		//add weighted colour R
	      *(dst++) += a*(*comp++);		// G
	      *(dst++) += a*(*comp++);		// B
	      *(dst++) += a;			//add alpha
	      datm1++; datm2++;
	    }
	    dst += incdst;
	  }
	}
        return true;
    }

    /** blends other buffer over current */
    bool blendOver(FrameBuffer<N>& fb, int x, int y)
    {
        if ((!fb.getData()) ||
            (x < 0 || y < 0) ||
            (x + fb.getWidth()>m_Width)||
            (y + fb.getHeight()>m_Height)) return false;

        int cpyh = fb.getHeight();
        int cpyw = fb.getWidth();
        int incdst = (m_Width-fb.getWidth()) * N;
        register float* src = fb.getData();
        register float* dst = &m_Buffer[(y*m_Width + x)*N];
        for(int h=cpyh; h!=0; h--) {
          for(register int len = cpyw; len!=0; len--) {
	    //float a1 = src[N-1];
	    float a2 = 1-src[N-1];
            for(register int comp = 0;
		comp<N-1;
		comp++, dst++, src++) {
              *(dst) = (*dst)*a2 + (*src);
            }
            dst++;	// do nothing with the alpha value. (?)
	    src++;
          }
          dst += incdst;
        }
        return true;
    }

    /** blends other buffer over current */
    bool blendOver(FrameBuffer<N>& fb)
    {
        if ((!fb.getData()) ||
            (fb.getWidth()!=m_Width)||
            (fb.getHeight()!=m_Height)) return false;
	if(N!=4) {
	  int cpylen = fb.getHeight()*fb.getWidth();
	  register float* src = fb.getData();
	  register float* dst = m_Buffer;
	  for(register int len = cpylen; len!=0; len--) {
	    float a2 = 1-src[N-1];
            for(register int comp = 0;
		comp<N-1;
		comp++, dst++, src++) {
              *(dst) = (*dst)*a2 + (*src);
            }
            dst++;	// do nothing with the alpha value.
	    src++;
	  }
	} else {
	  int cpylen = fb.getHeight()*fb.getWidth();
	  register float* src = fb.getData();
	  register float* dst = m_Buffer;
	  for(register int len = cpylen; len!=0; len--) {
	    register float a2 = 1-src[N-1];	//1-x

	    *(dst++) = (*dst)*a2 + (*src++);	//RGB...
	    *(dst++) = (*dst)*a2 + (*src++);
	    *(dst++) = (*dst)*a2 + (*src++);

	    dst++;	// do nothing with the alpha value.
	    src++;
	  }
	}
	return true;
    }

    /** blends other buffer over current
	alpha values in current buffer are used as transmittance
	and are modified accordingly by blending in new images */
    bool blendUnder(FrameBuffer<N>& fb, int x, int y)
    {
        if ((!fb.getData()) ||
            (x < 0 || y < 0) ||
            (x + fb.getWidth()>m_Width)||
            (y + fb.getHeight()>m_Height)) return false;

        int cpyh = fb.getHeight();
        int cpyw = fb.getWidth();
        int incdst = (m_Width-fb.getWidth()) * N;
        register float* src = fb.getData();
        register float* dst = &m_Buffer[(y*m_Width + x)*N];
        for(int h=cpyh; h!=0; h--) {
          for(register int len = cpyw; len!=0; len--) {
	    float a = src[N-1]*dst[N-1];
	    float t = 1-dst[N-1];
            for(register int comp = 0;
		comp<N-1;
		comp++, dst++, src++) {
              *(dst) = *(dst)*t + (*src)*a;
            }
            *(dst++) *= 1-(*(src++));	// modify transmittance of layer
          }
          dst += incdst;
        }
        return true;
    }

    /** blends other buffer under current */
    bool blendUnder(FrameBuffer<N>& fb)
    {
        if ((!fb.getData()) ||
            (fb.getWidth()!=m_Width)||
            (fb.getHeight()!=m_Height)) return false;
	if(N!=4) {
	  int cpylen = fb.getHeight()*fb.getWidth();
	  register float* src = fb.getData();
	  register float* dst = m_Buffer;
	  for(register int len = cpylen; len!=0; len--) {
	    float a2 = 1-src[N-1];
            for(register int comp = 0;
		comp<N-1;
		comp++, dst++, src++) {
              *(dst) = (*dst)*a2 + (*src);
            }
            dst++;	// do nothing with the alpha value.
	    src++;
	  }
	} else {	// N==4
	  int cpylen = fb.getHeight()*fb.getWidth();
	  register float* src = fb.getData();
	  register float* dst = m_Buffer;
	  for(register int len = cpylen; len!=0; len--) {
	    register float t = (1-dst[N-1]);

	    *(dst++) += *(src++)*t;
	    *(dst++) += *(src++)*t;
	    *(dst++) += *(src++)*t;

            *(dst++) += *(src++)*t;	// modify transmittance of layer
	  }
	}
	return true;
    }

    /** clears the buffer with colour col */
    bool clear(const vuColour<N> colour)
    {
        if(m_Buffer == 0) return false;
	float col[] = {colour[0],colour[1],colour[2],colour[3]};
        float *dst = m_Buffer;
        for(register int i = m_Width*m_Height; i!=0; i--) {
            memcpy(dst,col,sizeof(float)*N);
            dst += N;
        }
        return true;
    }

    /** prints the contents of the framebuffer to the screen
     intended for debugging purpose on _small_ framebuffers */
    bool print()
    {
        if(m_Buffer == 0) return false;
        float *buf = m_Buffer;
        for(int i = 0; i<m_Height; i++) {
            for(int j = 0;j<m_Width; j++) {
                printf("[%3i,%3i] = (%4.5f", j, i, *(buf++));
                for(int c = 1; c<N; c++, buf++){
                    printf(", %4.5f", *buf);
                }
                printf(")\n");
            }
            putchar('\n');
        }
        return true;
    }

protected:
    int m_Width, m_Height; //!< the height and width of the buffer
    float *m_Buffer; //!< ptr to the array of data that holds the buffer's data
};


/** An alpha mask is a frame buffer with just one component */
typedef FrameBuffer<1> AlphaMask;
/** This is a complete buffer.
    Storing red, green, and blue plus extra alpha blending weight for each
    pixel. */
typedef FrameBuffer<4> RGBABuffer;

//};	// end of namespace

#endif
