#ifndef _FRAMEBUF16_H_
#define _FRAMEBUF16_H_

#include <stdio.h>
#include <string.h>

#include "vuSimpleTypes.h"
#include "Colour.h"

#define P2_16	65536	// 2^16
#define P2_16_1	65535	// 2^16 - 1
#define	FB_WORD

typedef word fbtype;

/** Frame buffer class with basic functionality.
    Based on floats, provides RGBA channels.
    The buffer is based on words (16 bit unsigned).
*/
template <int N>
class FrameBuffer
{
  friend class Colour<N>;
public:

    /** default constructor */
    FrameBuffer() {m_Buffer = 0; }

    /** contructor */
    FrameBuffer(int width, int height)
    {
        m_Buffer = 0;
        setSize(width, height);
    }

    /** destructor */
    ~FrameBuffer()
    {
        freeBuffer();
    }

    	/** this will return the number of channels of the buffer */
    int getNumberOfChannels() { return N; }
    	/** this will return the height of the buffer */
    int getHeight() { return m_Height; }
    	/** this will return the width of the buffer */
    int getWidth() { return m_Width; }
    	/** this will return a ptr to the array of buffer data */
    word* getData() { return m_Buffer; }

    	/** this will clear the buffer and set the size of the buffer to the new size
		\return true if successful
		\return false if not successful */
    bool setSize(int width, int height)
    {
        freeBuffer();
        m_Buffer = new word[ width * height * N];
        if(!m_Buffer) return false;
        m_Width = width;
        m_Height = height;
        return true;
    }

    	//! this will clear the buffer and release it's memory
    bool freeBuffer()
    {
        if(m_Buffer) delete [] m_Buffer;
        m_Buffer = 0;
        m_Width = 0;
        m_Height = 0;
        return true;
    }

    /** copy into current frame buffer
        number of channel has to be equal
        \param fb frame buffer to take data from
        \param x,y position in current buffer to operate on
        */
    bool copy(FrameBuffer<N>& fb, int x, int y)
    {
        if ((!fb.m_Buffer) ||
            (N != fb.getNumberOfChannels()) ||
            (x < 0 || y < 0) ||
            (x + fb.m_Width>m_Width)||
            (y + fb.m_Height>m_Height)) return false;

	int h = m_Height;
	int incdst = m_Width * N;
	int incsrc = fb.m_Width * N;
	word* src = fb.m_Buffer;
	word* dst = m_Buffer[y*m_Width + x];
	for(h=m_Height; h!=0; h--)
	  {
	    memcpy(dst,src,len);
	    dst += incdst;
	    src += incsrc;
	  }

        return true;
    }

    /** add intensities into buffer
        \param fb frame buffer to take data from
        \param x,y position in current buffer to operate on
        */
    bool add(FrameBuffer<N>& fb, int x, int y)
    {
      if ((!fb.m_Buffer) ||
	  (x < 0 || y < 0) ||
	  (x + fb.m_Width>m_Width)||
	  (y + fb.m_Height>m_Height)) return false;

      int incdst = (m_Width-fb.m_Width) * N;
      register word* src = fb.m_Buffer;
      register word* dst = &m_Buffer[y*m_Width + x];
      for(register int h=m_Height; h!=0; h--)
	{
	  for(register int len = fb.m_Width * N;len!=0;len--)
	    *(dst++) += *(src++);
	  dst += incdst;
	}
        return true;
    }


    /** add intensities into buffer
        \param fb frame buffer to take data from
        \param x,y position in current buffer to operate on
        */
    bool addColourWithMask(FrameBuffer<1>& fb, Colour<N>& colour, int x, int y)
    {
        if ((!fb.getData()) ||
            (x < 0 || y < 0) ||
            (x + fb.getWidth()>m_Width)||
            (y + fb.getHeight()>m_Height)) return false;

	word col[] = {word(colour[0]*P2_16_1),
		      word(colour[1]*P2_16_1),
		      word(colour[2]*P2_16_1),
		      word(colour[3]*P2_16_1)};
        int cpyh = fb.getHeight();
        int cpyw = fb.getWidth();
        int incdst = (m_Width-fb.getWidth()) * N;
        register word* src = fb.getData();
        register word* dst = &m_Buffer[(y*m_Width + x)*N];
        for(int h=cpyh; h!=0; h--) {
          for(register int len = cpyw; len!=0; len--) {
	    register word *colval = col;
	    register word alpha = (dword(*src)*col[3])>>16;
            for(register int comp = N-1; comp!=0; comp--, colval++) {
              *(dst++) += dword(alpha*(*colval))>>16;		//add weighted colour
            }
            *(dst++) += word(dword(*(src++))*(*colval)>>16);			//add alpha
          }
          dst += incdst;
        }
        return true;
    }

    /** subtract intensities from current frame buffer
        \param fb frame buffer to take data from
        \param x,y position in current buffer to operate on
        */
    bool subColourWithMask(FrameBuffer<1>& fb, Colour<N>& colour, int x, int y)
    {
        if ((!fb.getData()) ||
            (x < 0 || y < 0) ||
            (x + fb.getWidth()>m_Width)||
            (y + fb.getHeight()>m_Height)) return false;

	word col[] = {word(colour[0]*P2_16_1),
		      word(colour[1]*P2_16_1),
		      word(colour[2]*P2_16_1),
		      word(colour[3]*P2_16_1)};
        int cpyh = fb.getHeight();
        int cpyw = fb.getWidth();
        int incdst = (m_Width-fb.getWidth()) * N;
        register word* src = fb.getData();
        register word* dst = &m_Buffer[(y*m_Width + x)*N];
        for(int h=cpyh; h!=0; h--) {
          for(register int len = cpyw; len!=0; len--) {
	    register word *colval = col;
            for(register int comp = N-1; comp!=0; comp--, colval++) {
              *(dst++) -= word((dword(*src)*(*colval))>>16);		//sub weighted colour
            }
            *(dst++) -= (dword(*(src++))*(*colval))>>16;			//sub alpha
          }
          dst += incdst;
        }
        return true;
    }

    /** add intensities into buffer using the difference of the two alpha masks
        \param fb frame buffer to take data from
        \param x,y position in current buffer to operate on
        */
    bool addColourWithM1subM2(FrameBuffer<1>& m1, FrameBuffer<1>& m2,
			      Colour<N>& colour, const int x, const int y)
    {
        if ((!m1.getData()) ||
	    (!m2.getData()) ||
	    ( m1.getWidth()!=m2.getWidth()) ||
	    ( m1.getHeight()!=m2.getHeight()) ||
            (x < 0 || y < 0) ||
            (x + m1.getWidth()>m_Width)||
            (y + m1.getHeight()>m_Height)) return false;

	word col[] = {word(colour[0]*P2_16_1),
		      word(colour[1]*P2_16_1),
		      word(colour[2]*P2_16_1),
		      word(colour[3]*P2_16_1)};

	if(N != 4) {
	  int cpyh = m1.getHeight();
	  int cpyw = m1.getWidth();
	  int incdst = (m_Width-m1.getWidth()) * N;
	  register word* datm1 = m1.getData();
	  register word* datm2 = m2.getData();
	  register word* dst = &m_Buffer[(y*m_Width + x)*N];
	  for(int h=cpyh; h!=0; h--) {
	    for(register int len = cpyw; len!=0; len--) {
	      dword a = (dword((*datm1)-(*datm2))*col[N-1])>>16;
	      for(int comp = 0; comp<N-1; comp++) {
		*(dst++) += (dword(a)*col[comp])>>16;		//add weighted colour
	      }
	      *(dst++) += a;			//add alpha
	      datm1++; datm2++;
	    }
	    dst += incdst;
	  }
	} else {	// N == 4
	  int cpyh = m1.getHeight();
	  int cpyw = m1.getWidth();
	  int incdst = (m_Width-m1.getWidth()) * N;
	  register word* datm1 = m1.getData();
	  register word* datm2 = m2.getData();
	  register word* dst = &m_Buffer[(y*m_Width + x)*N];
	  for(int h=cpyh; h!=0; h--) {
	    for(register int len = cpyw; len!=0; len--) {
	      register dword a = (dword((*datm1)-(*datm2))*col[N-1])>>16;
	      register word * colcomp = col;
	      *(dst++) += word((a*(*(colcomp++)))>>16);	      //add weighted colour R
	      *(dst++) += word((a*(*(colcomp++)))>>16);	      // G
	      *(dst++) += word((a*(*(colcomp++)))>>16);	      // B
	      *(dst++) += a;			//add alpha
	      datm1++; datm2++;
	    }
	    dst += incdst;
	  }
	}
        return true;
    }

    /** blends other buffer over current */
    bool blendOver(FrameBuffer<N>& fb, int x, int y)
    {
        if ((!fb.getData()) ||
            (x < 0 || y < 0) ||
            (x + fb.getWidth()>m_Width)||
            (y + fb.getHeight()>m_Height)) return false;

        int cpyh = fb.getHeight();
        int cpyw = fb.getWidth();
        int incdst = (m_Width-fb.getWidth()) * N;
        register word* src = fb.getData();
        register word* dst = &m_Buffer[(y*m_Width + x)*N];
        for(int h=cpyh; h!=0; h--) {
          for(register int len = cpyw; len!=0; len--) {
	    word a2 = P2_16_1-src[N-1];
            for(register int comp = 0;
		comp<N-1;
		comp++, dst++, src++) {
              *(dst) = word(((dword(*dst)*a2)>>16) + (*src));
            }
            dst++;	// do nothing with the alpha value.
	    src++;
          }
          dst += incdst;
        }
        return true;
    }

    /** blends other buffer over current */
    bool blendOver(FrameBuffer<N>& fb)
    {
        if ((!fb.getData()) ||
            (fb.getWidth()!=m_Width)||
            (fb.getHeight()!=m_Height)) return false;
	if(N!=4) {
	  int cpylen = fb.getHeight()*fb.getWidth();
	  register word* src = fb.getData();
	  register word* dst = m_Buffer;
	  for(register int len = cpylen; len!=0; len--) {
	    word a2 = P2_16_1-src[N-1];
            for(register int comp = 0;
		comp<N-1;
		comp++, dst++, src++) {
              *(dst) = word(((dword(*dst)*a2)>>16) + (*src));
            }
            dst++;	// do nothing with the alpha value.
	    src++;
	  }
	} else {
	  int cpylen = fb.getHeight()*fb.getWidth();
	  register word* src = fb.getData();
	  register word* dst = m_Buffer;
	  for(register int len = cpylen; len!=0; len--) {
	    register word a2 = ~src[N-1];	//1-x

	    *(dst++) = word((dword(dword(*dst)*a2)>>16) + (*src++));	//RGB...
	    *(dst++) = word((dword(dword(*dst)*a2)>>16) + (*src++));
	    *(dst++) = word((dword(dword(*dst)*a2)>>16) + (*src++));

	    dst++;	// do nothing with the alpha value.
	    src++;
	  }
	}
	return true;
    }

    /** blends other buffer over current
	alpha values in current buffer are used as transmittance
	and are modified accordingly by blending in new images */
    bool blendUnder(FrameBuffer<N>& fb, int x, int y)
    {
        if ((!fb.getData()) ||
            (x < 0 || y < 0) ||
            (x + fb.getWidth()>m_Width)||
            (y + fb.getHeight()>m_Height)) return false;

        int cpyh = fb.getHeight();
        int cpyw = fb.getWidth();
        int incdst = (m_Width-fb.getWidth()) * N;
        register word* src = fb.getData();
        register word* dst = &m_Buffer[(y*m_Width + x)*N];
        for(int h=cpyh; h!=0; h--) {
          for(register int len = cpyw; len!=0; len--) {
	    word a = word((dword(src[N-1])*dst[N-1])>>16);
	    word t = P2_16_1-dst[N-1];
            for(register int comp = 0;
		comp<N-1;
		comp++, dst++, src++) {
              *(dst) = word((dword(*dst)*t + dword(*src)*a)>>16);
            }
            *(dst++) *= P2_16_1-(*(src++));	// modify transmittance of layer
          }
          dst += incdst;
        }
        return true;
    }

    /** blends other buffer under current */
    bool blendUnder(FrameBuffer<N>& fb)
    {
        if ((!fb.getData()) ||
            (fb.getWidth()!=m_Width)||
            (fb.getHeight()!=m_Height)) return false;
	if(N!=4) {
	  int cpylen = fb.getHeight()*fb.getWidth();
	  register word* src = fb.getData();
	  register word* dst = m_Buffer;
	  for(register int len = cpylen; len!=0; len--) {
	    register dword t = dword(word(~dst[N-1]));
            for(register int comp = 0;
		comp<N-1;comp++) {
	      *(dst++) += word(dword(*(src++))*t>>16);
            }
            dst++;	// do nothing with the alpha value.
	    src++;
	  }
	} else {	// N==4
	  int cpylen = fb.getHeight()*fb.getWidth();
	  register word* src = fb.getData();
	  register word* dst = m_Buffer;
	  for(register int len = cpylen; len!=0; len--) {
	    register dword t = dword(word(~dst[N-1]));
	    if(t == dword(~0) || src[N-1]==0) {
	      dst += 4;
	      src += 4;
	    } else {
	      *(dst++) += word(dword(*(src++))*t>>16);
	      *(dst++) += word(dword(*(src++))*t>>16);
	      *(dst++) += word(dword(*(src++))*t>>16);

	      *(dst++) += word(dword(*(src++))*t>>16);
	    }
	  }
	}
	return true;
    }

    /** clears the buffer with colour col */
    bool clear(const Colour<N> colour)
    {
        if(m_Buffer == 0) return false;
	word col[] = {word(colour[0]*P2_16_1),
		      word(colour[1]*P2_16_1),
		      word(colour[2]*P2_16_1),
		      word(colour[3]*P2_16_1)};
	if(col[0]==0 && col[0]==col[1] && col[0]==col[2] && col[0]==col[3]) {
	  memset(m_Buffer,0,sizeof(word)*N*m_Width*m_Height);
	} else {
	  word *dst = m_Buffer;
	  for(register int i = m_Width*m_Height; i!=0; i--) {
            memcpy(dst,col,sizeof(word)*N);
            dst += N;
	  }
	}
	return true;
    }

    /** prints the contents of the framebuffer to the screen */
    bool print()
    {
        if(m_Buffer == 0) return false;
        word *buf = m_Buffer;
        for(int i = 0; i<m_Height; i++) {
            for(int j = 0;j<m_Width; j++) {
                printf("[%3i,%3i] = (%i", j, i, *(buf++));
                for(int c = 1; c<N; c++, buf++){
                    printf(", %i", *buf);
                }
                printf(")\n");
            }
            putchar('\n');
        }
        return true;
    }

protected:
    int m_Width, m_Height; //!< these are the height and width of the buffer
    word *m_Buffer;	//!< this is the ptr to the array that stores the buffer's data.
};


/** An alpha mask is a frame buffer with just one component */
typedef FrameBuffer<1> AlphaMask;
/** This is a complete buffer.
    Storing red, green, and blue plus extra alpha blending weight for each
    pixel. */
typedef FrameBuffer<4> RGBABuffer;

#endif
