////////////////////////////////////////////////////////////////////////
//                                                                    //
// Informatikpraktikum I "Implementation of the ShearWarp Algorithm"  //
// Februar - October 2002                                             //
//                                                                    //
// author: Sebastian Zambal                                           //
//         e9826978@student.tuwien.ac.at                              //
//                                                                    //
// file:   shearWarp.h (BCC-grids)                                    //
//                                                                    //
////////////////////////////////////////////////////////////////////////

//!Defines the value that represents orthogonal projection
#define VIEWING_MODE_ORTHO 0

//!Defines the value that represents perspective projection
#define VIEWING_MODE_PERSPECTIVE 1

//!Defines numerical value representing X
#define XDIR 0
//!Defines numerical value representing Y
#define YDIR 1
//!Defines numerical value representing Z
#define ZDIR 2
//!Defines numerical value representing W (for homogenous coordinates)
#define WDIR 3

//!Defines the threshold for the smallest opacity.
/*!
   THRESHOLD_RUNLENGTH is the smallest opacity, with which a voxel 
   is still saved in the runlength encoded volume
*/
#define THRESHOLD_RUNLENGTH 0.05

//!Defines the transparency threshold for creation of the 
//!intermediate image.
/*!
   When the intermediate image is constructed, when the opacity of 
   a pixel reaches MIN_TRANSPARENCY_MAKING_INTERMEDIATE the pixel
   is assumed to be opaque (and is not changed in the following iterations)
*/
#define MIN_TRANSPARENCY_MAKING_INTERMEDIATE 0.05


//!Defines the initial size of the OpenGL plane
#define OPENGL_PLANE_SIZE 140

//The zoom factor
#define ZOOM_FACTOR 250

#ifndef _VUVOLUMEBCCUNIMODAL3D1B1VALSHEARWARP_H_
#define _VUVOLUMEBCCUNIMODAL3D1B1VALSHEARWARP_H_

#include "../intensity.h"
#include "glos.h"
#include <GL/gl.h>
#include <GL/glu.h>
#include "vuSimpleTypes.h"
#include "vuNormalTable.h"
#include "vuVector.h"
#include "vuMatrix.h"

//!Structure for saving a single voxel
/*!In the RLEvoxel structure the sampling-value and the normal vector of a
   voxel are stored. This structure is needed for runlength encoding the
   volume data.
 */
struct RLEvoxel_bcc {
    //!The value of the voxel
    byte value;
    //!The amount of red in the color of the classified voxel
    float red;
    //!The amount of green in the color of the classified voxel
    float green;
    //!The amount of blue in the color of the classified voxel
    float blue;
    //!The amount of red in the color of the classified and shaded voxel
    float red_shaded;
    //!The amount of green in the color of the classified and shaded voxel
    float green_shaded;
    //!The amount of blue in the color of the classified and shaded voxel
    float blue_shaded;
    //!The opacity of the classified voxel
    float opacity;
    //!The corrected opacity
    float opacity_corr;
    //!The normal vector of the voxel
    vuVector normal;
};

//!Structure for saving a single voxel slice
/*!In the RLEslice structure the total shearing, the number of voxels,
   a pointer to the runlength-array and a pointer to the voxel-array
   are stored. This structure is needed for runlength encoding of the 
   volume data.
 */
struct RLEslice_bcc {
    //!The total sharing of the slice.
    float dim1_pos,dim2_pos;
    //!The number of voxels in the slice.
    dword size;
    //!The scaling. Needed for perspective projection.
    float scale;
    //!Pointer to the runlength-array.
    byte *runlength;
    //!Pointer to the voxel-array.
    RLEvoxel_bcc *voxel;
};

//!Structure for saving 
/*!This Structure saves the RGB-value, the transparency and the opacity of 
   a single pixel in the intermediate image.
 */
struct intermediatePixel_bcc {
    //!value of red
    float red;
    //!value of green
    float green;
    //!value of blue
    float blue;
    //!value of transparency
    float trans;
    //!offset for skipping runs of opaque pixels
    dword offset;
};

//!Implementation of the shear-warp algorithm for BCC-grids
class vu1512119;
typedef vu1512119 vuVolumeBccUnimodal3d1B1ValShearWarp;

//!Implements a shear-warp algorithm (based on BCC-grids) for this leaf of the tree
/*!This class renders the volume using a shear-warp algorithm
   implemented within the scope of an "Informatikpraktikum 1" 
   by Sebastian Zambal.
   The shear warp algorithm shears the slices of pixels of a
   given volume to achive different views of the volume. 
   Unfortunately the projection of the slices results
   in a skew image. This intermediate image then has to be
   warped to get the correct final image.
 */
class vu1512119 : public vu151211
{
public:

    //!The constructor. Sets up a new ShearWarp instance.    
    /*!
     * The constructor sets up a new ShearWarp instance and
     * resets the zoom value, the normal vector table, the pointer
     * to the runlength encoded data, the pointer to the currently
     * used runlength encoded data, the intermediate image, the 
     * viewing mode (orthogonal, perspective) and the OpenGL-texture.
     */
    vu1512119();

    //!The Destructor. Destroys the ShearWarp instance.
    /*!
     * The destructor frees all occupied memory: the OpenGL-Image,
     * the normal vector table and the runlength encoded data.
     */
    virtual ~vu1512119();

public:
    //!Sets the size for the canvas
    /*!
     * This method must be called by vuShearWarp when then window
     * is resized and the canvas gets a new size.
     *
     * @param width  the width of the canvas.
     * @param height the height of the canvas.
     */
    void setCanvasSize(int width, int height);

    //!Turns on/off OpenGL-warping.
    /*!
     * With this method between warping with software and 
     * warping with OpenGL can be choosen.
     *
     * @param userOpenGL 0 = warp with software, 
     *                   1 = warp with OpenGL
     */
    void setOrthogonalWarpOpenGL(int useOpenGL);

    //!Computes the runlength encoding of the volume.
    /*!
     * Does runlength encoding for each of the main viewing directions.
     * After execution of this method the attributes dataX, dataY and dataZ 
     * hold the runlength encoded volume data according to the different
     * main viewing directions.
     */
    void runlengthEncode();

    //!Removes the runlength encoding
    /*! 
     * Frees all memory that was allocated by the runlength encoded Data.
     */
    void removeRunlengthEncoding();

    //!Turn on/off the specular light
    /*!
     * This method must be called when the specular light
     * is to be turned on/off
     *
     * @param spec 0 for turn off specular light, 1 for turn on.
     */
    void setSpecular(int spec);

    //!Set the projection mode (orthogonal is default).
    /*!
     * Sets the viewing mode (i.e. the type of projection to the
     * final image) to either orthogonal or perspective projection. 
     *
     * @param mode the viewing/projection mode. Possible values
     *             are VIEWING_MODE_ORTHO and VIEWING_MODE_PERSPECTIVE.
     */
    void setViewing(int mode);

    //!Fast classification currently used?
    /*!
     * Returns whether fast classification currently is used.
     *
     * @return 0 = currently not using fast classification, 
     *         1 = using fast classification
     */
    int getFastClassification();

    //!Sets the camera viewing position for the render method.
    /*!
     * setViewVectors updates the viewing vector, the up vector and the right
     * vector. In other words this method sets a new camera viewing position.
     */
    void setViewVectors(const vuVector& view,const vuVector& up,const vuVector& right);

    //!Returns the Dimensions of the volume data
    void getDimensions(int &x, int &y, int &z);
    //!Returns the Maximum Size of the Volume
    int getMaxSize(void);

    //!Zooms the volume
    /*!
     * Sets the zoom-value.
     */
    void zoom(float zoomValue);
 
    //!Initializes OpenGL.
    /*!
     * OpenGL is used for graphics output.
     */
    void initOpenGL(void);

    //!Implements the abstract render() method of the vuVolume base class.
    /*! 
     * This method does the complete rendering by calling the
     * steps of the algorithm (i.e. other methods) one by one.
     */
    void render(void);

    //!Reimplements the read() method to do some extra volume data processing.
    virtual bool read(void);
    
    //!This method reads volume data stored using the .raw data type.
    virtual bool readRaw(void);

    //!Returns the minimum allowed distance between eye and projection-plane
    float getMinEyeDistance(void);

    //!Sets the current distance between eye and projection-plane
    /*!
     * @param distance the distance between eye and projection-plane
     */
    void setEyeDistance(float distance);

private:
    //!Computes the maximum stretch out of the volume and saves the value in attribute maxSize
    void computeMaxSize(void);

    //!Precomputes the normals of the voxels
    void computeNormals(void);

    //!Computes the main viewing direction
    void computeMainViewingDirection(void);

    //!Computes the correct permutation matrix
    void computePermutationMatrix(void);

    //!Computes the view matrix for coordinates in standard object space
    void computeViewMatrix(void);

    //!Computes the position of the eye point (needed for perspective projection)
    void computeEyePoint(void);

    //!Determines width, height and depth of the Volume and saves these values in runlengthWidth...
    void computeRunlengthSizes(void);

    //!Computes the Shear and Warp Factors for parallel projection
    void computeShearAndWarpFactorsOrtho(void);

    //!Shearing the runlength encoded data for parallel projection
    void shearOrtho(void);

    //!Compute the length of the Z-normalized viewing vector.
    /*!
     * @return the length of the Z-normalized viewing vector.
     */
    float computeZNormalizedViewingVectorLength(void);

    //!Computes the inverse 2D-warp-matrix
    void computeInvWarpOrtho(void);

    //!Inverse warping for orthogonal viewing for a single point with x- and y-coordinates
    void warpOrthoInv(float x, float y, float &x_result, float &y_result);

    //!The total orthogonal warping from intermediate image to OpenGL-texture
    void warpOrtho();

    //!Creating the intermediate image for parallel projection
    void makeIntermediateImageOrtho(void);

    //!change the format of the intermediate image to OpenGL-format
    void changeFormatIntermediate(void);

    //!draw the final image using OpenGL for warping
    void drawWarpOpenGL(void);

    //!Returns the value of the sample at the given coordinates
    byte getSample(dword x, dword y, dword z);

    //!Computes the Shear Matrix for perspective projection
    void computeShearPerspective(void);

    //!Computes the Warp Matrix for perspective projection
    void computeWarpPerspective(void);

    //!Computes position and the scaling for a single slice.
    /*!
     * @param pos_i the x-coordinate of the result
     * @param pos_j the y-coordinate of the result
     * @param scale the scaling factor for this slice
     * @param the slice for which computations are to be done
     */
    void computeSlicePositionScale(float &pos_i, float &pos_j, float &scale, int slice);

    //!Shears the runlength encoded data for perspective projection.
    void shearPerspective(void);

    //!Performs an inverse perspective warp.
    /*!
     * Applies the inverse of perspective warping to a single point.
     *
     * @param x the x-coordinate of the point that is to be transformed
     * @param y the y-coordinate of the point that is to be transformed
     * @param x_result the x-coordiante of the transformed point
     * @param y_result the y-coordinate of the transformed point
     */
    void warpPerspectiveInv(float x, float y, float &x_result, float &y_result);

    //!The total perspective warping from intermediate image to OpenGL-texture
    void warpPerspective(void);

    //!Creating the intermediate image for perspective projection.
    void makeIntermediateImagePerspective(void);

    //!Draws the final image with openGL.
    void drawOpenGL(void);

    //!Assign memory for the OpenGL-image.
    void createGLImage(void);
    

private:
    //!Stores an array of normal vectors
    float*       m_Normals;

    //!The viewing direction of the camera
    vuVector     m_View;

    //!The up direction of the camera
    vuVector     m_Up;

    //!The right direction of the camera
    vuVector     m_Right;

    //!The view matrix.
    /*!
     * Represents the matrix that transforms a point in object space
     * to its corresponding position in the final image. So this
     * matrix represents the total viewing transformation.
     */
    vuMatrix     viewMatrix;

    //!The permutation matrix.
    /*!
     * Represents the matrix that permutates the coordinates 
     * in such a way that the Z-axis becomes the main viewing
     * direction.
     */
    vuMatrix     permMatrix;

    //!The world matrix.
    /*!
     * Transforms a point from object coordinates to world
     * coordinates.
     */
    vuMatrix     worldMatrix;

    //!The shear matrix.
    /*!
     * This matrix performs the shearing step in the shear-warp
     * algorithm.
     */
    vuMatrix     shearMatrix;

    //!The warp matrix.
    /*!
     * This matrix performs the warping step in the shear-warp
     * algorithm
     */
    vuMatrix     warpMatrix;

    //!The inverse of the warp matrix.
    /*!
     * When warping with software, for each point in the
     * final image the corresponding point (actually 4 points
     * which are then interpolated) in the intermediate image
     * must be found. So we need the inverse of the warp matrix.
     */
    vuMatrix     invWarpMatrix;

    //!The perspective projection matrix (needed only for
    //!perspective viewing.
    /*!
     * This matrix represents a perspective projection.
     */
    vuMatrix     projMatrix;

    //!The inverse of the worldMatrix.
    /*!
     * Transforms from world coordinates to object coordinates.
     */
    vuMatrix     invWorldMatrix;

    //!Element 00 of the inverse 2D-Warp-Matrix for orthogonal projection
    float invWarpOrtho00;
    //!Element 01 of the inverse 2D-Warp-Matrix for orthogonal projection
    float invWarpOrtho01;
    //!Element 10 of the inverse 2D-Warp-Matrix for orthogonal projection
    float invWarpOrtho10;
    //!Element 11 of the inverse 2D-Warp-Matrix for orthogonal projection
    float invWarpOrtho11;

    //!The eye point (for perspective projection) in object-space.
    vuVector eye_o;
    //!The eye point (for perspective projection) in sheared space.
    vuVector eye_s;

    //!The current main viewing direction
    int mainViewingDir;

    //!The width of the canvas.
    /*!
     * Holds the width of the used canvas. This value is needed for computing
     * the horizontal scaling factor of the final image.
     */
    int canvasWidth;
    //!The height of the canvas.
    /*!
     * Holds the height of the used canvas. This value is needed for computing
     * the vertical scaling factor of the final image. 
     */
    int canvasHeight;

    //!Needed for OpenGL-texturing.
    GLuint m_GLShearWarp;

    //!Orthogonal or perspective viewing?
    int viewingMode;
    //!Use specular light for shading?
    int specular;
    //!Use OpenGL for orthogonal warping
    int orthoWarpOpenGL;

    //!Runlength encoded data for main viewing direction X
    RLEslice_bcc* dataX;
    //!Runlength encoded data for main viewing direction Y
    RLEslice_bcc* dataY;
    //!Runlength encoded data for main viewing direction Z
    RLEslice_bcc* dataZ;

    //!The currently used runlength encoded data
    /*!
     *Points to the currently used runlength encoded data. This 
     * depends on the main viewing direction
    */
    RLEslice_bcc* curr;

    //!Holds the direction in which the volume shall be traversed
    /*!
     * Shall the volume be traversed foreward or backward? The value 0 
     * means that the volume has to be traversed backward, 1 means 
     * it has to be traversed foreward.
    */
    byte direction;

    //!The maximum stretch-out of the volume
    /*!
     * Holds the maximum stretch-out of all three dimensions of the volume
     */
    dword maxSize;

    //!The depth of the volume 
    /*!
     * The depth of the volume (depends on the main viewing direction!)
     */
    dword volumeDepth;
    //!The width of the volume 
    /*!
     * The width of the volume (depends on the main viewing direction!)
     */
    dword volumeWidth;
    //!The height of the volume 
    /*!
     * The height of the volume (depends on the main viewing direction!)
     */
    dword volumeHeight;

    //!The shearing in dimension i
    /*!
     * si holds the value about which the positions of two adjacent slices 
     * of the volume differ in horizontal direction.
     */ 
    double si;
    //!The shearing in dimension j
    /*!
     * sj holds the value about which the positions of two adjacent planes 
     * of the volume differ in vertical direction
     */
    double sj;

    //!The value about which the front slice of the data is 
    //!shifted in the intermediate image horizontally.
    float ti;
    //!The value about which the front slice of the data is 
    //!shifted in the intermediate image vertically.
    float tj;

    //!The intermediate image
    /*!An array that stores the pixels of the intermediate image.
     */
    intermediatePixel_bcc* intermediate;

    //Distance between eye point and projection plane (only for perspective)
    float eye_distance;

    //!The left, right, up and down limits of the intermediate image
    int left;
    int right;
    int up;
    int down;

    //!The perspective scaling factor
    /*!The factor, by which the intermediate image must be scaled
       to get the right final size (only perspective projection).
    */
    float scale_total;

    //!Current zoom-value.
    float zoomValue;

    //!The image that later is processed by OpenGL
    GLubyte *glImage;

    //!The width of the OpenGL image.
    int glImageWidth;
    //!The height of the OpenGL image.
    int glImageHeight;
};

#endif














