////////////////////////////////////////////////////////////////////////
//                                                                    //
// Informatikpraktikum I "Implementation of the ShearWarp Algorithm"  //
// Februar - October 2002                                             //
//                                                                    //
// author: Sebastian Zambal                                           //
//         e9826978@student.tuwien.ac.at                              //
//                                                                    //
// file:   shearWarp.cpp (BCC-grids)                                  //
//                                                                    //
////////////////////////////////////////////////////////////////////////

#include "shearWarp.h"
#include <fstream.h>
#include <math.h>
#include <exception>

//-----------------------------------------------------------------------
//-- Constructor                                                      ---
//-----------------------------------------------------------------------
vu1512119::vu1512119()
{
    //initially no zoom
    zoomValue = 1;

    //initially don't use OpenGL for warping
    orthoWarpOpenGL = 0;

    //initially no normals
    m_Normals = 0;

    //initially there is no runlength encoded data
    dataX = 0;
    dataY = 0;
    dataZ = 0;

    //initially no current runlength encoding
    curr = 0;
    //initially no intermediate image
    intermediate = 0;
    //initially orthogonal projection
    viewingMode = VIEWING_MODE_ORTHO;
    //initially no specular shading
    specular = 0;

    m_GLShearWarp = 0;
}

//-----------------------------------------------------------------------
//-- Destructor                                                       ---
//-----------------------------------------------------------------------
vu1512119::~vu1512119() 
{
    // free memory
    if (glImage) {
	delete [] glImage;
	glImage = 0;
    }
    if (m_Normals) delete [] m_Normals;
    if (dataX) {
	for (dword i = 0; i < m_Dim1Size; ++i) { 
	    if (dataX[i].size>0) {
		delete [] dataX[i].voxel;
		delete [] dataX[i].runlength;
	    }
	}
	delete [] dataX;
    }
    if (dataY) {
	for (dword i = 0; i < m_Dim2Size; ++i) { 
	    if (dataY[i].size>0) {
		delete [] dataY[i].voxel;
		delete [] dataY[i].runlength;
	    }
	}
	delete [] dataY;
    }
    if (dataZ) {
	for (dword i = 0; i < m_Dim3Size; ++i) { 
            if (dataZ[i].size>0) {
		delete [] dataZ[i].voxel;
		delete [] dataZ[i].runlength;
	    }
	}
	delete [] dataZ;
    }
    if (intermediate) {
	delete [] intermediate;
    }
}

//-----------------------------------------------------------------------
//-- returns the size of dimensions of the volume data                ---
//-----------------------------------------------------------------------
void vu1512119::getDimensions(int &x, int &y, int &z) {
    x = m_Dim1Size;
    y = m_Dim2Size;
    z = m_Dim3Size;
}

//-----------------------------------------------------------------------
//-- Construct OpenGL image                                           ---
//-----------------------------------------------------------------------
void vu1512119::createGLImage(void) {

    word glSize = 64;
    while (glSize < maxSize*2) {
	glSize *= 2;
    }

    glImageWidth = glSize;
    glImageHeight = glSize;
    glImage = new GLubyte[glImageWidth*glImageHeight*4];
}

//-----------------------------------------------------------------------
//-- Compute maximum stretch-out of volume                            ---
//-----------------------------------------------------------------------
void vu1512119::computeMaxSize(void) 
{
    maxSize = m_Dim1Size;
    if (m_Dim2Size > maxSize) maxSize = m_Dim2Size;
    if (m_Dim3Size/2 > maxSize) maxSize = m_Dim3Size/2;
    maxSize += 6;
}

//-----------------------------------------------------------------------
//-- Set view vectors                                                 ---
//-----------------------------------------------------------------------
void vu1512119::setViewVectors(const vuVector& view, const vuVector& up, 
			       const vuVector& right)
{
  m_View  = view;
  m_Up    = up;
  m_Right = right;
}

//-----------------------------------------------------------------------
//-- Set viewing Mode                                                 ---
//-----------------------------------------------------------------------
void vu1512119::setViewing(int vm)
{
    viewingMode = vm;
}

//-----------------------------------------------------------------------
//-- Turn on/off specular light                                       ---
//-----------------------------------------------------------------------
void vu1512119::setSpecular(int spec)
{
    specular = spec;
}

//-----------------------------------------------------------------------
//-- Zoom                                                             ---
//-----------------------------------------------------------------------
void vu1512119::zoom(float value) 
{
    if (value > 0) zoomValue = value;
}

//-----------------------------------------------------------------------
//-- Turn on/off use of OpenGL                                        ---
//-----------------------------------------------------------------------
void vu1512119::setOrthogonalWarpOpenGL(int useOpenGL) 
{
    orthoWarpOpenGL = useOpenGL;
}


//-----------------------------------------------------------------------
//-- Set size of canvas                                               ---
//-----------------------------------------------------------------------
void vu1512119::setCanvasSize(int width, int height) 
{
    canvasWidth = width;
    canvasHeight = height;
}

//-----------------------------------------------------------------------
//-- Reimplements the read() method to do some extra volume data      ---
//-- processing.                                                      ---
//-----------------------------------------------------------------------
//!Reimplements the read() method to do some extra volume data processing
bool vu1512119::read()
{
    bool success = vu151211::read();
    if (!success) return false;

    //Allocate memory for the normals.
    if (m_Normals != 0) delete [] m_Normals;

    m_Normals = new float[m_DataSize*3];

    computeMaxSize();

    //initially value for distance between eye and projection plane
    eye_distance = getMinEyeDistance();

    createGLImage();
    computeNormals();

    runlengthEncode();

    return true;
}

//-----------------------------------------------------------------------
//-- Implements readRaw                                               ---
//-----------------------------------------------------------------------
bool vu1512119::readRaw(void)
{
    dword len;
    ifstream in;

    in.open(m_FileName, ios::in|ios::binary
#ifdef IS_NOCREATE_NEEDED
	    |ios::nocreate
#endif
	);
    if (!in.is_open()) return false;

    in.seekg(0, ios::end);
    len = in.tellg();
    in.seekg(0, ios::beg);

    in >> m_Dim1Size >> m_Dim2Size >> m_Dim3Size;
    if (in.fail()) return false;
    m_DataSize = m_Dim1Size*m_Dim2Size*m_Dim3Size;

    m_Data = new byte[m_DataSize];
    in.read((char*)m_Data, m_DataSize);
    if (in.fail()) return false;

    in.close();

    //Allocate memory for the normals.
    if (m_Normals != 0) delete [] m_Normals;

    m_Normals = new float[m_DataSize];

    computeNormals();

    return true;
}

//-----------------------------------------------------------------------
//-- Removes the runlength encoding from memory                       ---
//-----------------------------------------------------------------------
void vu1512119::removeRunlengthEncoding() 
{
    // if neccessary delete existing runlength-encoding
    if (dataX) {
	for (dword i = 0; i < m_Dim1Size; ++i) { 
	    if (dataX[i].size>0) {
		delete [] dataX[i].voxel;
		delete [] dataX[i].runlength;
	    }
	}
	delete [] dataX;
    }
    if (dataY) {
	for (dword i = 0; i < m_Dim2Size; ++i) { 
	    if (dataY[i].size>0) {
		delete [] dataY[i].voxel;
		delete [] dataY[i].runlength;
	    }
	}
	delete [] dataY;
    }
    if (dataZ) {
	for (dword i = 0; i < m_Dim3Size; ++i) { 
            if (dataZ[i].size>0) {
		delete [] dataZ[i].voxel;
		delete [] dataZ[i].runlength;
	    }
	}
	delete [] dataZ;
    }
}

//-----------------------------------------------------------------------
//-- Does the runlength encoding                                      ---
//-----------------------------------------------------------------------
void vu1512119::runlengthEncode()
{
    removeRunlengthEncoding();

    intermediate = new intermediatePixel_bcc[maxSize*maxSize*4];

    dword voxels = 0;           // the number of voxels in the current slice
    dword runs = 0;             // the number of runs in the current slice
    byte lengthr = 0;           // length of the run
    byte state = 0;             // are we in a run of transparent 
                                // (0) or non-transparent (1) voxels?
    dword voxelIndex = 0;       // index of voxel in array voxel
    dword runIndex = 0;         // index of run in array runlength
    dword indexRunlength = 0;   // index in the runlength encoded data
    dword indexVolume = 0;      // index in the original volume data
    dword maxSliceSize = m_Dim2Size * m_Dim1Size;  // uncompressed slice size

    m_View.makeUnit();
    if ((m_View[XDIR] < 0.1) && (m_View[YDIR] < 0.1) && (m_View[ZDIR] < 0.1)) {
	m_View[XDIR] = 0.3;
	m_View[YDIR] = 0.4;
	m_View[ZDIR] = 0.2;
	m_View.makeUnit();
    }

    dataZ = new RLEslice_bcc[m_Dim3Size];

    for (word z = 0; z < m_Dim3Size; ++z) {

	voxels = 0;
	runs = 0;
	state = 0;
	lengthr = 0;

	for (dword i = 0; i < maxSliceSize; ++i) {
            if (m_TFunc[m_Data[maxSliceSize*z + i]][3] > THRESHOLD_RUNLENGTH) {
		++voxels;
		if (state == 0) {
		    ++runs;
		    lengthr = 0;
		    state = 1;
		}
	    } else {
		if (state == 1) {
		    ++runs;
		    lengthr = 0;
		    state = 0;
		}
	    }
	    if ((lengthr == 255) || ((i+1) % m_Dim1Size == 0)) {
		runs += 3;
		lengthr = 0;
		if (state == 0) state = 1; else state = 0;
	    }
	}

        // set data of the slice
	dataZ[z].size = voxels;
	dataZ[z].dim1_pos = 0;
	dataZ[z].dim2_pos = 0;
	dataZ[z].scale = 1;

        // if slice is not empty...
	if (voxels > 0) {

	    dataZ[z].voxel = new RLEvoxel_bcc[voxels+1];
	    dataZ[z].runlength = new byte[runs+1];

	    state = 0;
	    indexRunlength = 0;
	    voxelIndex = 0;
	    runIndex = 0;

	    dataZ[z].runlength[0] = 0; // init first runlength

            // construct data for current slice
	    for (dword i = 0; i < maxSliceSize; ++i) {

		if (m_TFunc[m_Data[maxSliceSize*z + i]][3] > THRESHOLD_RUNLENGTH) {

		    byte val = m_Data[maxSliceSize*z + i];
		    dataZ[z].voxel[voxelIndex].value = val;
		    dataZ[z].voxel[voxelIndex].normal[XDIR] = m_Normals[(maxSliceSize*z + i)*3 + XDIR];
		    dataZ[z].voxel[voxelIndex].normal[YDIR] = m_Normals[(maxSliceSize*z + i)*3 + YDIR];
		    dataZ[z].voxel[voxelIndex].normal[ZDIR] = m_Normals[(maxSliceSize*z + i)*3 + ZDIR];

		    dataZ[z].voxel[voxelIndex].red = m_TFunc[val][0];
		    dataZ[z].voxel[voxelIndex].green = m_TFunc[val][1];
		    dataZ[z].voxel[voxelIndex].blue = m_TFunc[val][2];
		    dataZ[z].voxel[voxelIndex].opacity = m_TFunc[val][3];

		    ++voxelIndex;
		    if (state == 0) {
			++runIndex;
			dataZ[z].runlength[runIndex] = 0;
			state = 1;
		    }
		    ++dataZ[z].runlength[runIndex];
		} else {
		    if (state == 1) {
			++runIndex;
			dataZ[z].runlength[runIndex] = 0;
			state = 0;
		    }
		    ++dataZ[z].runlength[runIndex];
		}
		if ((dataZ[z].runlength[runIndex] == 255) || ((i+1) % m_Dim1Size == 0)) {
		    ++runIndex;
		    dataZ[z].runlength[runIndex] = 0;
		    if (state == 0) state = 1; else state = 0;
		}
	    }

	} else {
	    dataZ[z].voxel = 0;
	    dataZ[z].runlength = 0;
	}
    }

    //--------------------------------------------------------------
    // Now we do the same for the main viewing direction in y

    dword start = 0;
    indexVolume = 0;
    maxSliceSize = m_Dim1Size * m_Dim3Size / 2;  // uncompressed slice size

    dataY = new RLEslice_bcc[m_Dim2Size*2];

    for (word y = 0; y < m_Dim2Size; ++y) {

	for (int bcc = 0; bcc <= 1; ++bcc) {

	    voxels = 0;
	    runs = 0;
	    state = 0;
	    lengthr = 0;

            dword step = m_Dim1Size * m_Dim2Size;  // size of the steps in a row
	    dword row_length = (m_Dim3Size-2)*step;  // length of a row

            if ((m_Dim3Size % 2 == 1) && (bcc == 0)) {
		row_length -= step*2;
	    }

	    indexVolume = y * m_Dim1Size;
	    if (bcc == 1) indexVolume += step;   // 2nd row of BCC-Grid => start at 2nd Voxel

	    start = indexVolume;

            // find out the number of voxels and runs in the slice
	    for (dword i = 0; i < maxSliceSize; ++i) {
		if (m_TFunc[m_Data[indexVolume]][3] > THRESHOLD_RUNLENGTH) {
		    ++voxels;
		    if (state == 0) {
			++runs;
			lengthr = 0;
			state = 1;
		    }
		} else {
		    if (state == 1) {
			++runs;
			lengthr = 0;
			state = 0;
		    }
		}
		if ((lengthr == 255) || ((i+1) % (m_Dim3Size/2) == 0)) {
		    runs += 3;
		    lengthr = 0;
		    if (state == 0) state = 1; else state = 0;
		}


		if (i > 0) {
                    //if one row is finished -> increment start position for next row and
                    //continue at this new start position
		    if (indexVolume >= y*m_Dim1Size+row_length) {
			++start;
			indexVolume = start;

                    //...otherwise just go on tracing through the row
		    } else if (i > 0) {
			indexVolume += step * 2;
		    }
		}
	    }

            // set data of the slice
	    dataY[2*y+bcc].size = voxels;
	    dataY[2*y+bcc].dim1_pos = 0;
	    dataY[2*y+bcc].dim2_pos = 0;
	    dataY[2*y+bcc].scale = 1;

            // if slice is not empty...
	    if (voxels > 0) {

		dataY[2*y+bcc].voxel = new RLEvoxel_bcc[voxels+1];
		dataY[2*y+bcc].runlength = new byte[runs+1];

		state = 0;
		indexRunlength = 0;
		voxelIndex = 0;
		runIndex = 0;

	        dataY[2*y+bcc].runlength[0] = 0; // init first runlength

		indexVolume = y * m_Dim1Size;
	        if (bcc == 1) indexVolume += step;   // 2nd row of BCC-Grid => start at 2nd Voxel

		start = indexVolume;

                // construct data for current slice
		for (dword i = 0; i < maxSliceSize; ++i) {

		    if (i > 0) {
                    //if one row is finished -> increment start position for next row and
                    //continue at this new start position
			if (indexVolume >= y*m_Dim1Size+row_length) {
			    ++start;
			    indexVolume = start;
                    //...otherwise just go on tracing through the row
			} else if (i > 0) {
			    indexVolume += step * 2;
			}
		    }

		    if (m_TFunc[m_Data[indexVolume]][3] > THRESHOLD_RUNLENGTH) {

			byte val = m_Data[indexVolume];
			dataY[2*y+bcc].voxel[voxelIndex].value = val;
			dataY[2*y+bcc].voxel[voxelIndex].normal[XDIR] = m_Normals[indexVolume*3 + XDIR];
			dataY[2*y+bcc].voxel[voxelIndex].normal[YDIR] = m_Normals[indexVolume*3 + YDIR];
			dataY[2*y+bcc].voxel[voxelIndex].normal[ZDIR] = m_Normals[indexVolume*3 + ZDIR];
			dataY[2*y+bcc].voxel[voxelIndex].red = m_TFunc[val][0];
			dataY[2*y+bcc].voxel[voxelIndex].green = m_TFunc[val][1];
			dataY[2*y+bcc].voxel[voxelIndex].blue = m_TFunc[val][2];
			dataY[2*y+bcc].voxel[voxelIndex].opacity = m_TFunc[val][3];

			++voxelIndex;
			if (state == 0) {
			    ++runIndex;
			    dataY[2*y+bcc].runlength[runIndex] = 0;
			    state = 1;
			}
			++dataY[2*y+bcc].runlength[runIndex];
		    } else {
			if (state == 1) {
			    ++runIndex;
			    dataY[2*y+bcc].runlength[runIndex] = 0;
			    state = 0;
			}
			++dataY[2*y+bcc].runlength[runIndex];
		    }

		    //if (m_Dim3Size % 2 == 0) {
                    //if the capacity of the current runlength is exhausted or if there
                    //is a line-break: increase runIndex (new run with length 0) and change 
                    //current state 
		    if ((dataY[2*y+bcc].runlength[runIndex] == 255) || ((i+1) % (m_Dim3Size/2) == 0)) {
			++runIndex;
			dataY[2*y+bcc].runlength[runIndex] = 0;
			if (state == 0) state = 1; else state = 0;
		    }
		    //} else {
		    //if ((dataY[2*y+bcc].runlength[runIndex] == 255) || 
		    //	((i+1) % (((int)(m_Dim3Size/2))+(1-bcc)) == 0)) {
		    //	++runIndex;
		    //	dataY[2*y+bcc].runlength[runIndex] = 0;
		    //	if (state == 0) state = 1; else state = 0;
		    //}
		    //}
		}

	    } else {
		dataY[2*y+bcc].voxel = 0;
		dataY[2*y+bcc].runlength = 0;
	    }
	}
    }

    //--------------------------------------------------------------
    // Now we do the same for the main viewing direction in x

    indexVolume = 0;
    maxSliceSize = m_Dim2Size * m_Dim3Size / 2;  // uncompressed slice size

    dataX = new RLEslice_bcc[m_Dim1Size*2];

    for (word x = 0; x < m_Dim1Size; ++x) {

	for (int bcc = 0; bcc <= 1; ++bcc) {

	    voxels = 0;
	    runs = 0;
	    state = 0;
	    lengthr = 0;

	    indexVolume = x;
	    if (bcc == 1) indexVolume += m_Dim1Size * m_Dim2Size;

	    int even_correction = 0;
	    if (m_Dim3Size % 2 == 1) even_correction = -m_Dim2Size;

            // find out the number of voxels and runs in the slice
	    for (dword i = 0; i < maxSliceSize + even_correction; ++i) {

		if (m_TFunc[m_Data[indexVolume]][3] > THRESHOLD_RUNLENGTH) {
		    ++voxels;
		    if (state == 0) {
			++runs;
			lengthr = 0;
			state = 1;
		    }
		} else {
		    if (state == 1) {
			++runs;
			lengthr = 0;
			state = 0;
		    }
		}
		if ((lengthr == 255) || ((i+1) % m_Dim2Size == 0)) {
		    runs += 3;
		    lengthr = 0;
		    if (state == 0) state = 1; else state = 0;
		}
		if ((i + 1) % m_Dim2Size == 0) {  // line break in runlength encoding
		    indexVolume += m_Dim1Size * m_Dim2Size;
		}

		indexVolume += m_Dim1Size;
	    }

            // set data of the slice
 	    dataX[2*x+bcc].size = voxels;
	    dataX[2*x+bcc].dim1_pos = 0;
	    dataX[2*x+bcc].dim2_pos = 0;
	    dataX[2*x+bcc].scale = 1;

            // if slice is not empty...
	    if (voxels > 0) {

		dataX[2*x+bcc].voxel = new RLEvoxel_bcc[voxels+1];
		dataX[2*x+bcc].runlength = new byte[runs+1];

		state = 0;
		indexRunlength = 0;
		voxelIndex = 0;
		runIndex = 0;

	        dataX[2*x+bcc].runlength[0] = 0; // init first runlength

		indexVolume = x;
		if (bcc == 1) indexVolume += m_Dim1Size * m_Dim2Size;

                // construct data for current slice
		for (dword i = 0; i < maxSliceSize + even_correction; ++i) {

		    if (m_TFunc[m_Data[indexVolume]][3] > THRESHOLD_RUNLENGTH) {

			byte val = m_Data[indexVolume];
			dataX[2*x+bcc].voxel[voxelIndex].value = val;
			dataX[2*x+bcc].voxel[voxelIndex].normal[XDIR] = m_Normals[indexVolume*3 + XDIR];
			dataX[2*x+bcc].voxel[voxelIndex].normal[YDIR] = m_Normals[indexVolume*3 + YDIR];
			dataX[2*x+bcc].voxel[voxelIndex].normal[ZDIR] = m_Normals[indexVolume*3 + ZDIR];

			dataX[2*x+bcc].voxel[voxelIndex].red = m_TFunc[val][0];
			dataX[2*x+bcc].voxel[voxelIndex].green = m_TFunc[val][1];
			dataX[2*x+bcc].voxel[voxelIndex].blue = m_TFunc[val][2];
			dataX[2*x+bcc].voxel[voxelIndex].opacity = m_TFunc[val][3];

			++voxelIndex;
			if (state == 0) {
			    ++runIndex;
			    dataX[2*x+bcc].runlength[runIndex] = 0;
			    state = 1;
			}
			++dataX[2*x+bcc].runlength[runIndex];
		    } else {
			if (state == 1) {
			    ++runIndex;
			    dataX[2*x+bcc].runlength[runIndex] = 0;
			    state = 0;
			}
			++dataX[2*x+bcc].runlength[runIndex];
		    }
		    if ((dataX[2*x+bcc].runlength[runIndex] == 255) || ((i+1) % m_Dim2Size == 0)) {
			++runIndex;
			dataX[2*x+bcc].runlength[runIndex] = 0;
			if (state == 0) state = 1; else state = 0;
		    }
		    if ((i + 1) % m_Dim2Size == 0) {  // line break in runlength encoding
			indexVolume += m_Dim1Size * m_Dim2Size;
		    }

		    indexVolume += m_Dim1Size;
		}

	    } else {
		dataX[2*x+bcc].voxel = 0;
		dataX[2*x+bcc].runlength = 0;
	    }
	}
    }
}

//-----------------------------------------------------------------------
//-- Preprocesses volume data for rendering once it's been read.  -------
//-----------------------------------------------------------------------
void vu1512119::computeNormals(void)
{
    float norm[3];
    float len;
    dword w, wh;
    dword i, j, k, index;

    w = m_Dim1Size;
    wh = m_Dim1Size*m_Dim2Size*2;

    //
    // Compute all of the normals
    //
    index = 0;

    for(k=0;k<m_Dim3Size;++k)
    {
        for(j=0;j<m_Dim2Size;++j)
        {
            for(i=0;i<m_Dim1Size;++i)
            {
                if (i <= 1) {
                    norm[0] = m_Data[index+1]-m_Data[index];
		}
                else if (i >= m_Dim1Size-2) {
                    norm[0] = m_Data[index]-m_Data[index-1];
		}
                else {
                    norm[0] = (m_Data[index+1]-m_Data[index-1])*1;

		    if ((j > 1) && (j < m_Dim2Size-1) && (k > 1) && (k < m_Dim3Size-1)) {
		    norm[0] += (m_Data[index  +w +1]-m_Data[index  +w -1])*0.2;
		    norm[0] += (m_Data[index  -w +1]-m_Data[index  -w -1])*0.2;
		    norm[0] += (m_Data[index +wh +1]-m_Data[index +wh -1])*0.2;
		    norm[0] += (m_Data[index -wh +1]-m_Data[index -wh -1])*0.2;

		    norm[0] += (m_Data[index +wh +w +1]-m_Data[index +wh +w -1])*0.1;
		    norm[0] += (m_Data[index +wh -w +1]-m_Data[index +wh -w -1])*0.1;
		    norm[0] += (m_Data[index -wh +w +1]-m_Data[index -wh +w -1])*0.1;
		    norm[0] += (m_Data[index -wh -w +1]-m_Data[index -wh -w -1])*0.1;
		    }
		}

                if (j <= 1) {
                    norm[1] = m_Data[index+w]-m_Data[index];
		}
                else if (j >= m_Dim2Size-2) {
                    norm[1] = m_Data[index]-m_Data[index-w];
		}
                else {
                    norm[1] = (m_Data[index+w]-m_Data[index-w])*1;

		    if ((i > 1) && (i < m_Dim1Size-1) && (k > 1) && (k < m_Dim3Size-1)) {
                    norm[1] += (m_Data[index  +1 +w]-m_Data[index +1  -w])*0.2;
                    norm[1] += (m_Data[index  -1 +w]-m_Data[index -1  -w])*0.2;
                    norm[1] += (m_Data[index +wh +w]-m_Data[index +wh -w])*0.2;
                    norm[1] += (m_Data[index -wh +w]-m_Data[index -wh -w])*0.2; 

                    norm[1] += (m_Data[index +wh +1 +w]-m_Data[index +wh +1 -w])*0.1;
                    norm[1] += (m_Data[index +wh -1 +w]-m_Data[index +wh -1 -w])*0.1;
                    norm[1] += (m_Data[index -wh +1 +w]-m_Data[index -wh +1 -w])*0.1;
                    norm[1] += (m_Data[index -wh -1 +w]-m_Data[index -wh -1 -w])*0.1;
		    }
		}

                if (k <= 1) {
                    norm[2] = m_Data[index+wh]-m_Data[index];
		}
                else if (k >= m_Dim3Size-2) {
                    norm[2] = m_Data[index]-m_Data[index-wh];
		}
                else {
                    norm[2] = (m_Data[index+wh]-m_Data[index-wh])*1;

		    if ((i > 1) && (i < m_Dim1Size-1) && (j > 1) && (j < m_Dim2Size-1)) {
                    norm[2] += (m_Data[index +1 +wh]-m_Data[index +1 -wh])*0.2;
                    norm[2] += (m_Data[index -1 +wh]-m_Data[index -1 -wh])*0.2;
                    norm[2] += (m_Data[index +w +wh]-m_Data[index +w -wh])*0.2;
                    norm[2] += (m_Data[index -w +wh]-m_Data[index -w -wh])*0.2;

                    norm[2] += (m_Data[index +w +1 +wh]-m_Data[index +w +1 -wh])*0.1;
                    norm[2] += (m_Data[index +w -1 +wh]-m_Data[index +w -1 -wh])*0.1;
                    norm[2] += (m_Data[index -w +1 +wh]-m_Data[index -w +1 -wh])*0.1;
                    norm[2] += (m_Data[index -w -1 +wh]-m_Data[index -w -1 -wh])*0.1;
		    }
		}

                len = (float)sqrt((double)((norm[0]*norm[0])+
                                           (norm[1]*norm[1])+
                                           (norm[2]*norm[2])));

                if (len > 0.0f)
                {
                    norm[0] /= len;
                    norm[1] /= len;
                    norm[2] /= len;
                }

                m_Normals[index*3+XDIR] = norm[0];
                m_Normals[index*3+YDIR] = norm[1];
                m_Normals[index*3+ZDIR] = norm[2];

                ++index;
            }
        }
    }
}

//-----------------------------------------------------------------------
//-- Determines the main viewing direction from the viewing vector    ---
//-----------------------------------------------------------------------
void vu1512119::computeMainViewingDirection() {
    mainViewingDir = XDIR;

    double vXsq = m_View[XDIR]*m_View[XDIR];
    double vYsq = m_View[YDIR]*m_View[YDIR];
    double vZsq = m_View[ZDIR]*m_View[ZDIR];

    if ((vXsq >= vYsq) && (vXsq >= vZsq)) {        // main viewing direction: X
	mainViewingDir = XDIR;
    } else if ((vYsq >= vXsq) && (vYsq >=vZsq)) {  // main viewing direction: Y
	mainViewingDir = YDIR;
    } else {                                       // main viewing direction Z
	mainViewingDir = ZDIR;
    }
}

//-----------------------------------------------------------------------
//-- Determines the permutation matrix                                ---
//-----------------------------------------------------------------------
void vu1512119::computePermutationMatrix() 
{
  // depending on the main viewing direction a permutation matrix is choosen
  switch(mainViewingDir) {
      case XDIR: {
	  permMatrix[0][0] = 0; permMatrix[0][1] = 1; permMatrix[0][2] = 0; permMatrix[0][3] = 0;
	  permMatrix[1][0] = 0; permMatrix[1][1] = 0; permMatrix[1][2] = 1; permMatrix[1][3] = 0;
	  permMatrix[2][0] = 1; permMatrix[2][1] = 0; permMatrix[2][2] = 0; permMatrix[2][3] = 0;
	  permMatrix[3][0] = 0; permMatrix[3][1] = 0; permMatrix[3][2] = 0; permMatrix[3][3] = 1;
	  break;
      }
      case YDIR: {
	  permMatrix[0][0] = 0; permMatrix[0][1] = 0; permMatrix[0][2] = 1; permMatrix[0][3] = 0;
	  permMatrix[1][0] = 1; permMatrix[1][1] = 0; permMatrix[1][2] = 0; permMatrix[1][3] = 0;
	  permMatrix[2][0] = 0; permMatrix[2][1] = 1; permMatrix[2][2] = 0; permMatrix[2][3] = 0;
	  permMatrix[3][0] = 0; permMatrix[3][1] = 0; permMatrix[3][2] = 0; permMatrix[3][3] = 1;
	  break;
      }
      case ZDIR: {
	  permMatrix[0][0] = 1; permMatrix[0][1] = 0; permMatrix[0][2] = 0; permMatrix[0][3] = 0;
	  permMatrix[1][0] = 0; permMatrix[1][1] = 1; permMatrix[1][2] = 0; permMatrix[1][3] = 0;
	  permMatrix[2][0] = 0; permMatrix[2][1] = 0; permMatrix[2][2] = 1; permMatrix[2][3] = 0;
	  permMatrix[3][0] = 0; permMatrix[3][1] = 0; permMatrix[3][2] = 0; permMatrix[3][3] = 1;
	  break;
      }
  }
}


//-----------------------------------------------------------------------
//-- Computes the viewing matrix according to the given view- and     ---
//-- up-vectors.                                                      ---
//-----------------------------------------------------------------------
void vu1512119::computeViewMatrix(void)
{
  vuVector puffer;
  vuVector cross;
  vuVector up = m_Up;
  vuVector v = m_View;

  double to_degree = 180/M_PI;  // for dealing with angles in degree
  double alpha = 0;
  double beta = 0;

  vuMatrix rotate_view;  // a rotation matrix to bring the view-vector in correct position 
  vuMatrix rotate_up;    // a rotation matrix to bring the up-vector in correct position

  // init the matrices with the identity
  viewMatrix.makeIdentity();
  rotate_view.makeIdentity();
  rotate_up.makeIdentity();

  // unitVecMain ist the unit vector pointing in the direction of the 
  // main viewing direction which is Z
  vuVector unitVecMain;
  unitVecMain[0] = 0;
  unitVecMain[1] = 0;
  unitVecMain[2] = 1;
  unitVecMain[3] = 1;

  // compute the standard object coordinates for view- and up-vector
  puffer = v;
  v = puffer * permMatrix;
  puffer = up;
  up = puffer * permMatrix;

  // compute the direction-vector of the axis for the following
  // rotation (= v x unitVecMain)
  cross = v.cross(unitVecMain);
  cross.makeUnit();

  // compute angle between view-vector and unit-vector of main viewing
  // direction and set step2-matrix to the according rotation around cross
  v.makeUnit();
  alpha = -to_degree * acos(v[ZDIR]);
  rotate_view.makeRotate(cross,alpha);

  // --> now step2 represents the matrix that turns the viewing vector 
  //     into exactly the same direction as the main viewing vector

  // compute the value of the up vector when it is transformed by steps 1 and 2
  puffer = up;
  up = puffer * rotate_view;

  // compute angle between up-vector (without main-component)
  // and set step2-matrix to the according rotation around the main viewing axis
  up[ZDIR] = 0;
  up.makeUnit();

  // compute the angle for rotation to bring the up vector in correct position
  beta = to_degree * acos(up[YDIR]);
  if (up[XDIR] >= 0) beta *= (-1);
  rotate_up.makeRotateZ(beta);

  // compute final view-matrix and return the value
  viewMatrix = rotate_view * rotate_up;      // at Lacroute this is M'
  worldMatrix = permMatrix * viewMatrix;

  // if perspective: additionally apply projection to viewing matrix
  if (viewingMode == VIEWING_MODE_PERSPECTIVE) {
      projMatrix[0][0] = eye_distance; projMatrix[0][1] = 0; projMatrix[0][2] = 0; projMatrix[0][3] = 0;
      projMatrix[1][0] = 0; projMatrix[1][1] = eye_distance; projMatrix[1][2] = 0; projMatrix[1][3] = 0;
      projMatrix[2][0] = 0; projMatrix[2][1] = 0; projMatrix[2][2] = 1; projMatrix[2][3] = 0;
      projMatrix[3][0] = 0; projMatrix[3][1] = 0; projMatrix[3][2] = 1; projMatrix[3][3] = eye_distance;
      viewMatrix = worldMatrix * projMatrix;
  }
}

//-----------------------------------------------------------------------
//-- Determines width, height and depth of the Volume and saves these ---
//-- values in volumeWidth, volumeHeight and volumeDepth.             ---
//-----------------------------------------------------------------------
void vu1512119::computeRunlengthSizes()
{
    switch(mainViewingDir) {
	case XDIR: {
	    volumeDepth = m_Dim1Size * 2;
	    volumeWidth = m_Dim2Size;
	    volumeHeight = m_Dim3Size / 2;
	    break;
	}
	case YDIR: {
	    volumeDepth = m_Dim2Size * 2;  
	    volumeWidth = m_Dim3Size / 2;  
	    volumeHeight = m_Dim1Size; 
	    break;
	}
	case ZDIR: {
	    volumeDepth = m_Dim3Size;
	    volumeWidth = m_Dim1Size;
	    volumeHeight = m_Dim2Size;
	    break;
	}
    }
}

//-----------------------------------------------------------------------
//-- Computes the Shear Factors si, sj and the Warp Factors ti, tj    ---
//-- for orthogonal projection.                                       ---
//-----------------------------------------------------------------------
void vu1512119::computeShearAndWarpFactorsOrtho()
{
    if ((viewMatrix[0][0]*viewMatrix[1][1]-viewMatrix[1][0]*viewMatrix[0][1]) == 0) si = 1; 
    else si = ((viewMatrix[1][1]*viewMatrix[0][2]-viewMatrix[0][1]*viewMatrix[1][2])/
	       (viewMatrix[0][0]*viewMatrix[1][1]-viewMatrix[1][0]*viewMatrix[0][1]));
    if ((viewMatrix[0][0]*viewMatrix[1][1]-viewMatrix[1][0]*viewMatrix[0][1]) == 0) sj = 1;
    else sj = ((viewMatrix[0][0]*viewMatrix[1][2]-viewMatrix[1][0]*viewMatrix[0][2])/
	       (viewMatrix[0][0]*viewMatrix[1][1]-viewMatrix[1][0]*viewMatrix[0][1]));

    si = si / 2;
    sj = sj / 2;

    if (si >= 0) ti = 0.0; else ti = (-si*volumeDepth);
    if (sj >= 0) tj = 0.0; else tj = (-sj*volumeDepth);
}

//-----------------------------------------------------------------------
//-- Shears the slices for orthogonal viewing                         ---
//-----------------------------------------------------------------------
void vu1512119::shearOrtho()
{
    // normalize the viewing vector
    m_View.makeUnit();

    double shearPos_i;
    double shearPos_j;

    if (orthoWarpOpenGL == 0) {
	shearPos_i = maxSize - ((volumeWidth  + si * volumeDepth) / 2);
	shearPos_j = maxSize - ((volumeHeight + sj * volumeDepth) / 2);
    } else {
	shearPos_i = ti;
	shearPos_j = tj;
    }

    // select the used dataset...
    switch(mainViewingDir) {
	case XDIR: {
	    curr = dataX;
	    break;
	}
	case YDIR: {
	    curr = dataY;
	    break;
	}
	case ZDIR: {
	    curr = dataZ;
	    break;
	}
    }

    if (m_View[mainViewingDir] >= 0) direction = 0; else direction = 1;

    double bcc_correction_dim1 = 0;
    double bcc_correction_dim2 = 0;

    for (dword i = 0; i < volumeDepth; ++i) {
	curr[i].dim1_pos = shearPos_i + bcc_correction_dim1;
	curr[i].dim2_pos = shearPos_j + bcc_correction_dim2;
	shearPos_i += si;
	shearPos_j += sj;

        //folgendes ist sehr umstaendlich!!!
	switch(mainViewingDir) {
	    case XDIR: {
		if (bcc_correction_dim1 == 0) {
		    bcc_correction_dim1 = 0.5;
		    bcc_correction_dim2 = 0.5;
		} else {
		    bcc_correction_dim1 = 0.0;
		    bcc_correction_dim2 = 0.0;
		}
		break;
	    }
	    case YDIR: {
		if (bcc_correction_dim1 == 0) {
		    bcc_correction_dim1 = 0.5;
		    bcc_correction_dim2 = 0.5;
		} else {
		    bcc_correction_dim1 = 0.0;
		    bcc_correction_dim2 = 0.0;
		}
		break;
	    }
	    case ZDIR: {
		if (bcc_correction_dim1 == 0) {
		    bcc_correction_dim1 = 0.5; 
		    bcc_correction_dim2 = 0.5; 
		} else {
		    bcc_correction_dim1 = 0.0;
		    bcc_correction_dim2 = 0.0;
		}
		break;
	    }
	}
    }
}

//-----------------------------------------------------------------------
//-- Computes the length of the view vector normalized in Z           ---
//-----------------------------------------------------------------------
float vu1512119::computeZNormalizedViewingVectorLength() {
    //precompute length of z-normalized viewing vector (later used for opacity correction)
    double z,x,y;
    m_View.makeUnit();
    if      (mainViewingDir == XDIR) { z = m_View[XDIR]; y = m_View[YDIR]; x = m_View[ZDIR]; }
    else if (mainViewingDir == YDIR) { z = m_View[YDIR]; y = m_View[ZDIR]; x = m_View[XDIR]; }
    else                          { z = m_View[ZDIR]; y = m_View[XDIR]; x = m_View[YDIR]; }
    return sqrt((x/z)*(x/z)+(y/z)*(y/z)+1);
}

//-----------------------------------------------------------------------
//-- Produces the intermediate image                                  ---
//-----------------------------------------------------------------------
void vu1512119::makeIntermediateImageOrtho()
{
    dword from = 0;
    dword to = 0;

    dword pos = 0;
    dword add = 0;

    dword intermediateWidth = maxSize * 2;
    dword intermediateSize = intermediateWidth*intermediateWidth;

    int step = 0;

    double l = computeZNormalizedViewingVectorLength();

    //initialize intermediate image
    for (dword i = 0; i < intermediateSize; ++i) {
	intermediate[i].trans = 1;
	intermediate[i].red = 0;
	intermediate[i].green = 0;
	intermediate[i].blue = 0;
	intermediate[i].offset = 0;
    }

    if (direction == 0) { to = volumeDepth-1; step = +1; }
    else { from = volumeDepth-1; step = -1; }

    dword runIndex = 0;
    byte state1 = 0;
    byte state2 = 0;
    byte count1 = 0;
    byte count2 = 0;
    dword voxelIndex = 0;
    dword indexRunlength = 0;

    //for every slice in the runlength encoded volume...
    for (dword i = from; i != to + step; i += step) {

	runIndex = 0;
	state1 = 0;
	state2 = 0;
	count1 = 0;
	count2 = 0;
	voxelIndex = 0;
	indexRunlength = 0;

        //index of the voxels of each of the two voxel scanlines
        dword voxelPtr1 = 0;
	dword voxelPtr2 = 0;

        //index of the current runs
	dword runPtr1 = 0;
	dword runPtr2 = 0;

        //precompute the rounded total shearing (rounding dim1_pos and dim2_pos)
	dword shearX = (dword)(curr[i].dim1_pos);
	dword shearY = (dword)(curr[i].dim2_pos);

        //precompute the bilinear interpolation-coefficients
	double s1 = 1 - (curr[i].dim1_pos - shearX);
	double s2 = 1 - (curr[i].dim2_pos - shearY);

	double w1 = s2-s1*s2;
	double w2 = s1*s2;
	double w3 = 1-s1-s2+s1*s2;
	double w4 = s1-s1*s2;

        vuVector normal;

	if (curr[i].size > 0) {

            // do shading and opacity correction
	    for (dword k = 0; k < curr[i].size; ++k) {
		curr[i].voxel[k].normal.makeUnit();

		float diffuse = m_View.dot(curr[i].voxel[k].normal);

		if (diffuse <= 0) {
		    curr[i].voxel[k].red_shaded = 0.0;
		    curr[i].voxel[k].green_shaded = 0.0;
		    curr[i].voxel[k].blue_shaded = 0.0;
		    curr[i].voxel[k].opacity_corr = 0.0;
		} else {
		    curr[i].voxel[k].opacity_corr = 1 - pow((1-curr[i].voxel[k].opacity),(float)l);

                    // compute the factor that represents opacity correction and shading
		    float fac = curr[i].voxel[k].opacity_corr * (0.1 + 0.9 * diffuse);

		    curr[i].voxel[k].red_shaded = curr[i].voxel[k].red * fac;
		    curr[i].voxel[k].green_shaded = curr[i].voxel[k].green * fac;
		    curr[i].voxel[k].blue_shaded = curr[i].voxel[k].blue * fac;

                    // if specular light turned on, add specular light
		    if (specular == 1) {
		        float spec = pow(diffuse,150);

			curr[i].voxel[k].red_shaded   += spec;
			curr[i].voxel[k].green_shaded += spec;
			curr[i].voxel[k].blue_shaded  += spec;
			curr[i].voxel[k].opacity_corr += spec;

			if (curr[i].voxel[k].red_shaded   > 1.0) curr[i].voxel[k].red_shaded   = 1.0;
			if (curr[i].voxel[k].green_shaded > 1.0) curr[i].voxel[k].green_shaded = 1.0; 
			if (curr[i].voxel[k].blue_shaded  > 1.0) curr[i].voxel[k].blue_shaded  = 1.0;
			if (curr[i].voxel[k].opacity_corr > 1.0) curr[i].voxel[k].opacity_corr = 1.0;
		    }
		}
	    }

            // voxel-pointer 2 is already in right position; find value for voxel-pointer 1
	    do {
		indexRunlength += curr[i].runlength[runPtr1];
		if (state1 == 0) {
		    state1 = 1;
		} else {
		    state1 = 0;
		    voxelPtr1 += curr[i].runlength[runPtr1];
		}
		++runPtr1;
	    } while (indexRunlength < volumeWidth);

	    indexRunlength = 0;

	    pos = intermediateWidth * shearY + shearX;

	    add = 0;

	    dword skip;

	    dword max_intermediate_pos = maxSize*maxSize*4;

            //trace through the slice
	    while((voxelPtr1 < curr[i].size) && (pos < max_intermediate_pos)) {

                //if neccessary, skip zero-length runs
		while (curr[i].runlength[runPtr1] == 0) { 
		    ++runPtr1;
		    count1 = 0;
		    if (state1 == 0) state1 = 1; else state1 = 0;
		}
		while (curr[i].runlength[runPtr2] == 0) {
		    ++runPtr2;
		    count2 = 0;
		    if (state2 == 0) state2 = 1; else state2 = 0;
		}

                //if inside a transparent run, skip voxels
		if ((state1 == 0) && (curr[i].runlength[runPtr1]-count1 > 1) && 
		    (state2 == 0) && (curr[i].runlength[runPtr2]-count2 > 1)) {

		    if (curr[i].runlength[runPtr1]-count1 < curr[i].runlength[runPtr2]-count2) {
			skip = curr[i].runlength[runPtr1]-count1-1;
		    } else {
			skip = curr[i].runlength[runPtr2]-count2-1;
		    }

		    indexRunlength += skip;

		    add += skip;
		    if (add >= volumeWidth) {
			pos += (intermediateWidth - volumeWidth);
			add = add-volumeWidth;
			pos += skip-add;
		    } else {
			pos += skip;
		    }

		    count1 += skip;
		    count2 += skip;

		} else {

                    // only check this pixel, if minimum transparency is given
		    if (intermediate[pos].trans > MIN_TRANSPARENCY_MAKING_INTERMEDIATE) {

			double red = 0;
			double green = 0;
			double blue = 0;
			double opacity = 0;

			if (voxelPtr1 < curr[i].size) {
			    if (state1 == 1) {

				red     += w1*curr[i].voxel[voxelPtr1].red_shaded;
				green   += w1*curr[i].voxel[voxelPtr1].green_shaded;
				blue    += w1*curr[i].voxel[voxelPtr1].blue_shaded;
				opacity += w1*curr[i].voxel[voxelPtr1].opacity_corr;

				if ((curr[i].runlength[runPtr1]-count1 > 1) && 
				    (voxelPtr1+1 < curr[i].size)) {

				    red     += w2*curr[i].voxel[voxelPtr1+1].red_shaded;
				    green   += w2*curr[i].voxel[voxelPtr1+1].green_shaded;
				    blue    += w2*curr[i].voxel[voxelPtr1+1].blue_shaded;
				    opacity += w2*curr[i].voxel[voxelPtr1+1].opacity_corr;

				}

				++voxelPtr1;
			    } else {
			        if ((curr[i].runlength[runPtr1]-count1 == 1) &&   //non-transp. run ends after ptr1?
			            (curr[i].runlength[runPtr1+1]!=0) &&          //no zero-length run after current?
			            ((indexRunlength+1) % volumeWidth)!=0) {   //no line break?

				    red     += w2*curr[i].voxel[voxelPtr1].red_shaded;  // vorher voxelPtr1+1
				    green   += w2*curr[i].voxel[voxelPtr1].green_shaded;
				    blue    += w2*curr[i].voxel[voxelPtr1].blue_shaded;
				    opacity += w2*curr[i].voxel[voxelPtr1].opacity_corr;

				}
			    }
			}

			if (state2 == 1) {
			    red     += w3*curr[i].voxel[voxelPtr2].red_shaded;
			    green   += w3*curr[i].voxel[voxelPtr2].green_shaded;
			    blue    += w3*curr[i].voxel[voxelPtr2].blue_shaded;
			    opacity += w3*curr[i].voxel[voxelPtr2].opacity_corr;

			    if (curr[i].runlength[runPtr2]-count2 > 1) {

				red     += w4*curr[i].voxel[voxelPtr2+1].red_shaded;
				green   += w4*curr[i].voxel[voxelPtr2+1].green_shaded;
				blue    += w4*curr[i].voxel[voxelPtr2+1].blue_shaded;
				opacity += w4*curr[i].voxel[voxelPtr2+1].opacity_corr;

			    }
			    ++voxelPtr2;

			} else {
			    if ((curr[i].runlength[runPtr2]-count2 == 1) &&  //non-transp. run ends after ptr2?
			        (curr[i].runlength[runPtr2+1]!=0) &&         //no zero-length run after current?
			        ((indexRunlength+1) % volumeWidth)!=0) {  //no line break?

				red     += w4*curr[i].voxel[voxelPtr2].red_shaded;  //vorher voxelPtr2+1
				green   += w4*curr[i].voxel[voxelPtr2].green_shaded;
				blue    += w4*curr[i].voxel[voxelPtr2].blue_shaded;
				opacity += w4*curr[i].voxel[voxelPtr2].opacity_corr;

			    }
			}

			if (opacity > 0) {
			    double fac = intermediate[pos].trans;
			    intermediate[pos].offset = 1;

			    intermediate[pos].red   += red * fac;
			    intermediate[pos].green += green * fac;
			    intermediate[pos].blue  += blue * fac;

			    intermediate[pos].trans *= (1-opacity);
			}
 
		    } else {

			if (state1 == 1) {
			    ++voxelPtr1;
			}

			if (state2 == 1) {
			    ++voxelPtr2;
			} 
                        
		    }

                    //one step foreward through the volume
		    ++indexRunlength;

		    ++add;
		    if (add >= volumeWidth) { 
			pos += (intermediateWidth - volumeWidth);
			add = add-volumeWidth;
			++pos;
		    } else {
			++pos;
		    }

		    ++count1;
		    ++count2;

		    if (count1 >= curr[i].runlength[runPtr1]) {
			++runPtr1;
			count1 = 0;
			if (state1 == 1) state1 = 0; else state1 = 1;
		    }

		    if (count2 >= curr[i].runlength[runPtr2]) {
			++runPtr2;
			count2 = 0;
			if (state2 == 1) state2 = 0; else state2 = 1;
		    }
		}
	    }
	}
    }
}

//-----------------------------------------------------------------------
//-- Converts intermediate image to OpenGL-format                     ---
//-----------------------------------------------------------------------
void vu1512119::changeFormatIntermediate() {

    int w = (int)(si*volumeDepth);
    if (w < 0) w *= -1;
    w += volumeWidth;

    int h = (int)(sj*volumeDepth);
    if (h < 0) h *= -1;
    h += volumeHeight;

    int diffX = (int)((glImageWidth-w)/2.0);
    int diffY = (int)((glImageHeight-h)/2.0);

    for (word i = 0; i < glImageWidth; ++i) {
	for (word j = 0; j < glImageHeight; ++j) {
	    dword pos_gl = (i + j*glImageWidth)*4;
	    if ((i-diffX >= 0) && (j-diffY >= 0) && (i-diffX < w) && (j-diffY < h)) {

		dword pos_int = (i-diffX) + (j-diffY)*maxSize*2;

		glImage[pos_gl+0] = (GLubyte)(intermediate[pos_int].red*255);
		glImage[pos_gl+1] = (GLubyte)(intermediate[pos_int].green*255);
		glImage[pos_gl+2] = (GLubyte)(intermediate[pos_int].blue*255);
		glImage[pos_gl+3] = 255;

	    } else {
		glImage[pos_gl+0] = (GLubyte)0;
		glImage[pos_gl+1] = (GLubyte)0;
		glImage[pos_gl+2] = (GLubyte)0;
		glImage[pos_gl+3] = 255;
	    }
	}
    }
}

//-----------------------------------------------------------------------
//-- Draws and warps the intermediate image (warping is done by OpenGL --
//-----------------------------------------------------------------------
void vu1512119::drawWarpOpenGL(void) {
    //the positions of the edges of the OpenGL-polygon
    float pos[4][2];

    //the 2D warping matrix
    float m00 = viewMatrix[0][0];
    float m01 = viewMatrix[0][1];
    float m10 = viewMatrix[1][0];
    float m11 = viewMatrix[1][1];

    //compute the warped positions of the edges of the OpenGL-polygon
    pos[0][XDIR] = 0 * m00 + 0 * m01;
    pos[0][YDIR] = 0 * m10 + 0 * m11;

    pos[1][XDIR] = 1 * m00 + 0 * m01;
    pos[1][YDIR] = 1 * m10 + 0 * m11;

    pos[2][XDIR] = 1 * m00 + 1 * m01;
    pos[2][YDIR] = 1 * m10 + 1 * m11;

    pos[3][XDIR] = 0 * m00 + 1 * m01;
    pos[3][YDIR] = 0 * m10 + 1 * m11;

    //compute the translation of the OpenGL-plane to place 
    //it in the center of the screen
    float t[2];
    t[XDIR] = 0.5 * m00 + 0.5 * m01;
    t[YDIR] = 0.5 * m10 + 0.5 * m11;

    //clear screen, enable texturing...
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    glEnable(GL_TEXTURE_2D);
    glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

    //set parameters for texture mapping
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);

    //create new texture
    glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, glImageWidth,
		 glImageHeight, 0, GL_RGBA, GL_UNSIGNED_BYTE, glImage);

    //work with projection matrix
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();

    // the size of the OpenGL plane with the texture upon it
    float planeSize = 4*zoomValue*zoomValue;  

    // orthogonal projection
    glOrtho(-10,-10,10,10,-1000,1000);
	
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();

    // now draw the quad
    glBegin(GL_QUADS);

    glTexCoord2f(0, 0); glVertex3f(planeSize*(pos[0][XDIR]-t[XDIR]), planeSize*(pos[0][YDIR]-t[YDIR]), 0.0);
    glTexCoord2f(1, 0); glVertex3f(planeSize*(pos[1][XDIR]-t[XDIR]), planeSize*(pos[1][YDIR]-t[YDIR]), 0.0);
    glTexCoord2f(1, 1); glVertex3f(planeSize*(pos[2][XDIR]-t[XDIR]), planeSize*(pos[2][YDIR]-t[YDIR]), 0.0);
    glTexCoord2f(0, 1); glVertex3f(planeSize*(pos[3][XDIR]-t[XDIR]), planeSize*(pos[3][YDIR]-t[YDIR]), 0.0);

    glEnd();

    glFlush();

    glDisable(GL_TEXTURE_2D);
}

//-----------------------------------------------------------------------
//-- Compute the inverse warp matrix for orthogonal viewing           ---
//-----------------------------------------------------------------------
void vu1512119::computeInvWarpOrtho() 
{
    // the 2D-warp-Matrix
    float m00 = viewMatrix[0][0];
    float m01 = viewMatrix[0][1];
    float m10 = viewMatrix[1][0];
    float m11 = viewMatrix[1][1];

    // cmompute the inverse of the 2D-warp-Matrix
    float invDet = 1 / (m00 * m11 - (m01 * m10));

    invWarpOrtho00 = invDet * m11;
    invWarpOrtho01 = - invDet * m01;
    invWarpOrtho10 = - invDet * m10;
    invWarpOrtho11 = invDet * m00;
}

//-----------------------------------------------------------------------
//-- Inverse orthogonal warping of a single point                     ---
//-----------------------------------------------------------------------
void vu1512119::warpOrthoInv(float x, float y, float &x_result, float &y_result)
{
    // apply the inverse warp to the given coordinates
    x_result = invWarpOrtho00 * x + invWarpOrtho01 * y;
    y_result = invWarpOrtho10 * x + invWarpOrtho11 * y;
}

//-----------------------------------------------------------------------
//-- orthogonal warping                                               ---
//-----------------------------------------------------------------------
void vu1512119::warpOrtho()
{
    int glSize = glImageWidth * glImageHeight;
    int glHalf = glImageWidth / 2;

    int intermediateWidth = maxSize * 2;

    float int_x = 0.0f;
    float int_y = 0.0f;

    float gl_x = 0.0f;
    float gl_y = 0.0f;

    float maxSizef = (float)maxSize-1;

    int int_index = 0;
    int gl_index  = 0;
    while(gl_index < glSize) {

	gl_x = (gl_index % glImageWidth) - glHalf;
	gl_y = (gl_index / glImageWidth) - glHalf;

	warpOrthoInv(gl_x, gl_y, int_x, int_y);

	float d1 = (int_x - floor(int_x));
	float d2 = (int_y - floor(int_y));

	float w1 = 1-d1-d2+d1*d2;
	float w2 = d1-d1*d2;
	float w3 = d2-d1*d2;
	float w4 = d1*d2;

	int_index = (int)(floor(int_x + maxSizef) + floor(int_y + maxSizef) * intermediateWidth);


	if ((int_x > -maxSizef) && (int_x < maxSizef-1) &&
	    (int_y > -maxSizef) && (int_y < maxSizef-1)) {


  	    float red = intermediate[int_index].red*w1 + 
 	            intermediate[int_index+1].red*w2 +
 	            intermediate[int_index+intermediateWidth].red*w3 +
	            intermediate[int_index+intermediateWidth+1].red*w4;
	    float green = intermediate[int_index].green*w1 +
 	            intermediate[int_index+1].green*w2 +
 	            intermediate[int_index+intermediateWidth].green*w3 +
  	            intermediate[int_index+intermediateWidth+1].green*w4;
	    float blue = intermediate[int_index].blue*w1 + 
 	            intermediate[int_index+1].blue*w2 +
 	            intermediate[int_index+intermediateWidth].blue*w3 +
 	            intermediate[int_index+intermediateWidth+1].blue*w4;


	    glImage[gl_index*4+0] = (GLubyte)(red*255);
	    glImage[gl_index*4+1] = (GLubyte)(green*255);
	    glImage[gl_index*4+2] = (GLubyte)(blue*255);
	    glImage[gl_index*4+3] = 255;
	} else {
	    glImage[gl_index*4+0] = 0;
	    glImage[gl_index*4+1] = 0;
	    glImage[gl_index*4+2] = 0;
	    glImage[gl_index*4+3] = 255;
	}

	gl_index++;
    }
}

//-----------------------------------------------------------------------
//-- Gets the value of the voxel at the given coordinates             ---
//-----------------------------------------------------------------------
byte vu1512119::getSample(dword x, dword y, dword z)
{
    dword index = x + y * m_Dim1Size + z * m_Dim1Size * m_Dim2Size;
    byte result = 0;
    if (index < m_Dim1Size*m_Dim2Size*m_Dim3Size) {
	result = m_Data[index];
    } 
    return result;
}

//-----------------------------------------------------------------------
//-- Returns the maximum size of the data set                         ---
//-----------------------------------------------------------------------
int vu1512119::getMaxSize() {
    return maxSize;
}

//-----------------------------------------------------------------------
//-- Returns the minimum valid distance                               ---
//-- between eye and projection plane                                 ---
//-----------------------------------------------------------------------
float vu1512119::getMinEyeDistance() {
    return maxSize * 1.0f;
}

//-----------------------------------------------------------------------
//-- sets the distance between eye and projection plane               ---
//-----------------------------------------------------------------------
void vu1512119::setEyeDistance(float distance) {
    eye_distance = distance;
}

//-----------------------------------------------------------------------
//-- Computes the position of the eye point in object space           ---
//-----------------------------------------------------------------------
void vu1512119::computeEyePoint() 
{
    vuVector eye_w;
    eye_w[XDIR] = 0.0f;
    eye_w[YDIR] = 0.0f;
    eye_w[ZDIR] = - eye_distance;
    eye_w[WDIR] = 1.0f;

    vuMatrix invWorldMatrix = worldMatrix.inverse();

    eye_o = eye_w * invWorldMatrix * permMatrix;
}

//-----------------------------------------------------------------------
//-- Computes the Shear Factors si, sj and the Warp Factors ti, tj    ---
//-- for perspective projection.                                      ---
//-----------------------------------------------------------------------
void vu1512119::computeShearPerspective()
{
    //init the shearMatrix with the identity
    shearMatrix.makeIdentity();

    //change the three values in the matrix to get the final shear matrix
    shearMatrix[0][2] = - eye_o[XDIR] / eye_o[ZDIR];
    shearMatrix[1][2] = - eye_o[YDIR] / eye_o[ZDIR];
    shearMatrix[3][2] = - eye_o[WDIR] / eye_o[ZDIR];
}


//-----------------------------------------------------------------------
//-- Shear the planes of the volume for perspective projection        ---
//-----------------------------------------------------------------------
void vu1512119::computeSlicePositionScale(float &pos_i, float &pos_j, float &scale, 
					  int plane)
{
    int halfX = volumeWidth  / 2;
    int halfY = volumeHeight / 2;

    vuVector lu;
    vuVector ru;
    vuVector ll;
    vuVector rl;

    lu = vuVector(-halfX, -halfY, (float)plane/2.0f, 1.0f);  // left upper point of slice
    ru = vuVector( halfX, -halfY, (float)plane/2.0f, 1.0f);   // right upper point of slice
    ll = vuVector(-halfX,  halfY, (float)plane/2.0f, 1.0f);   // left lower point of slice
    rl = vuVector( halfX,  halfY, (float)plane/2.0f, 1.0f);    // right lower point of slice

    vuVector lu_t = lu * shearMatrix;
    vuVector ru_t = ru * shearMatrix;
    vuVector ll_t = ll * shearMatrix;
    vuVector rl_t = rl * shearMatrix;

    // homogenous coordinates! -> divide by 4th coordinate
    for (int i = 0; i < 4; ++i) {
	lu_t[i] = lu_t[i] / lu_t[WDIR];
	ru_t[i] = ru_t[i] / ru_t[WDIR];
	ll_t[i] = ll_t[i] / ll_t[WDIR];
	rl_t[i] = rl_t[i] / rl_t[WDIR];
    }

    if (scale_total < 0) {
	scale_total = (lu_t[XDIR] - rl_t[XDIR]) / (lu[XDIR] - rl[XDIR]);
    }

    scale = ((lu_t[XDIR] - ru_t[XDIR]) / (lu[XDIR] - ru[XDIR])) / scale_total;

    pos_i = lu_t[XDIR] / scale_total + maxSize;
    pos_j = lu_t[YDIR] / scale_total + maxSize;
}

//-----------------------------------------------------------------------
//-- Shear the planes of the volume for perspective projection        ---
//-----------------------------------------------------------------------
void vu1512119::shearPerspective()
{
    // normalize the viewing vector
    m_View.makeUnit();

    // in which direction and from where shall the volume be traversed:
    int step = 0;
    int start = 0;
    int end = 0;

    // select the used dataset...
    switch(mainViewingDir) {
	case XDIR: {
	    curr = dataX;
	    break;
	}
	case YDIR: {
	    curr = dataY;
	    break;
	}
	case ZDIR: {
	    curr = dataZ;
	    break;
	}
    }

    if (m_View[mainViewingDir] >= 0) {
	direction = 0;
	step = 1;
	start = 0;
	end = volumeDepth;
    } else {
	direction = 1;
	step = -1;
	start = volumeDepth-1;
	end = -1;
    }

    int z = 0;

    if (direction == 0) {
	z = -(volumeDepth / 2);
    } else {
	z = volumeDepth / 2;
    }

    // needs to be assigned a negative value -> computeSlicePositionScale
    // computes the total scale-value
    scale_total = -100.0f;

    for (int i = start; i != end; i += step) {
	computeSlicePositionScale(curr[i].dim1_pos,curr[i].dim2_pos,curr[i].scale,z);
	if (direction == 0) z += 1; else z -= 1;
    }
}


//-----------------------------------------------------------------------
//-- Computes the Warp- and the inverse Warp-Matrix for               ---
//-- perspective viewing (scaling of the most front slice is          ---
//-- included in the computation)                                     ---
//-----------------------------------------------------------------------
void vu1512119::computeWarpPerspective()
{
    vuMatrix scaleMatrix;
    scaleMatrix.makeIdentity();
    scaleMatrix.makeScale(scale_total,scale_total,1.0f);

    warpMatrix = scaleMatrix * shearMatrix.inverse() * permMatrix.inverse() * viewMatrix;

    invWarpMatrix = warpMatrix.inverse();
}

//-----------------------------------------------------------------------
//-- inverse perspective warping of a single point                    ---
//-----------------------------------------------------------------------
void vu1512119::warpPerspectiveInv(float x, float y, float &x_result, float &y_result)
{
    vuVector v;
    v[XDIR] = x;
    v[YDIR] = y;
    v[ZDIR] = 0.0f;
    v[WDIR] = 1.0f;

    // apply the inverse warp to the given coordinates
    vuVector result = v * invWarpMatrix;

    // the result must be divided by the 4th coordinate (homogenous coordinates!)
    x_result = result[XDIR] / result[WDIR];
    y_result = result[YDIR] / result[WDIR];
}

//-----------------------------------------------------------------------
//-- perspective warping                                              ---
//-----------------------------------------------------------------------
void vu1512119::warpPerspective()
{
    int glSize = glImageWidth * glImageHeight;
    int glHalf = glImageWidth / 2;

    int intermediateWidth = maxSize * 2;

    float int_x = 0.0f;
    float int_y = 0.0f;

    float gl_x = 0.0f;
    float gl_y = 0.0f;

    float maxSizef = (float)maxSize-1;

    int int_index = 0;
    int gl_index  = 0;
    while(gl_index < glSize) {

	gl_x = (gl_index % glImageWidth) - glHalf;
	gl_y = (gl_index / glImageWidth) - glHalf;

	warpPerspectiveInv(gl_x, gl_y, int_x, int_y);

	float d1 = (int_x - floor(int_x));
	float d2 = (int_y - floor(int_y));

	float w1 = 1-d1-d2+d1*d2;
	float w2 = d1-d1*d2;
	float w3 = d2-d1*d2;
	float w4 = d1*d2;

	int_index = (int)(floor(int_x + maxSizef) + floor(int_y + maxSizef) * intermediateWidth);


	if ((int_x > -maxSizef) && (int_x < maxSizef - 1) &&
	    (int_y > -maxSizef) && (int_y < maxSizef - 1)) {


  	    float red = intermediate[int_index].red*w1 + 
 	            intermediate[int_index+1].red*w2 +
 	            intermediate[int_index+intermediateWidth].red*w3 +
	            intermediate[int_index+intermediateWidth+1].red*w4;
	    float green = intermediate[int_index].green*w1 + 
 	            intermediate[int_index+1].green*w2 +
 	            intermediate[int_index+intermediateWidth].green*w3 +
  	            intermediate[int_index+intermediateWidth+1].green*w4;
	    float blue = intermediate[int_index].blue*w1 + 
 	            intermediate[int_index+1].blue*w2 +
 	            intermediate[int_index+intermediateWidth].blue*w3 +
 	            intermediate[int_index+intermediateWidth+1].blue*w4;


	    glImage[gl_index*4+0] = (GLubyte)(red*255);
	    glImage[gl_index*4+1] = (GLubyte)(green*255);
	    glImage[gl_index*4+2] = (GLubyte)(blue*255);
	    glImage[gl_index*4+3] = 255;
	} else {
	    glImage[gl_index*4+0] = 0;
	    glImage[gl_index*4+1] = 0;
	    glImage[gl_index*4+2] = 0;
	    glImage[gl_index*4+3] = 255;
	}

	gl_index++;
    }
}

//-----------------------------------------------------------------------
//-- Produces the intermediate image for perspective projection       ---
//-----------------------------------------------------------------------
void vu1512119::makeIntermediateImagePerspective()
{
    dword from = 0;
    dword to = 0;

    dword pos = 0;

    dword intermediateWidth = maxSize*2;
    dword intermediateSize = intermediateWidth*intermediateWidth;

    int step = 0;

    double l = computeZNormalizedViewingVectorLength();

    //initialize intermediate image
    for (dword i = 0; i < intermediateSize; ++i) {
	intermediate[i].trans = 1;
	intermediate[i].red = 0;
	intermediate[i].green = 0;
	intermediate[i].blue = 0;
	intermediate[i].offset = 0;
    }

    if (direction == 0) { to = volumeDepth-1; step = +1; }
    else { from = volumeDepth-1; step = -1; }

    dword runIndex = 0;
    byte state1 = 0;
    byte state2 = 0;
    byte count1 = 0;
    byte count2 = 0;
    dword voxelIndex = 0;
    dword indexRunlength = 0;

    //for every slice in the runlength encoded volume...
    for (dword i = from; i != to + step; i += step) {

	runIndex = 0;
	state1 = 0;
	state2 = 0;
	count1 = 0;
	count2 = 0;
	voxelIndex = 0;
	indexRunlength = 0;

        //index of the voxels of each of the two voxel scanlines
        dword voxelPtr1 = 0;
	dword voxelPtr2 = 0;

        //index of the current runs
	dword runPtr1 = 0;
	dword runPtr2 = 0;

        vuVector normal;

	if (curr[i].size > 0) {

            // do the shading and the opacity correction
	    for (dword k = 0; k < curr[i].size; ++k) {
		curr[i].voxel[k].normal.makeUnit();

		float diffuse = m_View.dot(curr[i].voxel[k].normal);

		float spec = 0;

		if (diffuse <= 0) { 
		    curr[i].voxel[k].red_shaded = 0.0;
		    curr[i].voxel[k].green_shaded = 0.0;
		    curr[i].voxel[k].blue_shaded = 0.0;
		    curr[i].voxel[k].opacity_corr = 0.0;
		} else {
		    curr[i].voxel[k].opacity_corr = 1 - pow((1-curr[i].voxel[k].opacity),(float)l);

		    float fac = curr[i].voxel[k].opacity_corr * (0.1 + 0.9 * diffuse);

		    curr[i].voxel[k].red_shaded = curr[i].voxel[k].red * fac;
		    curr[i].voxel[k].green_shaded = curr[i].voxel[k].green * fac;
		    curr[i].voxel[k].blue_shaded = curr[i].voxel[k].blue * fac;

                    // if specular light turned on, add it
		    if (specular == 1) {
			spec = pow(diffuse,150);

			curr[i].voxel[k].red_shaded   += spec;
			curr[i].voxel[k].green_shaded += spec;
			curr[i].voxel[k].blue_shaded  += spec;
			curr[i].voxel[k].opacity_corr += spec;

			if (curr[i].voxel[k].red_shaded > 1.0) curr[i].voxel[k].red_shaded = 1.0;
			if (curr[i].voxel[k].green_shaded > 1.0) curr[i].voxel[k].green_shaded = 1.0; 
			if (curr[i].voxel[k].blue_shaded > 1.0) curr[i].voxel[k].blue_shaded = 1.0;
			if (curr[i].voxel[k].opacity_corr > 1.0) curr[i].voxel[k].opacity_corr = 1.0;
		    }
		}
	    }

            //first voxel-pointer 2 is already in right position; find value for voxel-pointer 1
	    do {
		indexRunlength += curr[i].runlength[runPtr1];
		if (state1 == 0) {
		    state1 = 1;
		} else {
		    state1 = 0;
		    voxelPtr1 += curr[i].runlength[runPtr1];
		}
		++runPtr1;
	    } while (indexRunlength < volumeWidth);

	    indexRunlength = 0;

	    dword skip;

            //trace through the slice
	    while(voxelPtr1 < curr[i].size) {

                //if neccessary, skip zero-length runs
		while (curr[i].runlength[runPtr1] == 0) { 
		    ++runPtr1;
		    count1 = 0;
		    if (state1 == 0) state1 = 1; else state1 = 0;
		}
		while (curr[i].runlength[runPtr2] == 0) {
		    ++runPtr2;
		    count2 = 0;
		    if (state2 == 0) state2 = 1; else state2 = 0;
		}

                //if inside a transparent run, skip voxels
		if ((state1 == 0) && (curr[i].runlength[runPtr1]-count1 > 1) && 
		    (state2 == 0) && (curr[i].runlength[runPtr2]-count2 > 1)) {

		    if (curr[i].runlength[runPtr1]-count1 < curr[i].runlength[runPtr2]-count2) {
			skip = curr[i].runlength[runPtr1]-count1-1;
		    } else {
			skip = curr[i].runlength[runPtr2]-count2-1;
		    }

		    indexRunlength += skip;

		    count1 += skip;
		    count2 += skip;
		} else {

		    word xpos = indexRunlength % volumeWidth;
		    word ypos = indexRunlength / volumeWidth;

		    // positions of the left upper and right lower sample for the intermediate image
		    float sx1 = curr[i].dim1_pos + (float)xpos * curr[i].scale;
		    float sy1 = curr[i].dim2_pos + (float)ypos * curr[i].scale;
		    float sx4 = curr[i].dim1_pos + (float)(xpos+1) * curr[i].scale;
		    float sy4 = curr[i].dim2_pos + (float)(ypos+1) * curr[i].scale;

		    pos = intermediateWidth * ((dword)sy4) + ((dword)sx4);

                    // only check this pixel, if it still is transparent
		    if ((intermediate[pos].trans > MIN_TRANSPARENCY_MAKING_INTERMEDIATE) &&
			(floor(sx1) != floor(sx4)) && (floor(sy1) != floor(sy4))) {

			double red = 0;
			double green = 0;
			double blue = 0;
			double opacity = 0;


                        //precompute the bilinear interpolation-coefficients
			double s1 = 1 - (sx1 - (int)sx1);
			double s2 = 1 - (sy1 - (int)sy1);

			s1 /= curr[i].scale;
			s2 /= curr[i].scale;

			double w1 = s2-s1*s2;
			double w2 = s1*s2;
			double w3 = 1-s1-s2+s1*s2;
			double w4 = s1-s1*s2;

			if (voxelPtr1 < curr[i].size) {
			    if (state1 == 1) {

				red     += w1*curr[i].voxel[voxelPtr1].red_shaded;
				green   += w1*curr[i].voxel[voxelPtr1].green_shaded;
				blue    += w1*curr[i].voxel[voxelPtr1].blue_shaded;
				opacity += w1*curr[i].voxel[voxelPtr1].opacity_corr;

				if ((curr[i].runlength[runPtr1]-count1 > 1) &&
				    (voxelPtr1+1 < curr[i].size)) {

				    red     += w2*curr[i].voxel[voxelPtr1+1].red_shaded;
				    green   += w2*curr[i].voxel[voxelPtr1+1].green_shaded;
				    blue    += w2*curr[i].voxel[voxelPtr1+1].blue_shaded;
				    opacity += w2*curr[i].voxel[voxelPtr1+1].opacity_corr;

				}
				++voxelPtr1;
			    } else {
			        if ((curr[i].runlength[runPtr1]-count1 == 1) &&   //non-transp. run ends after ptr1?
			            (curr[i].runlength[runPtr1+1]!=0) &&          //no zero-length run after current?
			            ((indexRunlength+1) % volumeWidth)!=0) {   //no line break?

				    red     += w2*curr[i].voxel[voxelPtr1].red_shaded;  // vorher voxelPtr1+1
				    green   += w2*curr[i].voxel[voxelPtr1].green_shaded;
				    blue    += w2*curr[i].voxel[voxelPtr1].blue_shaded;
				    opacity += w2*curr[i].voxel[voxelPtr1].opacity_corr;

				}
			    }
			}

			if (state2 == 1) {

			    red     += w3*curr[i].voxel[voxelPtr2].red_shaded;
			    green   += w3*curr[i].voxel[voxelPtr2].green_shaded;
			    blue    += w3*curr[i].voxel[voxelPtr2].blue_shaded;
			    opacity += w3*curr[i].voxel[voxelPtr2].opacity_corr;


			    if (curr[i].runlength[runPtr2]-count2 > 1) {

				red     += w4*curr[i].voxel[voxelPtr2+1].red_shaded;
				green   += w4*curr[i].voxel[voxelPtr2+1].green_shaded;
				blue    += w4*curr[i].voxel[voxelPtr2+1].blue_shaded;
				opacity += w4*curr[i].voxel[voxelPtr2+1].opacity_corr;

			    }
			    ++voxelPtr2;
			} else {
			    if ((curr[i].runlength[runPtr2]-count2 == 1) &&  //non-transp. run ends after ptr1?
				(curr[i].runlength[runPtr2+1]!=0) &&         //no zero-length run after current?
				((indexRunlength+1) % volumeWidth)!=0) {  //no line break?

				red     += w4*curr[i].voxel[voxelPtr2].red_shaded;  //vorher voxelPtr2+1
				green   += w4*curr[i].voxel[voxelPtr2].green_shaded;
				blue    += w4*curr[i].voxel[voxelPtr2].blue_shaded;
				opacity += w4*curr[i].voxel[voxelPtr2].opacity_corr;
			    }
			}


			if (opacity > 0) {

			    intermediate[pos].offset = 1;

			    double fac = intermediate[pos].trans;

			    intermediate[pos].red   += red * fac;
			    intermediate[pos].green += green * fac;
			    intermediate[pos].blue  += blue * fac;

			    intermediate[pos].trans *= (1-opacity);
			}

		    } else {

			if (state1 == 1) {
			    ++voxelPtr1;
			}

			if (state2 == 1) {
			    ++voxelPtr2;
			} 

		    }

                    //one step foreward through the volume
		    ++indexRunlength;

		    ++count1;
		    ++count2;

		    if (count1 >= curr[i].runlength[runPtr1]) {
			++runPtr1;
			count1 = 0;
			if (state1 == 1) state1 = 0; else state1 = 1;
		    }

		    if (count2 >= curr[i].runlength[runPtr2]) {
			++runPtr2;
			count2 = 0;
			if (state2 == 1) state2 = 0; else state2 = 1;
		    }
		}
	    }
	}
    }
}

//-----------------------------------------------------------------------
//-- Draws the perspective intermediate Image using OpenGL            ---
//-----------------------------------------------------------------------
void vu1512119::drawOpenGL() 
{
    //clear screen, enable texturing...
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    glEnable(GL_TEXTURE_2D);
    glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

    //set parameters for texture mapping
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);

    //create new texture
    glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, glImageWidth,
		 glImageHeight, 0, GL_RGBA, GL_UNSIGNED_BYTE, glImage);

    //work with projection matrix
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();

    // the size of the OpenGL plane with the texture upon it
    float planeSize = 40*zoomValue*zoomValue;  

    glOrtho(-10,10,-10,10,-1000,1000);

    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();

    glBegin(GL_QUADS);

    glTexCoord2f(0, 0); glVertex3f(planeSize*(-1), planeSize*(-1), 0.0);
    glTexCoord2f(1, 0); glVertex3f(planeSize*(1), planeSize*(-1), 0.0);
    glTexCoord2f(1, 1); glVertex3f(planeSize*(1), planeSize*(1), 0.0);
    glTexCoord2f(0, 1); glVertex3f(planeSize*(-1), planeSize*(1), 0.0);

    glEnd();

    glFlush();

    glDisable(GL_TEXTURE_2D);
}

//-----------------------------------------------------------------------
//-- Initializes OpenGL                                               ---
//-----------------------------------------------------------------------
void vu1512119::initOpenGL(void)
{
    // selecting background color
    glClearColor(0.0, 0.0, 0.0, 0.0);

    glPixelStorei(GL_UNPACK_ALIGNMENT, 1);

    glGenTextures(1, &m_GLShearWarp);

    glBindTexture(GL_TEXTURE_2D, m_GLShearWarp);

    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);

    glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, glImageWidth,
		 glImageHeight, 0, GL_RGBA, GL_UNSIGNED_BYTE, glImage);
}


//-----------------------------------------------------------------------
//-- Renders the volume                                               ---
//-----------------------------------------------------------------------
void vu1512119::render() 
{
    computeMainViewingDirection();
    computePermutationMatrix();
    computeViewMatrix();
    computeRunlengthSizes();

    switch(viewingMode) {
	case VIEWING_MODE_ORTHO: {
	    computeShearAndWarpFactorsOrtho();
	    shearOrtho();
	    computeInvWarpOrtho();
	    makeIntermediateImageOrtho();

	    if (orthoWarpOpenGL == 0) {
		warpOrtho();
		drawOpenGL();
	    } else {
		changeFormatIntermediate();
		drawWarpOpenGL();
	    }

	    break;
	}
	case VIEWING_MODE_PERSPECTIVE: {
	    computeEyePoint();
	    computeShearPerspective();
	    shearPerspective();
	    computeWarpPerspective();
	    makeIntermediateImagePerspective();
	    warpPerspective();
	    drawOpenGL();
	    break;
	}
    }
}



























