//plain bcc raycaster
#include <GL/gl.h>
#include <fstream.h>
#include <math.h>
#include "vuParallelCamera.h"
#include "vuPerspectiveCamera.h"

#include "raycaster.h"

/** if enabled alpha will be computed more correctly (using the
 * formula below) */
#define DO_EXPALPHA
/** Since we use 1-exp(-alpha*smpdist*EXPALPHA) it is not possible to reach
    opacity values of one. 'alpha' is restricted to [0,1], the EXPALHA coefficient
    helps to enlarge this interval. */
#define EXPALPHA	3
#define SHININESS       10
#define GLOSS           0.2
//----------------------------------------------------------------------------
//------------------------- The default constructor --------------------------
//----------------------------------------------------------------------------

vu1512113::vu1512113() : m_LightDir(0,0,0), m_Specular(1,1,1,0)
{
  m_Camera = new vuPerspectiveCamera();
  m_Camera->setHeight(240);
  m_Camera->setWidth(320);
  refresh = false;
  m_SamplingDistance = 1.0f;
}

//----------------------------------------------------------------------------
//------------------------- The copy constructor -----------------------------
//----------------------------------------------------------------------------

vu1512113::vu1512113(const vu1512113& inst) : vu151211(inst)
{
}

//----------------------------------------------------------------------------
//------------------------- The destructor -----------------------------------
//----------------------------------------------------------------------------

vu1512113::~vu1512113()
{
}

//----------------------------------------------------------------------------
//------------------------- The assignment operator --------------------------
//----------------------------------------------------------------------------

vu1512113& vu1512113::operator=(const vu1512113& rhs)
{
    if (this != &rhs)
    {
        vu151211::operator=(rhs);

    }
    return *this;
}

void vu1512113::setImageSize(int sx, int sy) 
{
  if(m_Camera == NULL) {
      cerr << "no camera set" << endl;
      return;
  }
  if (m_Camera->getType() == vuCamera::vuPERSPECTIVE_CAMERA) {
    vuPerspectiveCamera *cptr = (vuPerspectiveCamera *)m_Camera;
    cptr->setAspect(float(sx)/sy);
  }
  m_Camera->setWidth(sx);
  m_Camera->setHeight(sy);
  m_Camera->init();
  m_Image.init(m_Camera->getWidth(),m_Camera->getHeight());
}

//----------------------------------------------------------------------------
//------------------------- public setViewVectors() --------------------------
//----------------------------------------------------------------------------

void vu1512113::setViewVectors(const vuVector& view,const vuVector& up,const vuVector& right)
{
//     m_View = view;
//     m_Shift1 = right*3.2f;
//     m_Shift2 = up*3.2f;
//     m_Shift0 = (m_Shift1+m_Shift2)*(-0.5f);

    // maybe we use the camera at some point (so far it's not used)
    m_Camera->setLookAtVector(view);
    m_Camera->setUpVector(up);
    m_Camera->setRightVector(right);
    m_Camera->init();
}

//----------------------------------------------------------------------------
//------------------------- public read() ------------------------------------
//----------------------------------------------------------------------------

bool vu1512113::read()
{
    bool success = vu151211::read();
    if (!success) return false;

    vuVector center(getCenter());
    m_Camera->setPosition(center);
    m_Camera->translateXYZ(0,0,-1.5f*(float)m_Dim1Size);
    m_Camera->init();

    preprocess();
    return true;
}

//----------------------------------------------------------------------------
//------------------------- public readRaw() ---------------------------------
//----------------------------------------------------------------------------

bool vu1512113::readRaw(void)
{
    dword len;
    ifstream in;

#ifdef IS_NOCREATE_NEEDED
    in.open(m_FileName, ios::in|ios::binary|ios::nocreate);
#else
	// The nocreate is not available on the IRIX boxes that we were compiling
	// this code on, so we had to remove this part from
    in.open(m_FileName, ios::in|ios::binary);
#endif

    if (!in.is_open()) return false;

    in.seekg(0, ios::end);
    len = in.tellg();
    in.seekg(0, ios::beg);

    in >> m_Dim1Size >> m_Dim2Size >> m_Dim3Size;
    if (in.fail()) return false;
    m_DataSize = m_Dim1Size*m_Dim2Size*m_Dim3Size;

    m_Data = new byte[m_DataSize];
    in.read((char *) (m_Data), int (m_DataSize));
    if (in.fail()) return false;

    in.close();

    preprocess();

    return true;
}

//----------------------------------------------------------------------------
//------------------------- private preprocess() -----------------------------
//----------------------------------------------------------------------------

void vu1512113::preprocess(void)
{
  // compute normals
  cout<<"preprocessing..."<<endl;
}

//----------------------------------------------------------------------------
//------------------------- public render() ----------------------------------
//----------------------------------------------------------------------------

void vu1512113::render(void)
{
  if (refresh)
    {
	glClearColor(0.0,0.0,0.0,0);
	glClear(GL_COLOR_BUFFER_BIT);
      

      renderImage();

      refresh = false;
    }
  displayFromImage();
  glFlush();
}

//----------------------------------------------------------------------------
//------------------------- public initOpenGL() ------------------------------
//----------------------------------------------------------------------------

void vu1512113::initOpenGL(void)
{
  //glEnable(GL_SECRET_FEATURE);
}

//----------------------------------------------------------------------------
//------------------------- private intersectRayWithBox() --------------------
//----------------------------------------------------------------------------

bool vu1512113::intersectRayWithBox(vuVector Ro, vuVector Rd, vuVector *f, vuVector *s, vuVector *llc, vuVector *urc)
{
  double tnear=-1.7E308,tfar=1.7E308,t1,t2;
  
  Rd.makeUnit();

  //
  // intersect ray with X slab
  //
  
  if (Rd[0] == 0.0)	// ray is parallel to YZ-plane
    {
      if ((Ro[0] < (*llc)[0]) || (Ro[0] > (*urc)[0]))
	return false;  //ray does not intersect data set
    }
  else
    {
      t1=((*llc)[0]-Ro[0])/Rd[0];
      t2=((*urc)[0]-Ro[0])/Rd[0];
      
      if (t1 < t2)
	{
	  tnear=t1;
	  tfar=t2;
	}
      else
	{
	  tnear=t2;
	  tfar=t1;
	}
    }
  
  //
  // intersect ray with Y slab
  //
  
  if (Rd[1] == 0.0)	// ray is parallel to XZ-plane
    {
      if ((Ro[1] < (*llc)[1]) || (Ro[1] > (*urc)[1]))
	return false;	// ray does not intersect data set
    }
  else
    {
      t1=((*llc)[1]-Ro[1])/Rd[1];
      t2=((*urc)[1]-Ro[1])/Rd[1];

      if (t1 > t2)
	{
	  double zw=t1;
	  t1=t2;
	  t2=zw;
	}
      
      if (t1 > tnear) tnear=t1;
      if (t2 < tfar)  tfar =t2;
      
      if (tfar < tnear)
	  return false;	// ray misses box
    }
  
  //
  // intersect ray with Z slab
  //
  
  if (Rd[2] == 0.0)	// ray is parallel to XY-plane
    {
      if ((Ro[2] < (*llc)[2]) || (Ro[2] > (*urc)[2]))
	return false; // ray does not intersect data set
    }
  else
    {
      t1=((*llc)[2]-Ro[2])/Rd[2];
      t2=((*urc)[2]-Ro[2])/Rd[2];
      
      if (t1 > t2)
	{
	  double zw=t1;
	  t1=t2;
	  t2=zw;
	}
      
      if (t1 > tnear) tnear=t1;
      if (t2 < tfar)  tfar =t2;
      
      if (tfar < tnear)
	  return false;	// ray misses box
    }

  vuVector Rd1(Rd);
  Rd *= tnear;
  *f = Ro + Rd;
  Rd1 *= tnear;
  *s = Ro + Rd;

  //PNT3 f1, s1;
  //VEC3 Rd1;
  //CopyVEC3(&Rd1,&Rd);
  //
  //MulVEC3Scal(&Rd,tnear);
  //AddPNT3VEC3(&f1,&Ro,&Rd);
  //
  //MulVEC3Scal(&Rd1,tfar);
  //AddPNT3VEC3(&s1,&Ro,&Rd1);
  //
  //(*f)[0] = f1.x;
  //(*f)[1] = f1.y;
  //(*f)[2] = f1.z;
  //
  //(*s)[0] = s1.x;
  //(*s)[1] = s1.y;
  //(*s)[2] = s1.z;

  return true;
}

//----------------------------------------------------------------------------
//------------------------- private renderImage() ----------------------------
//----------------------------------------------------------------------------

void vu1512113::renderImage()
{
  if(m_Camera == NULL) {
      cerr << "no camera found" << endl;
      return;
  }
  m_Image.init(m_Camera->getWidth(), m_Camera->getHeight());

  bool usingCameraLight=false;
  if(m_LightDir.norm2() == 0.0f) {
      m_LightDir = m_Camera->getLookAtVector();
      usingCameraLight=true;
  }
//  double pz = 1.3;  // pseudo-zoom
//  float PI=3.14159265359;
  int XRES=m_Camera->getWidth();
  int YRES=m_Camera->getHeight();
  unsigned short maxx = m_Dim1Size, maxy = m_Dim2Size, maxz = m_Dim3Size;

  vuVector l(m_Camera->getLookAtVector());
  l.makeUnit();

  vuVector dataSetLowerBound(0.0f, 0.0f, 0.0f);
  vuVector dataSetUpperBound((float)maxx, (float)maxy, (float)maxz);
  //vuVector dataSetUpperBound(64.0, 64.0, 64.0);

  vuVector *dslb = &dataSetLowerBound;
  vuVector *dsub = &dataSetUpperBound;

  //
  // shoot ray through each pixel
  //
  glBegin(GL_POINTS);
  for (int i=0; i<XRES; i++)
    {
      for (int j=0; j<YRES; j++)
	{
          vuRay ray(m_Camera->getRay((float)i,(float)j));
	  vuVector pos,pos2;

	  double val = 0.0;
	  vuColourRGBa colour(0.0);
	  float aalpha = 1;		// 1 - accumulated alpha
	  // traverse ray
	  if (intersectRayWithBox(ray.m_Position, ray.m_Direction, &pos, &pos2, dslb, dsub))
	  {
	      ray.m_Position = pos;
	      ray.advance();
	      bool terminate=false;
	      while ( (ray.m_Position[0] <= (*dsub)[0]) && (ray.m_Position[0] >= (*dslb)[0]) &&
		      (ray.m_Position[1] <= (*dsub)[1]) && (ray.m_Position[1] >= (*dslb)[1]) &&
		      (ray.m_Position[2] <= (*dsub)[2]) && (ray.m_Position[2] >= (*dslb)[2]) && 
		      !terminate )
	      {
		  val =getDataValue(ray.m_Position);
		  vuColourRGBa scol = m_TFunc[(int)val]; // sample colour
		  vuVector grad = getGradient(ray.m_Position);
		  grad.makeUnit();
		  
		  if(scol.getAlpha() > 0.001)
		  {
#ifdef DO_EXPALPHA
		      scol.setAlpha(1-exp(-scol.getAlpha()*m_SamplingDistance*EXPALPHA));
#endif    
		      // diffuse
		      float diffuse = grad.dot(m_LightDir);
		      if(diffuse<0.0f) { // two sided lighting
			  diffuse=-diffuse;
			  grad*=-1.0f;
		      }
		      scol *= diffuse;
		      
		      //do we always want to compute specular?
		      vuVector r(ray.m_Direction);
		      r -=  2*grad*grad.dot(r);
		      r.makeUnit();
		      float spec=r.dot(m_LightDir);
		      if(spec>0)
		      {
			  spec=(float)pow((r.dot(m_LightDir)),SHININESS)*GLOSS;
			  scol += m_Specular*spec;
		      }
			  
		      colour += scol*(scol.getAlpha()*aalpha);
		      aalpha *= 1-scol.getAlpha();
		      
		      if(aalpha<0.05)
			  terminate = true;
		  }
		  ray.advance();
	      }
	  }
	  
	  m_Image.set_xy(i,YRES-j, colour);
	  //colour.glColor();
	  //glVertex2f(i,YRES-j);
	}
      //printf("row %d\n",i);
      cout << "." << flush;
    }
    glEnd();
    if(usingCameraLight)
	m_LightDir = 0.0;
    cout << " done" << endl;
}

void vu1512113::displayFromImage ()

{
  int mx, my;

  m_Image.get_extents (mx, my);

  if(mx && my)
      m_Image.blit();
  else
    cout << "bad" << endl;
}
