#include <stdio.h>
#include "vuMisc/vuCommandLineTool.h"
#include "vuLightfield/Converter/vuSphericLightfieldConverter.h"
#include "vuFile/vuFileHelper.h"

/* 
   Usage: vul2vul --type=3B input.vul output.vul
*/

vuString g_ErrorMsg;
bool     g_IsVerbose = true;
vuString g_Type;
vuString g_OutputFile;
vuString g_InputFile;

vuBasicLightfieldConverter *g_Converter = NULL;

bool _isConverterAvailable(vuString &file)
{
  vuString type = vuFileHelper::getFileType(file);
  if (!type.hasPrefix("1611") || (type.getLength() < 5)) {
    g_ErrorMsg += "  - Input file is not a spherical lightfield file.\n";
    return false;
  }

  const char chr = type.operator[](4);
  switch (chr) {
    case '1': type = "1B"; break;
    case '2': type = "2B"; break;
    case '3': type = "3B"; break;
            
    case 'A': type = "1F"; break;
    case 'B': type = "2F"; break;
    case 'C': type = "3F"; break;
    default: {
      g_ErrorMsg += "  - Don't know type '";
      g_ErrorMsg += type;
      g_ErrorMsg += "'.";
      return false;
    }
  }

  type += g_Type;

  g_Converter = vuSphericLightfieldConverterFactory::getConverter(type);
  if (g_Converter == NULL) {
    g_ErrorMsg += "  - Could not find converter for type '";
    g_ErrorMsg += type;
    g_ErrorMsg += "'\n";
    return false;
  }
  return true;
}

void destroy()
{
  CHECKNDELETE(g_Converter);
}

void convert()
{  
  g_Converter->convert(g_InputFile, g_OutputFile, g_IsVerbose);
}

/* ------------------------------------------------------------------------ */
/* --- private parameter parsing functions -------------------------------- */
/* ------------------------------------------------------------------------ */


vuString _helpString(vuCommandLineTool &tool)
{
  vuString str;


  str += "Converting spherical lightfields to different types,";
  str += " e.g. 3 byte -> 2 float\n\n";

  str += "Usage: ";
  str += tool.toolName();
  str += " --type=tt inputFile outputFile\n\n";
  str += "  --type=tt    result type (1B,2B,3B,1F,2F or 3F),   e.g.";
  str += " --type=1B\n";
  str += "  inputFile    a vuVolume spherical lightfield file, e.g.";
  str += " engine_3B.vul\n";
  str += "  outputFile   a vuVolume spherical lightfield file, e.g.";
  str += " engine_1F.vul\n";
 
  return str;
}

bool _parseParameters(int argc, const char **argv)
{
  bool isOk = true;
  vuCommandLineTool tool(argc, argv);

  g_Type = tool.stringForParameter("--type");

  if (tool.hasParameter("--help")) {
    g_ErrorMsg += _helpString(tool);
    return false;
  }

  g_ErrorMsg += "Following error(s) occured:\n";

  if (!tool.hasParameter("--type")) {
    g_ErrorMsg += "  - Result type not set (use '--type=3B').\n";
    isOk = false;
  }
  else  if (g_Type != "1B" && g_Type != "2B" && g_Type != "3B" &&
	    g_Type != "1F" && g_Type != "2F" && g_Type != "3F") {
    g_ErrorMsg += "  - Result type is not properly set. ";
    g_ErrorMsg += "(Chose from 1B, 2B, 3B, 1F, 2F or 3F)\n";
    isOk = false;
  }

  bool isValid;
  word fileCount = tool.numberOfNonParameters(isValid);

  if (fileCount == 0) {
    g_ErrorMsg += "  - Neither an input nor an output file is specified.\n";
    isOk = false;
  }
  else if (fileCount == 1)  {
    g_ErrorMsg += "  - No output file specified.\n";
    isOk = false;
  }
  else if (fileCount == 2 && isValid)  {
    g_InputFile  = tool.getArgument(argc-2);
    g_OutputFile = tool.getArgument(argc-1);
    if (!tool.fileExists(g_InputFile)) {
      g_ErrorMsg += "  - InputFile does not exist ('";
      g_ErrorMsg += g_InputFile + "').\n";
      isOk = false;
    }
    else if (!_isConverterAvailable(g_InputFile)) {
      // g_ErrorMsg set by _isConverterAvailable()
      isOk = false;
    }
    if (g_OutputFile.isEmpty()) {
      g_ErrorMsg += "  - No output file specified.\n";
      isOk = false;
    }
  }
  else if (fileCount > 2)  {
    g_ErrorMsg += "  - More than one input and one output file specified.\n";
    isOk = false;
  }
  else {
    g_ErrorMsg += "  - The input and output file are expected to be at the ";
    g_ErrorMsg += " end of the line.\n";
    isOk = false;
  }

  g_ErrorMsg += "\nType '" + tool.toolName() + " --help' for more information!\n";

  if (isOk) g_ErrorMsg = "";

  return isOk;
}

/* ------------------------------------------------------------------------ */
/* --- main function ------------------------------------------------------ */
/* ------------------------------------------------------------------------ */

int main(int argc, const char **argv)
{
  if (!_parseParameters(argc, argv)) {
    cerr << g_ErrorMsg << endl;
    exit(0);
  }

  convert();

  destroy();

  return 0;
}
