#include "FourierSlicer.h"
#include <stdio.h>
#include "vuMisc/vuCommandLineTool.h"
#include "vuFile/vuFileHelper.h"
#include <GL/glut.h>

/* 
   Usage: slicer OPTIONS input.vuf output.vul

   OPTIONS: --views=100 [--show] [--scale=1] [--bias=0] [--filter=d0_c3_2ef]
*/

FourierSlicer_ *g_Slicer     = NULL;
vuString        g_ErrorMsg;
vuString        g_OutputFile;
vuString        g_InputFile;
vuString        g_TimingFileName;
vuString        g_FilterName = "d0_c3_2ef";
float           g_Scale      = 1.0;
float           g_Bias       = 0.0;
int             g_NumOfViews = -1;
int             g_Width      = -1;
int             g_Height     = -1;
bool            g_Visualize  = false;

void init(void) {
  vuString fileType = vuFileHelper::getFileType(g_InputFile.c_str());

  if (fileType == "1712A") {
    g_Slicer = new FourierSlicer<1>(g_InputFile.c_str(),
				    g_NumOfViews,
				    g_Scale,
				    g_Bias,
				    g_FilterName,
				    g_TimingFileName);
  }
  else if (fileType == "1712B") {
    g_Slicer = new FourierSlicer<2>(g_InputFile.c_str(),
				    g_NumOfViews,
				    g_Scale,
				    g_Bias,
				    g_FilterName,
				    g_TimingFileName);
  }
  else if (fileType == "1712C") {
    g_Slicer = new FourierSlicer<3>(g_InputFile.c_str(),
				    g_NumOfViews,
				    g_Scale,
				    g_Bias,
				    g_FilterName,
				    g_TimingFileName);
  }
  else {
    cerr << "\nError: fileType '" <<fileType<< "' is not supported!\n" << endl;
    exit(0);
  }
  g_Width  = g_Slicer->getImageWidth();
  g_Height = g_Slicer->getImageHeight();
}

void destroy(void) {
  delete g_Slicer;
}

//*****************************************//
//**** The glut stuff                  ****//
//*****************************************//

void display(void)
{
  g_Slicer->lazyCalculateAndDisplay(g_OutputFile);
}

void keyboard(unsigned char key, int x, int y)
{
  switch(key)
    {
    case 27:
    case 'q':
    case 'Q':
      destroy();
      exit(0);
      break;
    default:
      break;
    }
  glutPostRedisplay();
}

void reshape(int width, int height)
{

  glViewport(0, 0, width, height);

  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective(60.0f, 1.0f, 0.1f, 10.0f);
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  glPixelZoom((float)width/g_Width,(float)height/g_Height);
  glTranslated(0.0, 0.0, -2.5);
}

vuString _helpString(vuCommandLineTool &tool)
{
  vuString str;


  str += "Creating spherical lightfields with arbitrary number of channels"
         " using the Fourier Volume Rendering technique.\n\n";
  str += "Usage: ";
  str += tool.toolName();
  str += " mandatory_options [additional_options] inputFile outputFile\n";
  str += "\nmandatory_options:\n";
  str += "  --views=nnn    number of views,              e.g. --views=100\n";
  str += "\nadditional_options:\n";
  str += "  --show         shows the views in a window,  e.g. --show\n";
  str += "  --scale=fff    sets the image scaling,       e.g. --scale=10.5\n";
  str += "  --bias=fff     sets the image bias,          e.g. --bias=128\n";
  str += "  --timingFile=f stores timing infos in file f e.g.";
  str += " --timingFile=timings.txt\n";
  str += "  --filter=name  the interpolation filter,     e.g.";
  str += " --filter=d0_c3_2ef\n";
  str += "  --help         prints this help text\n";
  str += "\ninputFile:\n";
  str += "  a unimodal fourier data file (*.vuf),        e.g. engine.vuf\n";
  str += "\noutputFile:\n";
  str += "  a vuVolume spherical lightfield file,        e.g.";
  str += " engine_100_3F.vul\n";
  str += "\n\n";
  str += "Hint: In case the output is black only, use the --scale option.\n";
  
  return str;
}

bool _parseParameters(int argc, const char **argv)
{
  bool isOk = true;
  vuCommandLineTool tool(argc, argv);

  if (tool.hasParameter("--help")) {
    g_ErrorMsg += _helpString(tool);
    return false;
  }

  g_NumOfViews = tool.intForParameter   ("--views");
  g_Visualize  = tool.hasParameter      ("--show");
  if (tool.hasParameter("--scale"))
    g_Scale = tool.floatForParameter ("--scale");
  if (tool.hasParameter("--bias"))
    g_Bias = tool.floatForParameter ("--bias");
  if (tool.hasParameter("--filter"))
    g_FilterName = tool.stringForParameter ("--filter");
  if (tool.hasParameter("--timingFile"))
    g_TimingFileName = tool.stringForParameter("--timingFile");
  
  g_ErrorMsg += "Following error(s) occured:\n";

  if (g_Scale <= 0.0f) {
    g_ErrorMsg += "  - Scale must be larger than 0\n";
    isOk = false;
  }

  if (g_FilterName.isEmpty()) {
    g_ErrorMsg += "  - filterName is set to empty string";
    isOk = false;
  }

  if (!tool.hasParameter("--views")) {
    g_ErrorMsg += "  - Number of views not set (use '--views=100').\n";
    isOk = false;
  }
  else if (g_NumOfViews < 1 || g_NumOfViews > 10000) {
    g_ErrorMsg += "  - Number of views not properly set. ";
    g_ErrorMsg += "(Must be between 1 and 10000)\n";
    isOk = false;
  }

  bool isValid;
  word fileCount = tool.numberOfNonParameters(isValid);

  if (fileCount == 0) {
    g_ErrorMsg += "  - Neither an input nor an output file is specified.\n";
    isOk = false;
  }
  else if (fileCount == 1)  {
    g_ErrorMsg += "  - No output file specified.\n";
    isOk = false;
  }
  else if (fileCount == 2 && isValid)  {
    g_InputFile  = tool.getArgument(argc-2);
    g_OutputFile = tool.getArgument(argc-1);
    if (!tool.fileExists(g_InputFile)) {
      g_ErrorMsg += "  - InputFile does not exist ('";
      g_ErrorMsg += g_InputFile + "').\n";
      isOk = false;
    }
    if (g_OutputFile.isEmpty()) {
      g_ErrorMsg += "  - No output file specified.\n";
      isOk = false;
    }
  }
  else if (fileCount > 2)  {
    g_ErrorMsg += "  - More than one input and one output file specified.\n";
    isOk = false;
  }
  else {
    g_ErrorMsg += "  - The input and output file are expected to be at the ";
    g_ErrorMsg += " end of the line.\n";
    isOk = false;
  }

  g_ErrorMsg += "\nType '" + tool.toolName() + " --help' for more information!\n";

  if (isOk) g_ErrorMsg = "";

  return isOk;
}

int main(int argc, const char **argv)
{
  if (!_parseParameters(argc, argv)) {
    cerr << g_ErrorMsg << endl;
    exit(0);
  }

  init();

  if (g_Visualize) {
    glutInitDisplayMode(GLUT_DOUBLE | GLUT_RGBA | GLUT_DEPTH);
    glutInitWindowSize(g_Width, g_Height);
    glutInitWindowPosition(0, 0);

    glutCreateWindow("fourier slicer");

    glutDisplayFunc(display);
    glutKeyboardFunc(keyboard);
    glutReshapeFunc(reshape);
    glutMainLoop();
  }
  else {
    g_Slicer->lazyCalculateAndLog(g_OutputFile);
  }

  destroy();
  return 0;
}
