#include "Scanner.h"
#include <stdio.h>
#include "vuMisc/vuCommandLineTool.h"
#include <GL/glut.h>

/* 
   Usage: raycaster OPTIONS input.vud lightfield.vul

   OPTIONS: --views=100 --width=200 --height=200 [--tfunc=file.tf] [--show]
*/

Scanner *g_Scanner   = NULL;
vuString g_ErrorMsg;
vuString g_TFuncFile;
vuString g_OutputFile;
vuString g_InputFile;
int      g_Views     = -1;
int      g_Width     = -1;
int      g_Height    = -1;
bool     g_Visualize = false;

void init(void) {
  g_Scanner = new Scanner(g_InputFile.c_str(), g_Width, g_Height, g_Views);
  if (!g_TFuncFile.isEmpty()) {
    g_Scanner->loadTFuncFromFile(g_TFuncFile.c_str());
  }
}

void destroy(void) {
  delete g_Scanner;
}

//*****************************************//
//**** The glut stuff                  ****//
//*****************************************//

void display(void)
{
  g_Scanner->lazyCalculateAndDisplay(g_OutputFile);
}

void keyboard(unsigned char key, int x, int y)
{
  switch(key)
    {
    case 27:
    case 'q':
    case 'Q':
      destroy();
      exit(0);
      break;
    default:
      break;
    }
  glutPostRedisplay();
}

void reshape(int width, int height)
{

  glViewport(0, 0, width, height);

  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective(60.0f, 1.0f, 0.1f, 10.0f);
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  glPixelZoom((float)width/g_Width,(float)height/g_Height);
  glTranslated(0.0, 0.0, -2.5);
}

vuString _helpString(vuCommandLineTool &tool)
{
  vuString str;


  str += "Creating spherical RGB lightfields for vuVolume using a raycaster.\n\n";
  str += "Usage: ";
  str += tool.toolName();
  str += " mandatory_options [additional_options] inputFile outputFile\n";
  str += "\nmandatory_options:\n";
  str += "  --views=nnn    number of views,              e.g. --views=100\n";
  str += "  --width=www    width of each view in pixel,  e.g. --width=200\n";
  str += "  --height=hhh   height of each view in pixel, e.g. --height=200\n";
  str += "\nadditional_options:\n";
  str += "  --tfunc=fff    transfer function file,       e.g.";
  str += " --tfunc=engine.tf\n";  
  str += "  --show         shows the views in a window,  e.g. --show\n";
  str += "\ninputFile:\n";
  str += "  a regular 3D vuVolume data file,             e.g. engine.vud\n";
  str += "\noutputFile:\n";
  str += "  a vuVolume spherical RGB lightfield file,    e.g.";
  str += " engine_200x200_100.vul\n";
 
  return str;
}

bool _parseParameters(int argc, const char **argv)
{
  bool isOk = true;
  vuCommandLineTool tool(argc, argv);

  if (tool.hasParameter("--help")) {
    g_ErrorMsg += _helpString(tool);
    return false;
  }

  g_TFuncFile = tool.stringForParameter("--tfunc");
  g_Views     = tool.intForParameter   ("--views");
  g_Width     = tool.intForParameter   ("--width");
  g_Height    = tool.intForParameter   ("--height");
  g_Visualize = tool.hasParameter      ("--show");

  g_ErrorMsg += "Following error(s) occured:\n";
  if (tool.hasParameter("--tfunc")) {
    if (!tool.fileExists(g_TFuncFile)) {
      g_ErrorMsg += "  - Could not load transfer function from '";
      g_ErrorMsg += g_TFuncFile + "'.\n";
      isOk = false;
    }
  }

  if (!tool.hasParameter("--views")) {
    g_ErrorMsg += "  - Number of views not set (use '--views=100').\n";
    isOk = false;
  }
  else if (g_Views < 1 || g_Views > 10000) {
    g_ErrorMsg += "  - Number of views not properly set. ";
    g_ErrorMsg += "(Must be between 1 and 10000)\n";
    isOk = false;
  }

  if (!tool.hasParameter("--width")) {
    g_ErrorMsg += "  - Width not set (use '--width=250').\n";
    isOk = false;
  }
  else if (g_Width < 1 || g_Width > 2048) {
    g_ErrorMsg += "  - Width not properly set. ";
    g_ErrorMsg += "(Must be between 1 and 2048)\n";
    isOk = false;
  }

  if (!tool.hasParameter("--height")) {
    g_ErrorMsg += "  - Height not set (use '--height=250').\n";
    isOk = false;
  }
  else if (g_Height < 1 || g_Height > 2048) {
    g_ErrorMsg += "  - Height not properly set. ";
    g_ErrorMsg += "(Must be between 1 and 2048)\n";
    isOk = false;
  }

  bool isValid;
  word fileCount = tool.numberOfNonParameters(isValid);

  if (fileCount == 0) {
    g_ErrorMsg += "  - Neither an input nor an output file is specified.\n";
    isOk = false;
  }
  else if (fileCount == 1)  {
    g_ErrorMsg += "  - No output file specified.\n";
    isOk = false;
  }
  else if (fileCount == 2 && isValid)  {
    g_InputFile  = tool.getArgument(argc-2);
    g_OutputFile = tool.getArgument(argc-1);
    if (!tool.fileExists(g_InputFile)) {
      g_ErrorMsg += "  - InputFile does not exist ('";
      g_ErrorMsg += g_InputFile + "').\n";
      isOk = false;
    }
    if (g_OutputFile.isEmpty()) {
      g_ErrorMsg += "  - No output file specified.\n";
      isOk = false;
    }
  }
  else if (fileCount > 2)  {
    g_ErrorMsg += "  - More than one input and one output file specified.\n";
    isOk = false;
  }
  else {
    g_ErrorMsg += "  - The input and output file are expected to be at the ";
    g_ErrorMsg += " end of the line.\n";
    isOk = false;
  }

  g_ErrorMsg += "\nType '" + tool.toolName() + " --help' for more information!\n";

  if (isOk) g_ErrorMsg = "";

  return isOk;
}

int main(int argc, const char **argv)
{
  if (!_parseParameters(argc, argv)) {
    cerr << g_ErrorMsg << endl;
    exit(0);
  }

  init();

  if (g_Visualize) {
    glutInitDisplayMode(GLUT_DOUBLE | GLUT_RGBA | GLUT_DEPTH);
    glutInitWindowSize(g_Width, g_Height);
    glutInitWindowPosition(0, 0);

    glutCreateWindow("raycaster");

    glutDisplayFunc(display);
    glutKeyboardFunc(keyboard);
    glutReshapeFunc(reshape);
    glutMainLoop();
  }
  else {
    g_Scanner->lazyCalculateAndLog(g_OutputFile);
  }

  destroy();
  return 0;
}
