#include "Scanner.h"
#include <iostream>
#include <stdio.h>
#include <math.h>
#include "General/vuLinAlg/vuVector.h"
#include "General/vuCamera/vuParallelCamera.h"
#include "vuLightfield/Converter/vuSphericLightfieldFile.h"
#include <GL/gl.h>
#include <GL/glut.h>

//***********************************************************************
//***                     PUBLIC methods                              ***
//***********************************************************************

Scanner::Scanner(const char *fileName, dword width, dword height, dword views)
{
  m_Width  = width;
  m_Height = height;
  m_Views  = views;

  cerr << "calculate view directions..." << endl;
  // initialize sphere
  m_Sphere  = new vuUDSphere(views);
  m_Sphere->lazyCalculate();

  cerr << "initialize raycaster..." << endl;
  // initialize raycaster
  m_Raycast = new vu1112113();
  m_Raycast->setFileName(fileName);
  _initTransferFunction();
  m_Raycast->doPreviewDraw(false);
  m_Raycast->doSpecular(false);
 
  m_Raycast->read();
  m_Raycast->initOpenGL();

  // camera will be deleted by m_Raycast...
  vuParallelCamera *camera = new vuParallelCamera;
  m_Raycast->setCamera(camera);

  camera->setXRange(int(m_Raycast->getDim1Size()*1.0));
  camera->setYRange(int(m_Raycast->getDim2Size()*1.0));

  m_Raycast->setImageSize(width, height);

  m_Raycast->setSamplingDistance(0.5);
  m_Raycast->setTFuncMethod(TFUNC_PIAW);
  m_isComputed = false;
}

Scanner::~Scanner()
{
  if (m_Raycast    != NULL) delete m_Raycast;
  if (m_Sphere     != NULL) delete m_Sphere;
}

void Scanner::loadTFuncFromFile(const char *fileName)
{
  cerr << "........ load transfer function" << endl;
  try {
    if (m_TransFunct.loadTF(fileName)) {
      m_TransFunct.generateFunction();
      m_Raycast->setTransferFunc(m_TransFunct);
    }
    else
      cerr << "Warning: could not load TFunc form '" << fileName <<"'"<< endl;
  }
  catch (const char *msg) {
    cerr << msg << endl;
  }
}

void Scanner::lazyCalculateAndDisplay(const char *fileName)
{
  _lazyCalculate(fileName, true);
}

void Scanner::lazyCalculateAndLog(const char *fileName)
{
  _lazyCalculate(fileName, false);
}


//***********************************************************************
//***                     PRIVATE methods                             ***
//***********************************************************************

//Initialize the transfer function for the data.
void Scanner::_initTransferFunction()
{
  m_TransFunct.addOpacity(1,0.01);
  m_TransFunct.addOpacity(11,0.01);
  m_TransFunct.addOpacity(12,1);
  m_TransFunct.addOpacity(32,1);
  m_TransFunct.addColour(1,vuColourRGBa(0.f));
  m_TransFunct.addColour(11,vuColourRGBa(1.f));
  m_TransFunct.addColour(32,vuColourRGBa(0.f,0.f,0.6f));
  m_TransFunct.setOpacitySmoothing(0);
  m_TransFunct.setColourSmoothing(0);

  m_TransFunct.generateFunction();
  m_Raycast->setTransferFunc(m_TransFunct);
}

/*
  Raycast Camera Defaults: lookAt=[0,0,-1] up=[0,1,0] right=[1,0,0]
*/

//Render next view
void Scanner::_renderView(dword idx, vuSphericView3B *view, bool verbose)
{
  Point3d           point  = m_Sphere->getPointAt(idx);
  vuParallelCamera *camera = NULL;
  vuVector          lookAt(point.x, point.y, point.z);
  vuVector          up(0,1,0);
  vuVector          right;

  //setting the camera
  camera = (vuParallelCamera *)m_Raycast->getCameraPtr();
  lookAt.makeUnit();
  lookAt *= -1;

  vuVector pos = m_Raycast->getCenter() - (lookAt * 1000);
  _calcViewVectors(lookAt, up, right);

  camera->setPosition(pos);
  camera->setRightVector(right);
  camera->setUpVector(up);
  camera->setLookAtVector(lookAt);

  try{
    camera->init();
  }
  catch(const char *msg) {
    cerr << "catched exception" << endl;
    cerr << msg << endl;
  }

  // render the view
  if (verbose) {
    cerr << "render view: ";
    cerr.width(_numberOfDigits(m_Views - 1));
    cerr << idx << " ";
  }

  m_Raycast->doRefresh();
  m_Raycast->render();

  // copy view to lightfield
  byte            *dest = view->getMap()->getBuffer();
  const byte      *src  = m_Raycast->getImage()->get_rgb();

  for (dword i = 3 * m_Width * m_Height; i>0; i--) *(dest++) = *(src++);

  view->setLookFrom(vuVector(point.x, point.y, point.z));
  view->setUp(up);
  cerr << "Done." << endl;
}

word Scanner::_numberOfDigits(word number)
{
  if (number > 9999) return 5;
  if (number >  999) return 4;
  if (number >   99) return 3;
  if (number >    9) return 2;

  return 1;
}

void Scanner::_lazyCalculate(const char *fileName, bool display)
{
  if (!m_isComputed) {
    vuSphericLightfieldFile3B *outFile = NULL;
    vuSphericView3B           *view = new vuSphericView3B(m_Width, m_Height);

    outFile = new vuSphericLightfieldFile3B(m_Width,m_Height,m_Views,fileName);

    outFile->open();
    outFile->writeHeader();

    if (display) {
      glClearColor(0.0f, 0.0f, 0.0f, 0.0f);
      glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
      glutSwapBuffers();
    }

    for (dword i=0; i<m_Views; i++) {
      if (display) {
	glClearColor(0.0f, 0.0f, 0.0f, 0.0f);
	glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
      }

      _renderView(i, view);
      outFile->writeView(view);

      if (display) {
	view->getMap()->glRender();
	glutSwapBuffers();
      }
    }
    outFile->close();
    CHECKNDELETE(view);
    CHECKNDELETE(outFile);
    m_isComputed = true;
  }
}

void Scanner::_calcViewVectors(vuVector& lookAt, vuVector& up, vuVector& right)
{
  lookAt.makeUnit();

  if (fabs(lookAt.dot(up)) > 0.999) up = vuVector(0,0,1).makeUnit();

  up = (up - (lookAt.dot(up) * lookAt)).makeUnit();
  right  = up.cross(lookAt).makeUnit() * -1;
}
