#include "FourierSlicer.h"
#include "General/vuLinAlg/vuVector.h"
#include "vuLightfield/Converter/vuSphericLightfieldFile.h"
#include "General/vuFourier/vuFourierFilter/vuTorstensFourierFilter.h"
#include "General/vuMisc/vuTimer.h"
#include <GL/gl.h>
#include <GL/glut.h>
#include <iostream>
#include <stdio.h>
#include <math.h>

//***********************************************************************
//***                     PUBLIC methods                              ***
//***********************************************************************

template <int S>
FourierSlicer<S>::FourierSlicer(vuString fileName, dword views,
				float scale, float bias,
				vuString filterName,
				vuString timingFileName)
{
  m_NumOfViews   = views;
  m_TimingStream = NULL;
  
  if (!timingFileName.isEmpty())
    m_TimingStream = new ofstream(timingFileName, ios::out);
  
  cerr << "calculate view directions..." << endl;
  // initialize sphere
  m_Sphere  = new vuUDSphere(views);
  m_Sphere->lazyCalculate();

  cerr << "initialize fourier volume..." << endl;
  // initialize raycaster
  m_Fourier = new vu1712_1<S>;
  m_Fourier->setScale(scale);
  m_Fourier->setBias(bias);
  vuFourierFilter *filter = vuTorstensFourierFilter::getFilter(filterName);
  if (filter == NULL) {
    cerr << "WARNING: Could not load filter with name: '";
    cerr << filterName << "' use 'd0_c3_2ef' instead." << endl;
    filter = vuTorstensFourierFilter::getFilter("d0_c3_2ef");
    filterName = "d0_c3_2ef";
  }
  m_Fourier->setFilter(filter);

  m_Fourier->setFileName(fileName);
  if (m_Fourier->read()) {
    m_Width  = m_Fourier->getImageWidth();
    m_Height = m_Fourier->getImageHeight();

    m_ExternalWidth  = (dword)((float)m_Width  / M_SQRT2);
    m_ExternalHeight = (dword)((float)m_Height / M_SQRT2);

    if (((m_Width  - m_ExternalWidth)  % 2) == 1) m_ExternalWidth  -= 1;
    if (((m_Height - m_ExternalHeight) % 2) == 1) m_ExternalHeight -= 1;

    cerr << endl;
    cerr << "********************************************************" << endl;
    cerr << "views     = " << views << endl;
    cerr << "scale     = " << scale << endl;
    cerr << "bias      = " << bias  << endl;
    cerr << "dimension = " << m_ExternalWidth << "x" << m_ExternalHeight<<endl;
    cerr << "channels  = " << S << endl;
    cerr << "filter    = " << filterName << endl;
    cerr << "********************************************************" << endl;
    cerr << endl;

    m_isComputed = false;
  }
  else {
    m_isComputed = true;
    cerr << "ERROR: could not load '" << fileName << "'" << endl;
  }

  if (m_TimingStream != NULL) {
    *m_TimingStream<<"          file: "<<fileName.getLastPathComponent()<<endl;
    *m_TimingStream<< "        filter: " << filterName << endl;
    *m_TimingStream<< "         views: " << views      << endl;
    *m_TimingStream<< "    dimensions: "<<m_Fourier->getXSize()<<"^3"<< endl;
    *m_TimingStream<< "      channels: " << S << endl;
    *m_TimingStream<< "implementation: template interleaved" << endl;
    *m_TimingStream<< endl;

    *m_TimingStream << "                  lookAt                      ";
    *m_TimingStream << "timing in ms" << endl;

    *m_TimingStream << "        x           y           z        ";
    *m_TimingStream << "slicing          fft" << endl;
  }
}

template <int S>
FourierSlicer<S>::~FourierSlicer()
{
  CHECKNDELETE(m_Fourier);
  CHECKNDELETE(m_Sphere);
  CHECKNDELETE(m_TimingStream);
}

template <int S>
void FourierSlicer<S>::lazyCalculateAndDisplay(const char *fileName)
{
  _lazyCalculate(fileName, true);
}

template <int S>
void FourierSlicer<S>::lazyCalculateAndLog(const char *fileName)
{
  _lazyCalculate(fileName, false);
}

template <int S>
dword FourierSlicer<S>::getImageWidth()
{
  return m_ExternalWidth;
}

template <int S>
dword FourierSlicer<S>::getImageHeight()
{
  return m_ExternalHeight;
}



//***********************************************************************
//***                     PRIVATE methods                             ***
//***********************************************************************

//Initialize the transfer function for the data.
template <int S>
void FourierSlicer<S>::_initTransferFunction()
{
  // do nothing
}

//Render next view
template <int S>
void FourierSlicer<S>::_renderView(dword idx, vuSphericView<S,byte> *view,
				   bool verbose)
{
  Point3d  point  = m_Sphere->getPointAt(idx);
  vuVector lookAt(point.x, point.y, point.z);
  vuVector up(0,1,0);
  vuVector right;

  lookAt *= -1;
  m_Fourier->calcViewVectors(lookAt, up, right);

  // render the view
  if (verbose) {
    cerr << "render view: ";
    cerr.width(_numberOfDigits(m_NumOfViews - 1));
    cerr << idx << " ";
  }

  m_Fourier->setViewVectors(lookAt, up, right);
  if (m_TimingStream != NULL) {
    double time1 = vuTimer::getMilliSecondsAsDouble();
    m_Fourier->interpolateSlice();
    double time2 = vuTimer::getMilliSecondsAsDouble();
    m_Fourier->transformSlice();
    double time3 = vuTimer::getMilliSecondsAsDouble();

    m_TimingStream->setf(ios::fixed|ios::showpoint);
    m_TimingStream->width(12);
    *m_TimingStream <<  lookAt[0];
    m_TimingStream->width(12);
    *m_TimingStream <<  lookAt[1];
    m_TimingStream->width(12);
    *m_TimingStream <<  lookAt[2];
    m_TimingStream->width(14);
    *m_TimingStream << time2 - time1;
    m_TimingStream->width(14);
    *m_TimingStream << time3 - time2;
    *m_TimingStream << endl;
    m_Fourier->computeUnscaledImage(false);
  }
  else
    m_Fourier->computeUnscaledImage();
  
  // copy view to lightfield
  float *src = m_Fourier->getUnscaledImage()->getBuffer();
  byte  *dest = view->getMap()->getBuffer();
  float scale = m_Fourier->getScale();
  float bias  = m_Fourier->getBias();

  dword Xborder = (m_Width  - m_ExternalWidth)  / 2;
  dword Yborder = (m_Height - m_ExternalHeight) / 2;

  src += Yborder * m_Width * S;
  for (dword j=0; j<m_ExternalHeight; j++) {
    src += Xborder * S;
    for (dword i=0; i<m_ExternalWidth; i++) {
      for (int k=0; k<S; k++) {
	register float tmp = *(src++) * scale + bias;
	if (tmp < 0.0f) tmp = 0.0f;
	else if (tmp > 255.0f) tmp = 255.0f;
	*(dest++) = (byte)tmp;
      }
    }
    src +=Xborder * S;
  }

  view->setLookFrom(vuVector(point.x, point.y, point.z));
  view->setUp(up);
  cerr << "Done." << endl;
}

template <int S>
word FourierSlicer<S>::_numberOfDigits(word number)
{
  if (number > 9999) return 5;
  if (number >  999) return 4;
  if (number >   99) return 3;
  if (number >    9) return 2;

  return 1;
}

template <int S>
void FourierSlicer<S>::_lazyCalculate(const char *fileName, bool display)
{
  if (!m_isComputed) {
    vuSphericLightfieldFile<S,byte> *outFile = NULL;
    vuSphericView<S,byte>           *view    = NULL;

    view = new vuSphericView<S,byte>(m_ExternalWidth, m_ExternalHeight);

    outFile = new vuSphericLightfieldFile<S,byte>(m_ExternalWidth,
						  m_ExternalHeight,
						  m_NumOfViews,fileName);

    outFile->open();
    outFile->writeHeader();

    if (display) {
      glClearColor(0.0f, 0.0f, 0.0f, 0.0f);
      glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
      glutSwapBuffers();
    }

    for (dword i=0; i<m_NumOfViews; i++) {
      if (display) {
	glClearColor(0.0f, 0.0f, 0.0f, 0.0f);
	glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
      }

      _renderView(i, view);
      outFile->writeView(view);

      if (display) {
	view->getMap()->glRender();
	glutSwapBuffers();
      }
    }
    outFile->close();
    CHECKNDELETE(view);
    CHECKNDELETE(outFile);
    m_isComputed = true;
  }
}
