#include <stdio.h>
#include "Volume/Regular/Unimodal/3d/1B/Intensity/SimpleFVR/vuSimpleFVR.h"
#include "vuMisc/vuCommandLineTool.h"
#include "vuTFunc/vuTFDesign.h"

/* 
   Usage: vud2vuf OPTIONS input.vud output.vuf

   OPTIONS:
*/


vuString    g_ErrorMsg;
vuString    g_OutputFile;
vuString    g_InputFile;
vuTFDesign *g_TFunc   = NULL;
float       g_MultPad = M_SQRT2;
float       g_AddPad  = 0.0;
float       g_Scale   = 1.0f;

void doIt(void) {
  cout << "multPad = " << g_MultPad << endl;
  cout << "addPad  = " << g_AddPad  << endl;
  cout << "scale   = " << g_Scale   << endl;
  vu1112119::convertVUD2VUF(g_InputFile, g_OutputFile,
			    g_MultPad, g_AddPad, g_Scale, g_TFunc);
}

bool loadTFuncFromFile(vuString fileName)
{
  if (fileName.isEmpty()) return true;

  cerr << "........ load transfer function" << endl;
  try {
    g_TFunc = new vuTFDesign();
    if (g_TFunc->loadTF(fileName)) {
      g_TFunc->generateFunction();
    }
    else
      return false;
  }
  catch (const char *msg) {
    cerr << msg << endl;
    CHECKNDELETE(g_TFunc);
    return false;
  }
  return true;
}

vuString _helpString(vuCommandLineTool &tool)
{
  vuString str;


  str += "Creates frequency data volumes of type '*.vuf'.\n\n";
  str += "Usage: ";
  str += tool.toolName();
  str += " [additional_options] inputFile outputFile\n";
  str += "\nadditional_options:\n";
  str += "  --tfunc=fff    transfer function file,       e.g.";
  str += " --tfunc=engine.tf\n";  
  str += "  --multPad=fff  padding factor (sqrt(2)),     e.g. --multPad=1.0\n";
  str += "  --addPad=fff   padding delta  (0.0),         e.g. --addPad=128\n";
  str += "  --scale=fff    image scaling  (1.0),         e.g. --scale=25.5\n";
  str += "  --help         prints this help text\n";
  str += "\ninputFile:\n";
  str += "  a regular 3D vuVolume data file,             e.g. engine.vud\n";
  str += "\noutputFile:\n";
  str += "  a fourier volume rendering file,             e.g. engine_3F.vuf\n";
  
  return str;
}

bool _parseParameters(int argc, const char **argv)
{
  bool isOk = true;
  vuCommandLineTool tool(argc, argv);

  if (tool.hasParameter("--help")) {
    g_ErrorMsg += _helpString(tool);
    return false;
  }

  if (tool.hasParameter("--multPad"))
      g_MultPad = tool.floatForParameter("--multPad");
  if (tool.hasParameter("--addPad"))
      g_AddPad = tool.floatForParameter ("--addPad");
  if (tool.hasParameter("--scale"))
      g_Scale = tool.floatForParameter  ("--scale");

  g_ErrorMsg += "Following error(s) occured:\n";

  if (tool.hasParameter("--tfunc")) {
    vuString fileName = tool.stringForParameter("--tfunc");
    if (!loadTFuncFromFile(fileName)) {
      g_ErrorMsg += "  - Could not load transfer function from '";
      g_ErrorMsg += fileName + "'.\n";
      isOk = false;
    }
  }

  if (g_MultPad < 1.0f) {
    g_ErrorMsg += "  - MultPad must be larger than or equal to 1.0\n";
    isOk = false;
  }

  if (g_AddPad < 0.0f) {
    g_ErrorMsg += "  - AddPad must be larger than or equal to 0.0\n";
    isOk = false;
  }

  bool isValid;
  word fileCount = tool.numberOfNonParameters(isValid);

  if (fileCount == 0) {
    g_ErrorMsg += "  - Neither an input nor an output file is specified.\n";
    isOk = false;
  }
  else if (fileCount == 1)  {
    g_ErrorMsg += "  - No output file specified.\n";
    isOk = false;
  }
  else if (fileCount == 2 && isValid)  {
    g_InputFile  = tool.getArgument(argc-2);
    g_OutputFile = tool.getArgument(argc-1);
    if (!tool.fileExists(g_InputFile)) {
      g_ErrorMsg += "  - InputFile does not exist ('";
      g_ErrorMsg += g_InputFile + "').\n";
      isOk = false;
    }
    if (g_OutputFile.isEmpty()) {
      g_ErrorMsg += "  - No output file specified.\n";
      isOk = false;
    }
  }
  else if (fileCount > 2)  {
    g_ErrorMsg += "  - More than one input and one output file specified.\n";
    isOk = false;
  }
  else {
    g_ErrorMsg += "  - The input and output file are expected to be at the ";
    g_ErrorMsg += " end of the line.\n";
    isOk = false;
  }

  g_ErrorMsg += "\nType '" + tool.toolName() + " --help' for more information!\n";

  if (isOk) g_ErrorMsg = "";

  return isOk;
}

int main(int argc, const char **argv)
{
  if (!_parseParameters(argc, argv)) {
    cerr << g_ErrorMsg << endl;
    exit(0);
  }

  doIt();

  return 0;
}
