#ifndef _HWTIMER_H_
#define _HWTIMER_H_

#if defined(WIN32)
#include <windows.h>
#else
#include <sys/time.h>
#endif

/**
 * This class implements a basic hardware clock on systems
 * that have hardware which support a high performance
 * counter.
 *
 *
 * Author: Steve Kilthau
 * Date:   January 4, 2001
 */
class vuHWTimer
{
public:
    //!The default constructor.
    /*!If the system offers a hardware clock, the default constructor sets up 
       the appropriate values.
    */
    vuHWTimer();
    //!The copy constructor.
    vuHWTimer(const vuHWTimer& t);
    //!The destructor.
    ~vuHWTimer();

    //!The assignment operator.
    vuHWTimer& operator=(const vuHWTimer& rhs);

    //!Determines whether the system offers a hardware clock.
    /*!\return True if a hardware clock is supported.
    */
    bool isSupported(void) const;

    //!Gets the elapsed time as a float.
    /*!The elapsed time is the time since the last call to
       GetElapsedTimeDouble/GetElapsedTimeFloat, or since
       the return from the last call to SleepFloat/SleepDouble.
    */
    float getElapsedTimeFloat(void);
    //!Gets the elapsed time as a double.
    /*!The elapsed time is the time since the last call to
       GetElapsedTimeDouble/GetElapsedTimeFloat, or since
       the return from the last call to SleepFloat/SleepDouble.
    */
    double getElapsedTimeDouble(void);

    // Causes the program to halt until s seconds have
    // elapsed since the last call to GetElapsedTimeFloat,
    // GetElapsedTimeDouble, SleepFloat, or SleepDouble.
    // Before the function returns, the timer is updated.

    //!Halts the program for s seconds.
    /*!Causes the program to halt until s seconds have
       elapsed since the last call to GetElapsedTimeFloat,
       GetElapsedTimeDouble, SleepFloat, or SleepDouble.
       Before the function returns, the timer is updated.
    */
    void sleepFloat(float s);
    //!Halts the program for s seconds.
    /*!Causes the program to halt until s seconds have
       elapsed since the last call to GetElapsedTimeFloat,
       GetElapsedTimeDouble, SleepFloat, or SleepDouble.
       Before the function returns, the timer is updated.
    */
    void sleepDouble(double s);

private:
#if defined(WIN32)
    LARGE_INTEGER  m_LastCount;  //!< the last time the time was checker
    float          m_FloatFreq;  //!< the frequency of the clock
    double         m_DoubleFreq; //!< the frequency of the clock
#else
    struct timeval m_LastCount; //!< The last time the time was checked....
#endif
    bool           m_IsSupported; //!< true iff a hardware clock is supported
};

#endif
