#include "vuHWTimer.h"

#if defined(WIN32)

    vuHWTimer::vuHWTimer()
    {
        LARGE_INTEGER freq;

        if (QueryPerformanceFrequency(&freq))
        {
            m_IsSupported = true;
            m_FloatFreq   = (float)freq.QuadPart;
            m_DoubleFreq  = (double)freq.QuadPart;
        }
        else
        {
            m_IsSupported = false;
            m_FloatFreq  = 1.0f;
            m_DoubleFreq = 1.0;
        }
    }

    vuHWTimer::vuHWTimer(const vuHWTimer& t)
    {
        m_LastCount   = t.m_LastCount;
        m_FloatFreq   = t.m_FloatFreq;
        m_DoubleFreq  = t.m_DoubleFreq;
        m_IsSupported = t.m_IsSupported;
    }

    vuHWTimer::~vuHWTimer()
    {
    }

    vuHWTimer& vuHWTimer::operator=(const vuHWTimer& rhs)
    {
        if (this != &rhs)
        {
            m_LastCount   = rhs.m_LastCount;
            m_FloatFreq   = rhs.m_FloatFreq;
            m_DoubleFreq  = rhs.m_DoubleFreq;
            m_IsSupported = rhs.m_IsSupported;
        }
        return *this;
    }

    bool vuHWTimer::isSupported(void) const
    {
        return m_IsSupported;
    }

    float vuHWTimer::getElapsedTimeFloat(void)
    {
        LARGE_INTEGER count;
        float time;

        QueryPerformanceCounter(&count);
        time = (float)(count.QuadPart - m_LastCount.QuadPart)/m_FloatFreq;
        m_LastCount = count;

        return time;
    }

    double vuHWTimer::getElapsedTimeDouble(void)
    {
        LARGE_INTEGER count;
        double time;

        QueryPerformanceCounter(&count);
        time = (double)(count.QuadPart - m_LastCount.QuadPart)/m_DoubleFreq;
        m_LastCount = count;

        return time;
    }

    void vuHWTimer::sleepFloat(float s)
    {
        LARGE_INTEGER count;
        float time = -1.0f;

        if (s < 0.0f)
            return;

        while(time < s)
        {
            QueryPerformanceCounter(&count);
            time = (float)(count.QuadPart - m_LastCount.QuadPart)/m_FloatFreq;
        }
        m_LastCount = count;

        return;
    }

    void vuHWTimer::sleepDouble(double s)
    {
        LARGE_INTEGER count;
        double time = -1.0f;

        if (s < 0.0f)
            return;

        while(time < s)
        {
            QueryPerformanceCounter(&count);
            time = (double)(count.QuadPart - m_LastCount.QuadPart)/m_DoubleFreq;
        }
        m_LastCount = count;

        return;
    }

#else

#define TIMEVALDIFFFLOAT(t1, t2) (((((float)(t1.tv_sec-t2.tv_sec))*1000000.0f) + ((float)(t1.tv_usec-t2.tv_usec)))/1000000.0f)
#define TIMEVALDIFFDOUBLE(t1, t2)                            \
          (((((double)(t1.tv_sec-t2.tv_sec))*1000000.0) +     \
             ((double)(t1.tv_usec-t2.tv_usec)))/1000000.0)

    vuHWTimer::vuHWTimer()
    {
        struct timeval tv;

        m_IsSupported = (gettimeofday(&tv, 0) == 0);
    }

    vuHWTimer::vuHWTimer(const vuHWTimer& t)
    {
        m_LastCount.tv_sec   = t.m_LastCount.tv_sec;
        m_LastCount.tv_usec  = t.m_LastCount.tv_usec;
        m_IsSupported        = t.m_IsSupported;
    }

    vuHWTimer::~vuHWTimer()
    {
    }

    vuHWTimer& vuHWTimer::operator=(const vuHWTimer& rhs)
    {
        if (this != &rhs)
        {
            m_LastCount.tv_sec   = rhs.m_LastCount.tv_sec;
            m_LastCount.tv_usec  = rhs.m_LastCount.tv_usec;
            m_IsSupported        = rhs.m_IsSupported;
        }
        return *this;
    }

    bool vuHWTimer::isSupported(void) const
    {
        return m_IsSupported;
    }

    float vuHWTimer::getElapsedTimeFloat(void)
    {
        struct timeval count;
        float time;

        gettimeofday(&count, 0);
        time = TIMEVALDIFFFLOAT(count, m_LastCount);
        m_LastCount.tv_sec  = count.tv_sec;
        m_LastCount.tv_usec = count.tv_usec;

        return time;
    }

    double vuHWTimer::getElapsedTimeDouble(void)
    {
        struct timeval count;
        double time;

        gettimeofday(&count, 0);
        time = TIMEVALDIFFDOUBLE(count, m_LastCount);
        m_LastCount.tv_sec  = count.tv_sec;
        m_LastCount.tv_usec = count.tv_usec;

        return time;
    }

    void vuHWTimer::sleepFloat(float s)
    {
        struct timeval count;
        float time = -1.0f;

        if (s < 0.0f)
            return;

        while(time < s)
        {
            gettimeofday(&count, 0);
            time = TIMEVALDIFFFLOAT(count, m_LastCount);
        }
        m_LastCount.tv_sec  = count.tv_sec;
        m_LastCount.tv_usec = count.tv_usec;

        return;
    }

    void vuHWTimer::sleepDouble(double s)
    {
        struct timeval count;
        double time = -1.0f;

        if (s < 0.0f)
            return;

        while(time < s)
        {
            gettimeofday(&count, 0);
            time = TIMEVALDIFFDOUBLE(count, m_LastCount);
        }
        m_LastCount.tv_sec  = count.tv_sec;
        m_LastCount.tv_usec = count.tv_usec;

        return;
    }

#endif
