#ifndef _VUTFPREINTEGRATED_H_
#define _VUTFPREINTEGRATED_H_

#include "vuColour.h"
#include "vuTFIntensity.h"

//!Provides a preintegrated transfer function based on vuTFIntensity class
/*
  Details for this idea can be found in the paper "High-Quality
  Pre-Integrated Volume Rendering Using Hardware-Accelerated Pixel
  Shading" by Klaus Engel, Martin Kraus, Thomas Ertl

  Two methods are possible. The non-alpha weighted method corresponds
  to the approximated pre-integration proposed in section 3.5 of the
  paper. useAlphaWeighting( false/true )
  
  Section 3.4 of the paperdescribes a more precise version using the
  alpha values and sampling distance for correct attenuation. This
  would lead to a non-symmetric LUT, which is not representable by a
  1D summed area table.
  
  The other approach uses alpha weighting during preintegration. It's
  similar to the one in equation (10) of the paper. The only
  difference is that I use the integrated alpha rather than 1/sb-sf
  for normalization of the integral.  The difference of the two
  methods becomes visible for rapid jumps of the alpha value in the
  transfer together with a change in colour.

  It would be interesting to have an implementation of the exact
  method (paper 3.4).  This would require a 2-D LUT.

  To have comparable results with different sampling distances I
  modify the returned alpha value before using it for compositing:

  opacity = 1 - exp ( -alpha * SamplingDistance * STRETCHALPHA)

  'alpha' is restricted to [0,1]. The STRETCHALPHA coefficient helps
  to enhance this interval to reach opacities closer to 1.
*/

class vuTFPreintegrated : public vuTFIntensity
{
public:
    //!Constructor
    /*! \param ncomp number of components  RGBA should be built with ncomp = 4
      \param range range for which the look up table should be defined */
    vuTFPreintegrated();
    vuTFPreintegrated(dword ncomp, dword range);

    /** Copy constructor. */
    vuTFPreintegrated(const vuTFIntensity& inst);

    /** Destructor.  Frees the memory for the LUT */
    virtual ~vuTFPreintegrated();

    //assignment operator
    vuTFPreintegrated& operator=(const vuTFIntensity& rhs);
    

    /** do the preintegration for the entire transfer function */
    void preintegrate();

    /** Gives the integrated value between two intensities.
	The current impementation doesn't make a difference between front and back
	and will always return positive values. But e.g. for isosurfaces this can be used
	to determine inside and outside.
     \param front intensity at the front end of the line segment
     \param back intensity at the back end of the line segment
     \param d distance between the samples
     \param col a vuColour<N> with N according to the number of components in the
                transfer function for returning the integrated results.
     \return false if number of components in col doesn't match m_NComponents
    */
    bool integrate(float back, float front, float d, vuColourN & col);

    // do alpha weighted pre-integration
    void useAlphaWeighting(bool useit) { m_AlphaWeighted = useit; };
    

 protected:
    /** allocate memory.
	also calls vuTFIntensity::init()
    */
    bool init(dword ncomp, dword range);
    /** free memory.
	Calls also vuTFIntensity::cleanup() */
    void cleanup();

    //!The preintegrated function table for the transfer function.
    float *m_PITable;

    //! do an alpha weighted pre-integration
    bool m_AlphaWeighted;
    
};


#endif
