#include <stdio.h>
#include <iostream.h>

#include "vuColour31a.h"
#include "vuColour7a.h"
#include "vuColour9a.h"
#include "vuColourRGBa.h"
#include "vuColourXYZa.h"
#include "vuTFIntensity.h"

//----------------------------------------------------------------------------
// default constructor
//----------------------------------------------------------------------------
vuTFIntensity::vuTFIntensity()
{ 
  m_Table = 0;
  m_Light = 0;
  init(4,256); 
}

//----------------------------------------------------------------------------
// customizing constructor
//----------------------------------------------------------------------------
vuTFIntensity::vuTFIntensity(dword ncomp, dword range)
{ 
    m_Table = 0; 
    m_Light = 0;
    init(ncomp,range);
}

//----------------------------------------------------------------------------
//------------------------- The Copy Constructor -----------------------------
//----------------------------------------------------------------------------

vuTFIntensity::vuTFIntensity(const vuTFIntensity& inst)
{
    m_NComp = inst.m_NComp;
    m_Range = inst.m_Range;
    m_TableLength = inst.m_TableLength;
    m_Table = new float[m_TableLength];
    if(!m_Table) throw "Out of memory while creating transfer function table (class vuTFIntensity).\n";
    for (dword i=0;i<m_TableLength;++i)
        m_Table[i] = inst.m_Table[i];
    for (dword i=0;i<m_NComp;++i)
        m_Light[i] = inst.m_Light[i];
}

vuTFIntensity::~vuTFIntensity()
{
    cleanup();
}

//----------------------------------------------------------------------------
//------------------------- The assignment operator --------------------------
//----------------------------------------------------------------------------

vuTFIntensity& vuTFIntensity::operator=(const vuTFIntensity& rhs)
{
    if (this != &rhs)
    {
      if(m_Range == rhs.m_Range) {
	if(m_NComp == rhs.m_NComp) {		// same number of components -> just copy
	  for (dword i=0;i<m_TableLength;++i)
	    m_Table[i] = rhs.m_Table[i];
	  for(dword i=0;i<m_NComp;i++)
	    m_Light[i] = rhs.m_Light[i];
	} else if(m_NComp == 4 && rhs.m_NComp > 4) {	// do conversion from spectral to RGBa
	  vuColourRGBa col;
	  for (dword i=0, index=0;i<m_Range;i++) {
	    rhs.getRGBa(i,col);
	    for(dword j=0;j<m_NComp;j++)
	      m_Table[index++] = col[j];
	  }
	} else if(m_NComp == 8 && rhs.m_NComp == 32) {	// do conversion from spec31 to V7
	    vuColour7a col;
	    for (dword i=0, index=0;i<m_Range;i++) {
		col.from(vuColour31a(&rhs.m_Table[i*rhs.m_NComp]));
		for(dword j=0;j<m_NComp;j++)
		    m_Table[index++] = col[j];
	    }
	    col.from(vuColour31a(rhs.m_Light));
	    for(dword i=0;i<m_NComp;i++)
		m_Light[i] = col[i];
	} else
	    cout << "Incompatible number of components in transfer function."<<endl;}
      
    }
    return *this;
}

//----------------------------------------------------------------------------
//------------------------- private initialization ---------------------------
//----------------------------------------------------------------------------

bool vuTFIntensity::init(dword ncomp, dword range)
{
  if(ncomp == 0 || range == 0) {
    m_Table = 0;
    m_Light = 0;
    m_NComp = 0;
    m_Range = 0;
    m_TableLength = 0;
    return true;
  }
  m_NComp = ncomp;
  m_Range = range;
  m_TableLength = m_NComp*m_Range;

  m_Table = new float[m_TableLength];
  if(!m_Table) return false;


  m_Light = new float[m_NComp];
  if(!m_Light) return false;

  //printf("table created %i %i %i\n",(int)m_NComp, (int)m_Range, (int)m_TableLength); fflush(NULL);
  for (dword i=0;i<m_TableLength;i++)
    m_Table[i] = 0.0f;
  for (dword i=0;i<m_NComp;i++)
    m_Light[i] = 1.0f;
  return true;
}

//------------------------------------------------------------------------------
void vuTFIntensity::cleanup()
{
  if(m_Table) delete [] m_Table;
  m_Table = 0;
  if(m_Light) delete [] m_Light;
  m_Light = 0;
}

//------------------------------------------------------------------------------
bool vuTFIntensity::resize(dword ncomp, dword range)
{
    cleanup();
    return init(ncomp, range);
}

//----------------------------------------------------------------------------
float vuTFIntensity::getOpacityAtPos(dword i)
{
  return m_Table[i*m_NComp+(m_NComp-1)];
}

//----------------------------------------------------------------------------
void vuTFIntensity::getRGBa(dword i, vuColourRGBa& rgba) const
{
  if (m_NComp == 4) rgba = &m_Table[i*m_NComp];
  else if(m_NComp == 32) {
    // I normalize the spectrum after the light has been modulated by reflectance.
    // This is not right, but the given reflectances are too dark and this is a hack
    // to brighten them... ,-|
    vuColour31a c31a(&m_Table[i*m_NComp]);
    //float normal = vuColourXYZa(c31a)[1];	// use Y (luminance) as normalization divisor
    c31a *= vuColour31a(m_Light);
    rgba.from(c31a);
    //if(normal > 0) rgba /= normal;
    rgba.clampTo01();
  } else if(m_NComp == 8) {
    // I normalize the spectrum after the light has been modulated by reflectance.
    // This is not right, but the given reflectances are too dark and this is a hack
    // to brighten them... ,-|
    vuColour7a c7a(&m_Table[i*m_NComp]);
    //float normal = vuColourXYZa(c31a)[1];	// use Y (luminance) as normalization divisor
    c7a *= vuColour7a(m_Light);
    rgba.from(c7a);
    //if(normal > 0) rgba /= normal;
    rgba.clampTo01();
  } else if(m_NComp == 10) {
    vuColour9a c9a(&m_Table[i*m_NComp]);
    //float normal = vuColourXYZa(c31a)[1];	// use Y (luminance) as normalization divisor
    c9a *= vuColour9a(m_Light);
    c9a.to(rgba);
    //if(normal > 0) rgba /= normal;
    rgba.clampTo01();
  } /* include conversion for other colour models here */
}

void vuTFIntensity::normalizeAlphaToOne() 
{
    dword i,c;
    float *fp = m_Table;
    for(i=0;i<m_Range;i++) {
	float alpha = fp[m_NComp-1];
	for(c=0;c<m_NComp-1;c++,fp++)
	    *fp *= alpha;
	*(fp++) = 1.0f;	// set alpha to one
    }
}

void vuTFIntensity::fromMap(const vuMap& map) 
{
    dword i,c;
    float *fp = m_Table;
    for(i=0;i<m_Range;i++) {
	for(c=0;c<m_NComp-1;c++,fp++)
	    *fp = ((float)map[i])/255.0f;
	*(fp++) = 1.0f;	// set alpha to one
    }
}
