#ifndef _VUTFDESIGN_H_
#define _VUTFDESIGN_H_

#include "vuColour.h"
#include "vuColourRGBa.h"
#include "vuTFIntensity.h"
#include "vuDVector.h"
#include "vuSimpleTypes.h"
#include "vuParser.h"

//! A transfer function based on the one described by Design Galleries.

/*! This class is a powerful transfer function that lets you define
    transfer functions of any complexity.  The transfer function is
    built from two types of control nodes.

    First, there are opacity control nodes which control the opacity
    transfer function. These define an opacity polyline function
    for assigning opacity to byte density values.  Specifically, they
    define vertices which are then joined using straight lines.  
    Before being applied, the opacity polyline is smoothed to avoid 
    any sharp changes.

    Second, there are colour control nodes which define colour regions
    for the transfer function.  These combine the rgb values into one
    type of control node, to be applied over a density region.  For
    simplicity, the colour control nodes are defined at a given density
    value, and define the colour for all density values before, up to
    the next colour control node.  The colour regions are smoothed as 
    well so as to avoid sharp colour changes.

    Together, the two control nodes define an opacity transfer function
    with colours applied to the different regions.  There can be an
    unlimited number of control nodes, limited only by the number of
    density values.
*/

class vuTFDesign : public vuTFIntensity, public vuParser
{
public:
    //-------------------------------------------------------------------------------
    //!The structure representing opacity control nodes.
    struct OpacityNode
    {
        OpacityNode(byte i=0, float o=0) : intensity(i), opacity(o) {};

        dword intensity;
        float opacity;
    };

    //-------------------------------------------------------------------------------
    //!The structure for representing colour nodes.
    struct ColourNode 
    {
        ColourNode(byte intensity = 0, dword ncomp = 32, const float val = 0);
        ColourNode(byte intensity, dword ncomp, const vuColourN& _col);
        ColourNode(byte intensity, dword ncomp, const float* _col);

        dword intensity;
        float col[32];
    };

    //-------------------------------------------------------------------------------
    //functions for vuTFDesign
public:
    //!Constructor creating an empty transfer function.
    /*!The empty transfer function is seethrough and black
       throughout.  To get different behaviour, opacity and
       colour control nodes have to be added.
    */
    vuTFDesign(dword ncomp = 4, dword range = 256);
    virtual ~vuTFDesign() {};
    
    //!Adds an opacity control node to the transfer function.
    /*!\return The index of the new opacity control node.
    */
    dword addOpacity(byte intensity, float alpha);
    //!Removes a previously added opacity control node.
    void removeOpacity(dword index);
    //!Returns the number of opacity control nodes defined for the function.
    dword getNumOpacities() const;
    /*!Returns the opacity control node with the given index. */
    const OpacityNode& getOpacity(dword index) const;
    //!Returns the relative smoothing of the opacity function.
    float getOpacitySmoothing() const;
    //!Sets the relative smoothing of the opacity function.
    /*!\param rate smoothing rate, set between 0 and 1, with 1 being max smoothing.
    */
    void setOpacitySmoothing(float rate);

    //!Adds a colour control node to the transfer function.
    /*!\returns The index of the new colour control node.
    */
    dword addColour(dword intensity, const vuColourN& _col);
    //!Adds a colour control node to the transfer function.
    /*!\returns The index of the new colour control node.
    */
    dword addColour(dword intensity, const float* _col);
    //!Removes a previously added colour control node.
    void removeColour(dword index);
    //!Returns the number of colour control nodes defined for the function.
    dword getNumColours() const;
    //!Returns the colour control node at the given index.
    const ColourNode& getColour(dword index) const;
    //!Returns index of a colour node
    dword getColourNodeIndex(const ColourNode& cnode) const;
    
    /** Remove all colour and opacity nodes and insert default nodes at 0 and m_Range-1. */
    virtual void clearAllNodes();
    //!Returns the relative smoothing between the colour nodes.
    float getColourSmoothing() const;
    //!Sets the relative smoothing between the colour control nodes.
    /*!\param A smoothing rate, set between 0 and 1, with 1 being max smoothing.
    */
    void setColourSmoothing(float rate);

    //!Generates the transfer function using the defined control nodes.
    /*!The added opacity and colour control nodes don't take effect until
       this method is called.  The function is generated as described in
       the comments on the class, with smoothing automatically applied.
    */
    virtual void generateFunction();

    /** Interpolates the opacity nodes as alpha values into the transfer function. */
    virtual void generateOpacities();

    /** Load a specifiaction for a transfer function from a text file.
	Calls virtual parseTFunc()
     \todo Detailed descripion of file format to come...*/
    bool loadTF(const char* fname);
    /** Save the specified transfer function.
	Calls virtual writeTFunc()
     */
    bool saveTF(const char* fname) const;

 protected:
    //!Smooths the passed in array, of num elements, by the given percentage.
    void smooth(float *array,float percent) const;

    /** Parses the file.
	Calls parseTFunc(). */
    void Parse(const char* filename) throw (const char *);
    /** Parse the file.
        Customize this function to add own features to the loader. Make use of
	readOpacity(), readColour() and maybe see example parser in source file.
	This class is derived from vuParser and provides some helper functions for file
	parsing.
    */
    virtual void parseTFunc();
    /** Save the specified transfer function to output stream.
        Override to save extended properties of derived transfer function.
	On error throw const char *
    */
    virtual void writeTFunc(ofstream &ofp) const;
    
    /** write number of components in this tfunc to file */
    bool writeNComp(ofstream &ofp) const;
    
    /**Reads a reset line that can setup a different number of
       components.  If successfull the transfer function will be reset
       to the new number of components */
    bool readNComp();

    /** Read a line defining an opacity control point.
	If there is a syntactic error a char* exception is thrown.
	On succesful reading the opacity node is added to the transfer function.
	\return true if an opacity line was found otherwise false.
    */
    bool readOpacity();
    /** Writing equicalent to readOpacity().
     Writes information about all opacity nodes to the output file stream ofp. */
    bool writeOpacities(ofstream &ofp) const;
    
    /** Read a line defining a colour control point.
	If there is a syntactic error a char* exception is thrown.
	On succesful reading the colour node is added to the transfer function.
	\return true if an colour line was found otherwise false.
    */
    bool readColour();
    /** Writing equivalent to readColour().
     Writes information of all colour nodes to the output file stream ofp. */
    bool writeColours(ofstream &ofp) const;
    
//------------------------------------------------------------------------------
// member variables
    
    //!The opacity control nodes.
    vuDVector<OpacityNode> m_Opacities;
    //!The relative smoothing of the opacity function.
    float m_OpacitySmoothing;

    //!The colour control nodes.
    vuDVector<ColourNode> m_Colours;
    //!The relative smoothing of the colour function.
    float m_ColourSmoothing;

};

#endif
 
