#ifndef _SPALETTE_H_
#define _SPALETTE_H_

#include <iostream.h>

#include "vuColour31a.h"
#include "vuColour7a.h"
#include "vuColourRGBa.h"
#include "vuDVector.h"
#include "vuSimpleTypes.h"

#include "SOptimizer.h"

#define SPAL_RANGE	100000.0f	//  +/- max value for a bound

/*
static char _c31t7ascii[] = 
#include "c31t7.mat"
;
*/

/** The spectral palette organizes light sources and reflectances.
    
    This is a framework for designing spectra for lights and
    reflectances. The SOptimizer is used to find spectra fulfilling
    certain design criteria.
*/
class SPalette 
{
 protected:
//-----------------------------------------------------------------------------
/** Hold RGB value, weight and status information for a certain
  refl/light pair. */
    class SDesignColour
	{
    public:
	    //! constructor.
	    SDesignColour() : design(false), freecolour(-1) {};
	    //! customized constructor.
	    SDesignColour(const vuColourRGBa& _rgbw, bool _design = false)
		: design(_design), freecolour(-1), rgbw(_rgbw) {};
	    //! copy constructor
	    SDesignColour(SDesignColour& rhs)
		{
		    design = rhs.design;
		    rgbw = rhs.rgbw;
		    freecolour = rhs.freecolour;
		};
	    
	    SVector getCol() const {return SVector(3,rgbw.getData());};
	    
	    float getWeight() const {return rgbw[3];};
	    
	    /* Returns rgb value for this node and weight in alpha */
	    vuColourRGBa& getRGBW() { return rgbw;};
	    
	    bool		design;		//!< colour to be designed?
	    /** ID for a 'free colour', -1 = no 'free colour' */
	    int			freecolour;
	    
	    vuColourRGBa	rgbw;		//the colour
	};
//-----------------------------------------------------------------------------
/** holds ID and bounds for a 'free' colour */
    class SFreeColour 
	{
    public:
	    //! constructor, initializing ub(1), lb(0), ID(0)
	    SFreeColour(dword _id=0,
			const vuColourRGBa& _lb= vuColourRGBa(0.0f),
			const vuColourRGBa& _ub= vuColourRGBa(1.0f))
		: ub(_ub), lb(_lb), useBounds(true), ID(_id), used(0) {};
	    //! copy constructor
	    SFreeColour(SFreeColour& rhs)
		{
		    ID = rhs.ID;
		    ub = rhs.ub;
		    lb = rhs.lb;
		    used = 0;
		    useBounds = rhs.useBounds;
		};
	    //! upper and lower bound
	    vuColourRGBa ub, lb;
	    bool  useBounds;
	    dword ID;
	    dword used;
	    dword position;
	};

//-----------------------------------------------------------------------------
/** record for a spectrum for a reflectance or a light source */
    class SSpectrum
	{
    public:
	    //! constructor, initializing ub(1), lb(0), ID(0)
	    SSpectrum(vuColour31a _spec=vuColour31a(0.0f), dword _id=0)
		: ID(_id), spec(_spec), design(false), used(0),
		ub(SPAL_RANGE), lb(0.0f), useBounds(true)
		{ setName("unnamed");};
	    //!copy constructor
	    SSpectrum(SSpectrum& rhs)
		{
		    ID = rhs.ID;
		    spec = rhs.spec;
		    design = rhs.design;
		    used = 0;
		    ub = rhs.ub;
		    lb = rhs.lb;
		    useBounds = rhs.useBounds;
		    setName(rhs.name);
		};
	    //! set a name for the spectrum
	    void setName(const char* _name)
		{
		    if(name && strlen(_name)<127)
			strcpy(name,_name);
		    else name[0]=0;
		}
	    friend ostream& operator<< (ostream& os, const SSpectrum& s)
		{
		    os << s.name << endl;
		    if(s.name[0]==0) os << "unnamed" << endl;
		    else s.spec.write(os) << endl;
		    return os;
		}
	    friend istream& operator>> (istream& is, SSpectrum& s)
		{
		    is >> s.name;
			s.spec.read(is);
		    return is;
		}
	    
	    dword	ID;		//!< ID
	    vuColour31a spec;		//!< spectrum
	    bool	design;		//!< to be designed?
	    dword	used;		//!< design and used?
	    vuColour31a ub,lb;		//!< upper and lower bound
	    bool	useBounds;	//!< use bounds?
	    char	name[128];	//!< name for the spectrum
	};

//-----------------------------------------------------------------------------
// public functions

 public:
    /** constructor */
    SPalette();
    /** destructor */
    ~SPalette();

    /** loads a palette definition
	\return false=error, true=success */
    bool load(const char* filename);
    /** saves a palette definition
	\return false=error, true=success */
    bool save(const char* filename);
    /** loads a single spectrum
	\return false=error, true=success */
    bool loadSpectrum(int rid, int lid, const char* filename);
    /** saves a single spectrum
	\return false=error, true=success */
    bool saveSpectrum(int rid, int lid, const char* filename);

    /** clears all entries for lights and reflectances */
    void reset();
    
    /** Creates a spectrum from the given design criteria.  This
	function builds a matrix and solves for a (positivity
	constrained) least squares solution using SOptimizer. */
    bool createSpectrum();

// general status settings    
    /** Sets m_WSmoothness. */
    void setSmoothnessWeight(float weight);
    /** Sets m_WErrorMin */
    void setErrorMinWeight(float weight);
    /** When finding a colour for a given spectrum you can make it correct
        in the V31 or in the V7 space. This function tells which one to use.
        To minimize the difference between the results of the two colour
        spaces use setErrorMinWeight(). */
    void useV7(bool useit=true) { m_UseV7 = useit;};
    
    //! sets the initial values of the design colours to the actual colours
    void matchDesignColours();
    
// stuff for the lights
    //! returns number of lights.
    dword getNLights() const { return m_Lights.getLength();};
    /** Add a light spectrum to the palette.
    \return index in the list*/
    int addLight(const vuColour31a& light);
    /** Remove a light spectrum from the palette.
    \index Index in the list of lights.*/
    void removeLight(word index);

// stuff for reflectances
    //! returns number of lights.
    dword getNRefls() const
	{ return m_Refls.getLength();};
    /** Add a reflectance spectrum.
     \return index in the list*/
    int addReflectance(const vuColour31a& refl);
    /** Remove a reflectance spectrum from the palette.
    \index Index in the list of reflectances.*/
    void removeReflectance(word index);
    
// misc stuff on lights and reflectances 
    //! returns handle to a spectrum
    /*! Either rid or lid has to be -1, the other holds
      the index for the light/reflectance you're inquiring about. */
    vuColour31a& getSpec(int rid, int lid)
	{ return getSpecRec(rid,lid).spec;};
    //! use bounds? returns handle for bool
    /*! Either rid or lid has to be -1, the other holds
      the index for the light/reflectance you're inquiring about. */
    bool& useSpecBounds(int rid, int lid)
	{ return getSpecRec(rid,lid).useBounds;};
    //! handle to upper bound of spectrum
    /*! Either rid or lid has to be -1, the other holds
      the index for the light/reflectance you're inquiring about. */
    vuColour31a& getSpecUB(int rid, int lid)
	{return getSpecRec(rid,lid).ub;};
    //! handle to lower bound of spectrum
    /*! Either rid or lid has to be -1, the other holds
      the index for the light/reflectance you're inquiring about. */
    vuColour31a& getSpecLB(int rid, int lid)
	{return getSpecRec(rid,lid).lb;};
    //! set name for spectrum
    /*! Either rid or lid has to be -1, the other holds
      the index for the light/reflectance you're inquiring about. */
    void setSpecName(int rid, int lid, const char* name)
	{ getSpecRec(rid,lid).setName(name);};
    //! get name of spectrum
    /*! Either rid or lid has to be -1, the other holds
      the index for the light/reflectance you're inquiring about. */
    const char* getSpecName(int rid, int lid)
	{ return getSpecRec(rid,lid).name;};
    //! tell whether the spectrum should be designed (returns bool handle)
    /*! Either rid or lid has to be -1, the other holds
      the index for the light/reflectance you're inquiring about. */
    bool& getSpecDesignState(int rid, int lid)
	{ return getSpecRec(rid,lid).design;};
    
// stuff for the design colours    
    //! compute the actual colour for a certain light-reflectance incidence
    vuColourRGBa getRLColour(dword rid, dword lid, bool useV7=false);
    /** Sets the desired colour for a reflectance-light combination.
        The alpha value is interpreted as a weight for that colour. */
    void setDesignRGBW(dword rid, dword lid, const vuColourRGBa& col);
    /** returns handle to the currently set 'design colour' */
    vuColourRGBa& getDesignRGBW(dword rid, dword lid);
    /** This flag indicates whether the entry should be used as
	a design criterium */
    bool& getDesignState(dword rid, dword lid)
	{ return m_DesCol[rid][lid].design;}
    
// stuff for free colours
    /** Attaches a 'free colour' slot to a light-reflectance combination.
	The fcid is the index of the 'free colour'. If set to -1 the
	only the design colour will be used during optimization. */
    void attachFreeColour(dword rid, dword lid, int fcid);
    /** Removes an attached 'free colour' from a design node. */
    void detachFreeColour(dword rid, dword lid)
	{attachFreeColour(rid,lid,-1);};
    //! returns ID of free colour attached to a design colour
    int getFreeColourID(int rid, int lid) {
	if(rid>=0 && lid>=0) return m_DesCol[rid][lid].freecolour;
	return -1;
    };
    
    //! Returns number of slots for 'free colours' .
    dword getNFreeColour() { return m_FreeCol.getLength();};
    /** creates a 'free colour' slot, if it doen't already exist,
        lb and ub can be used to make a lower and upper bound */
    void setFreeColour(dword fcid,
		       const vuColourRGBa& lb= vuColourRGBa(0.0f),
		       const vuColourRGBa& ub= vuColourRGBa(1.0f));
    //! get handle to upper bound of free colour
    vuColourRGBa& getFreeColUB(dword fcid)
	{ return m_FreeCol[fcid].ub;}
    //! get handle to lower bound of free colour
    vuColourRGBa& getFreeColLB(dword fcid)
	{ return m_FreeCol[fcid].lb;}
    //! should the bounds be used? (handle to flag)
    bool& getFreeColBoundState(dword fcid)
	{ return m_FreeCol[fcid].useBounds;}

//----------------------------------------------------------------------------
// internal functions
    
 private:
    //! returns handle to a spectral class
    SSpectrum& getSpecRec(int rid, int lid) {
	if(rid>=0) return m_Refls[rid];
	else if(lid>=0) return m_Lights[lid];
	else return m_Refls[0];	// panic
    }
    /** marks 'free colour'-slots that are actually used and checks
	design colour mask for inconsistencies. */
    bool checkSetup();
    //! Returns weighted error minimizing matrix for the given spectrum.
    SMatrix makeErrorMat(const vuColour31a& diag);
    //! Returns weighted smoothing matrix (31x31).
    SMatrix makeSmoothingMat();
    /** Returns a V31tRGB matrix accumulated with the diagonal matrix
	of the given Spectrum. */
    SMatrix makeCXFMat(const vuColour31a& diag)
	{ return makeCXFMat(diag, m_UseV7);}
    /** Returns a V31tRGB matrix accumulated with the diagonal matrix
	of the given Spectrum. 
	The switch useV7 decides whether to do diagonal multiply in V7.  
    */
    SMatrix makeCXFMat(const vuColour31a& diag, bool useV7);
    
//-----------------------------------------------------------------------------
// member variables
    
 private:
    /** A growable array (vuDVector) of lights.  Use add/removeLight()
	for accessing this array to maintain consistence with m_DesCol. */
    vuDVector<SSpectrum>	m_Lights;
    /** A growable array (vuDVector) of reflectances.  Use
	add/removeReflectance() for accessing this array to maintain
	consistence with m_DesCol. */
    vuDVector<SSpectrum>	m_Refls;
    
    /** A 2D field of RGB values holding the desired values for each
	light*reflectance combination. The alpha values of the colours
	are used as weights for the minimization. (Indexing is
	m_DesCol[reflID][lightID]) */
    vuDVector< vuDVector<SDesignColour> > m_DesCol;
    
    /** slots for 'free' colours. */
    vuDVector< SFreeColour >	m_FreeCol;
    
    /** perform optimization using V7 colour Xform? */
    bool	m_UseV7;
    
    /** Weight for smoothness as a design criterion. 0 - inactive.
	Appends a 2nd derivative filter toeplitz(m_Smoothness*(-1 2 -1)...)
	to the optimization matrix. */
    float	m_WSmoothness;
    //! switch for whether to use smoothness or not
    bool	m_UseSmoothness;
    
    /** Weight for the error minimizing matrix. 0 - inactive.
	This matrix is a combination of (c31tRGB - (c7tRGB*c31t7)) for
	each light source. The resulting vector is an indication for the
	error in the 7model. */
    float	m_WErrorMin;
    //! switch for using the error minimizing matrix
    bool	m_UseErrorMin;
    
    /** The optimizer used for fitting spectra into the given criteria. */
    SOptimizer			m_Optimizer;

    SMatrix	m_C31t7;	//!<for conversion from V31 to V7

    dword	m_NDesCol;	//!<number of design colours used
    dword	m_NDesSpec;	//!<number of spectra used for design
    SMatrix	m_CXF31toRGB;	//!<colour conversion matrix.
    SMatrix	m_CXF7toRGB;	//!<colour conversion matrix.
    SMatrix	m_CXF31to7;	//!<colour conversion matrix.
};
 
#endif










