/** !\file SPalette.cpp implementation for SPalette.h
Steven Bergner, 02/11/02
*/
#include <ctype.h>
#include "SPalette.h"
#define USE_ERRORMAT		// use difference matrix for error minimize.
/* note on USE_ERRORMAT
   - former uses of errorminmat produced ill-conditioned matrix
   - there is another method just using 31->RGB and 31->7->RGB conversion
     independently, which looked more stable (variant B)
   - freecolour isn't implemented correctly for this workaround version
   - a rotten old matrix has been found in vuColour7a.h
   - now variant A also works and is compatible with freecolour
   that's it. so, no use to read these notes ;-)
   - maybe I make freecolours work with variant B...
 */

#define SQRT_31		5.5677643628300219221194712989185

namespace coool 
{
    using namespace coool;

SPalette::SPalette() : 
    m_UseV7(false), m_WSmoothness(0.0f), m_WErrorMin(0.0f),
    m_CXF31toRGB(3,31,vuColourRGBa::getCXF31toRGB()),
    m_CXF7toRGB(3,7,vuColourRGBa::getCXF7toRGB()),
    m_CXF31to7(7,31,vuColour7a::getCXF31to7())
{
    // add white reference spectra to each list
    SSpectrum tmp(vuColour31a(1.0f));
    m_Lights.add(tmp);
    m_Refls.add(tmp);
}

SPalette::~SPalette()
{
    //vuDVectors destruct themselves :-)
}

void SPalette::reset()
{
    m_Lights.removeRange(0,m_Lights.getLength());
    m_Refls.removeRange(0,m_Refls.getLength());
}

int SPalette::addLight(const vuColour31a& light)
{
    dword index = m_Lights.getLength();	//vuDVector always adds to the end
    SSpectrum tmp(light,index);
    m_Lights.add(tmp);
    //update m_DesCol dimensions and entries
    SDesignColour dcol;
    dword nrefl = m_Refls.getLength();
    for(dword i=0;i<nrefl;i++)
	m_DesCol[i][index] = dcol;
    return (int)index;
}

void SPalette::removeLight(word index)
{
    m_Lights.remove(index);
    dword nrefl = m_Refls.getLength();
    for(dword i=0;i<nrefl;i++)
	m_DesCol[i].remove(index);
}

int SPalette::addReflectance(const vuColour31a& refl)
{
    dword index = m_Refls.getLength();	//vuDVector always adds to the end
    SSpectrum tmp(refl,index);
    m_Refls.add(tmp);
    //update m_DesCol dimensions and entries
    SDesignColour dcol;
    int nlights = (int)m_Lights.getLength();
    for(int i=nlights-1;i>=0;i--)
    {
	m_DesCol[index][i] = dcol;
    }
    return (int)index;
}

void SPalette::removeReflectance(word index)
{
    m_Refls.remove(index);
    m_DesCol.remove(index);	// just remove a row from the 2D array
}

//! compute the actual colour for a certain light-reflectance incidence
vuColourRGBa SPalette::getRLColour(dword rid, dword lid, bool useV7)
{
    if(rid<m_Refls.getLength() && lid<m_Lights.getLength())
    {
	if(useV7)
	    return vuColourRGBa(vuColour7a(m_Lights[lid].spec)*
				vuColour7a(m_Refls[rid].spec)
				).clampTo01();
	else
	    return vuColourRGBa(m_Lights[lid].spec*m_Refls[rid].spec
				).clampTo01();
    }
    return vuColourRGBa(0.0f);
}

void SPalette::setDesignRGBW(dword rid, dword lid, const vuColourRGBa& col)
{
    if(rid<m_Refls.getLength() && lid<m_Lights.getLength())
    {
	m_DesCol[rid][lid].rgbw = col;
	m_DesCol[rid][lid].design = true;
    }
}

/** returns handle to the currently set 'design colour' */
vuColourRGBa& SPalette::getDesignRGBW(dword rid, dword lid)
{
    if(rid<m_Refls.getLength() && lid<m_Lights.getLength())
	return m_DesCol[rid][lid].rgbw;
    else throw "vuSpecPalette::getDesignRGBW: Out of bounds!";
}

void SPalette::attachFreeColour(dword rid, dword lid, int fcid)
{
    if(rid<m_Refls.getLength() && lid<m_Lights.getLength()) {
	if(fcid >= (int)m_FreeCol.getLength()) 
	    setFreeColour(fcid);
	m_DesCol[rid][lid].freecolour = fcid;
    };
}

void SPalette::setFreeColour(dword fcid, 
			     const vuColourRGBa& lb, const vuColourRGBa& ub)
{
    SFreeColour tmp(fcid,lb,ub);
    m_FreeCol[fcid] = tmp;
}

//! sets the initial values of the design colours to the actual colours
void SPalette::matchDesignColours()
{
    for(dword r=0;r<getNRefls();r++)
	for(dword l=0;l<getNLights();l++)
	{
	    vuColourRGBa rgbw(getRLColour(r,l,m_UseV7));
	    rgbw[3] = 1;
	    setDesignRGBW(r,l,rgbw);
	}
}

//! loads a palette definition
bool SPalette::load(const char* filename)
{
    ifstream is(filename);
    if(is.good()) {
	reset();
	try{
	    while(!is.eof())
	    {
		char rl='x';
		is>>ws;		// remove leading whitespace
		is>>rl;
		switch(tolower(rl)) {
		    case 'r': 
		    {
			int id = addReflectance(vuColour31a());
			is>>m_Refls[id];
		    } break;
		    case 'l':
		    {
			int id = addLight(vuColour31a());
			is>>m_Lights[id];
		    } break;
		    case 'x':
			//we're done very soon...
			break;
		    default:
			cout << "Error in palette definition" << endl;
			cout << "char: " << rl << endl;
			return false;
		}
	    }
	} catch (const char* msg) {
	    cout << "Error loading palette: " << msg <<endl;
	    return false;
	}
    } else return false;
    return true;
}

//! saves a palette definition
bool SPalette::save(const char* filename)
{
    ofstream os(filename);
    if(os.good()) {
	try{
	    for(int id=0;id<(int)m_Refls.getLength();id++)
		os << 'r' << m_Refls[id] << endl;
	    for(int id=0;id<(int)m_Lights.getLength();id++)
		os << 'l' << m_Lights[id] << endl;
	} catch (const char* msg) {
	    cout << "Error saving palette: " << msg <<endl;
	    return false;
	}
    } else return false;
    return true;
}
/** loads a single spectrum
    \return false=error, true=success */
bool SPalette::loadSpectrum(int rid, int lid, const char* filename)
{
    if(rid<(int)m_Refls.getLength() && lid<(int)m_Lights.getLength())
    {
		ifstream is(filename);
		if(is.good()) {
			SSpectrum &specrec = getSpecRec(rid, lid);
			try{
				is>>specrec;
			} catch (const char* msg) {
				cerr << "Error loading spectrum: " << msg <<endl;
			return false;
			}
		} else return false;
    } else return false;
    return true;
}

/** saves a single spectrum
    \return false=error, true=success */
bool SPalette::saveSpectrum(int rid, int lid, const char* filename)
{
    if(rid<(int)m_Refls.getLength() && lid<(int)m_Lights.getLength())
    {
	ofstream os(filename);
	if(os.good()) {
	    SSpectrum &specrec = getSpecRec(rid, lid);
	    try{
		os<<specrec;
	    } catch (const char* msg) {
		cout << "Error saving spectrum: " << msg <<endl;
		return false;
	    }
	} else return false;
    } else return false;
    return true;
}


/** Marks 'free colour'-slots that are actually used and checks
    design colour mask for inconsistencies. */
bool SPalette::checkSetup()
{
    for(dword c=0;c<m_FreeCol.getLength();c++)
	m_FreeCol[c].used=0;
    for(dword c=0;c<m_Lights.getLength();c++)
	m_Lights[c].used=0;
    for(dword c=0;c<m_Refls.getLength();c++)
	m_Refls[c].used=0;

    m_NDesCol=0;
    for(dword r=0;r<m_Refls.getLength();r++)
    {
	for(dword l=0;l<m_Lights.getLength();l++)
	{
	    if(m_DesCol[r][l].getWeight()>0 && m_DesCol[r][l].design)
	    {
		//we can't design a light and a reflectance if
		//their colour is a design criterium
		if(m_Lights[l].design && m_Refls[r].design)
		    return false;
		else if(m_Lights[l].design || m_Refls[r].design)
		{
		    m_NDesCol++;
		    if(m_DesCol[r][l].freecolour>=0)
			m_FreeCol[m_DesCol[r][l].freecolour].used++;
		    if(m_Lights[l].design)
			m_Lights[l].used++;
		    else //if(m_Refls[r].design)
			m_Refls[r].used++;
		    //cout<<r<<"r   (checksetup)  l"<<l<<endl;
		}
	    } else m_DesCol[r][l].design=false;
	}
    }
    if(m_WSmoothness == 0) m_UseSmoothness = false;
    if(m_WErrorMin == 0) m_UseErrorMin = false;
    
    return true;
}

/** Sets m_WSmoothness. */
void SPalette::setSmoothnessWeight(float weight) {
    if(weight <= 0.0) {
	m_WSmoothness = 0.0f;
	m_UseSmoothness = false;
    } else {
	m_WSmoothness = weight;
	m_UseSmoothness = true;
    }
};

/** Sets m_WErrorMin */
void SPalette::setErrorMinWeight(float weight) {
    if(weight <= 0.0) {
	m_WErrorMin = 0;
	m_UseErrorMin = false;
    } else {
	m_UseErrorMin = true;
	m_WErrorMin = weight;
    }
};

//! Returns weighted error minimizing matrix for the given spectrum.
SMatrix SPalette::makeErrorMat(const vuColour31a& diag)
{
    SMatrix res(3,31);
    SMatrix dmat31(31,31);
    dmat31.makeDiag(SVector(31,diag.getData()));
    res = m_CXF31toRGB*dmat31;
    SMatrix dmat7(7,7);
    //cout << "light7: " << m_CXF31to7*SVector(31,diag.getData()) << endl;
    dmat7.makeDiag(m_CXF31to7*SVector(31,diag.getData()));
    //cout << "DIAG7: " << dmat7 << cout;
    
    res -= (m_CXF7toRGB*dmat7*m_CXF31to7);
    res *= m_WErrorMin;
    return res;
}

//! Returns weighted smoothing matrix (31x31).
SMatrix SPalette::makeSmoothingMat()
{
    SMatrix res(31,31);
/* we don't use 1st derivative
    static float vd1[] = {1,-1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
                          0,0,0,0,0,0,0,0,0,0,0,0,0};
    res.makeToeplitz(SVector(31,vd2),false);
*/  
    static float vd2data[] = {2,-1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
			0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
    SVector vd2(31,vd2data);
    vd2 *= 1/vd2.norm()/SQRT_31; //sqrt(31.0f);
    res.makeToeplitz(vd2,true);
    res *= m_WSmoothness;
    return res;
}

/** Returns a V31tRGB matrix accumulated with the diagonal matrix
    of the given Spectrum. 
    The switch useV7 decides whether to do diagonal multiply in V7.  
*/
SMatrix SPalette::makeCXFMat(const vuColour31a& diag, bool useV7)
{
    if(useV7) {
	SMatrix dmat7(7,7);
	dmat7.makeDiag(m_CXF31to7*SVector(diag));
	return m_CXF7toRGB*dmat7*m_CXF31to7;
    } else {
	SMatrix dmat31(31,31);
	dmat31.makeDiag(SVector(diag));
	return m_CXF31toRGB*dmat31;
    }
}

/** Creates a spectrum from the given design criteria.  This
    function builds a matrix and  solves for a (positivity constrained)
    least squares solution using SOptimizer. */
bool SPalette::createSpectrum()
{
  try{
	
    if(!checkSetup()) return false;
    
// determine column of design matrix for this 'free colour'
    dword posfc=0;
    for(dword c=0;c<m_FreeCol.getLength();c++)
	if(m_FreeCol[c].used) m_FreeCol[c].position = (posfc++);
    dword ndesl=0;
    for(dword c=0;c<m_Lights.getLength();c++)
	if(m_Lights[c].used) ndesl++;
    dword ndesr=0;
    for(dword c=0;c<m_Refls.getLength();c++)
	if(m_Refls[c].used) ndesr++;

    if(ndesr+ndesl == 0) return false;
    
//prepare M and y for constrained minimization of |Mx - y|
    dword nrows = ( 3*m_NDesCol + (m_UseErrorMin?3:0)*m_NDesCol +
	      (m_UseSmoothness?31:0)*(ndesr+ndesl));
    dword ncols = 3*posfc + 31*ndesl + 31*ndesr;
    SMatrix M(nrows,ncols,0.0f);
    SVector y(nrows,0.0f);

//make upper and lower bounds
    SVector lb(ncols,-SPAL_RANGE),ub(ncols,SPAL_RANGE);
    bool useBounds = false;
    posfc=0;
    for(dword c=0;c<m_FreeCol.getLength();c++)
	if(m_FreeCol[c].used) {
	    useBounds=true;
	    ub.insert(SVector(3,m_FreeCol[c].ub.getData()), 3*posfc);
	    lb.insert(SVector(3,m_FreeCol[c].lb.getData()), 3*posfc);
	    posfc++;
	}
    //ub.insert(SVector(posfc*3,1.0f));	//ub for 'free colours' = 1    

    SMatrix id33(3,3);
    id33.makeIdentity();
    dword curcol=3*posfc;
    dword currow=0;
//make matrix for the lights that have to be designed
    for(dword l=0;l<m_Lights.getLength();l++)
	if(m_Lights[l].used) 
	{
	    SSpectrum &speclig = m_Lights[l];
	    if(speclig.useBounds) {
		useBounds=true;
		ub.insert(SVector(speclig.ub),curcol);
		lb.insert(SVector(speclig.lb),curcol);
	    }
	    for(dword r=0;r<m_Refls.getLength();r++)
		if(m_DesCol[r][l].design)
		{
		    float weight = m_DesCol[r][l].getWeight();
		    SSpectrum &specrec = m_Refls[r];
		    M.insert(weight * makeCXFMat(specrec.spec,m_UseV7),
			     currow,curcol);
		    if(m_DesCol[r][l].freecolour>=0) 
		    {
			SFreeColour& fc = m_FreeCol[m_DesCol[r][l].freecolour];
			M.insert(-weight*id33,currow,
				 3*fc.position);
		    } else 		    
			y.insert(weight*m_DesCol[r][l].getCol(),currow);
		    
		    currow+=3;
		    if(m_UseErrorMin)
		    {
#ifdef USE_ERRORMAT
			M.insert(weight*makeErrorMat(specrec.spec),
				 currow,curcol);
#else
			M.insert((m_WErrorMin*weight) * 
				 makeCXFMat(specrec.spec,!m_UseV7),
				 currow,curcol);
			y.insert((m_WErrorMin*weight)*
				 m_DesCol[r][l].getCol(),currow);
#endif  // of USE_ERRORMAT
			currow+=3;
		    }
		}
	    if(m_UseSmoothness)
	    {
		M.insert(makeSmoothingMat(),currow,curcol);
		currow+=31;
	    }
	    curcol+=31;
	}
    
//make matrix for the reflectances that have to be designed
    for(dword r=0;r<m_Refls.getLength();r++)
	if(m_Refls[r].used) 
	{
	    SSpectrum &specref = m_Refls[r];
	    if(specref.useBounds) {
		useBounds=true;
		ub.insert(SVector(specref.ub),curcol);
		lb.insert(SVector(specref.lb),curcol);
	    }
	    for(dword l=0;l<m_Lights.getLength();l++)
		if(m_DesCol[r][l].design)
		{
		    float weight = m_DesCol[r][l].getWeight();
		    SSpectrum &specrec = m_Lights[l];
		    M.insert(weight * makeCXFMat(specrec.spec,m_UseV7),
			     currow,curcol);
		    if(m_DesCol[r][l].freecolour>=0) {
			SFreeColour& fc = m_FreeCol[m_DesCol[r][l].freecolour];
			M.insert(-weight*id33,currow,
				 3*fc.position);
		    } else
			y.insert(weight*m_DesCol[r][l].getCol(),currow);

		    currow+=3;
		    if(m_UseErrorMin)
		    {
#ifdef USE_ERRORMAT
			M.insert(weight*makeErrorMat(specrec.spec),
				 currow, curcol);
#else
			M.insert((m_WErrorMin*weight)*
				 makeCXFMat(specrec.spec,!m_UseV7),
				 currow, curcol);
			y.insert((m_WErrorMin*weight)*
				 m_DesCol[r][l].getCol(),currow);
#endif
			currow+=3;
		    }
		}
	    if(m_UseSmoothness)
	    {
		M.insert(makeSmoothingMat(),currow,curcol);
		currow+=31;
	    }
	    curcol+=31;
	}

//show stuff
//      cout << lb << "-----" << endl;
//      cout << ub << "-----" << endl;
//      cout << M << "-----" << endl;
//      cout << y << "-----" << endl;
//    m_Optimizer.setVerbose();
    
    
//optimize using constraints
    SVector x;
    if(useBounds)
	x = m_Optimizer.minimize(M,y,lb,ub);	// constrained opt.
    else 
	x = m_Optimizer.minimize(M,y);		// unconstrained opt.
    
//read results into light and reflectance slots
    double *res = x.getData() + 3*posfc;	// skip free colours
    for(dword l=0;l<m_Lights.getLength();l++)
	if(m_Lights[l].used) 
	{
	    m_Lights[l].spec = vuColour31a (res);
	    res+=31;
	}
    
    for(dword r=0;r<m_Refls.getLength();r++)
	if(m_Refls[r].used) 
	{
	    m_Refls[r].spec = vuColour31a (res);
	    res+=31;
	}
  }catch (const char* msg)
  {
      cout << "SPalette::createSpectrum() EXCEPTION: " << msg <<endl;
      return false;
  }
  
  return true;
}
 
}

