#ifndef _SOPTIMIZER_H_
#define _SOPTIMIZER_H_

#include "vuSimpleTypes.h"

#include "SMatrix.h"

/** The spectral optimizer performs a constrained least squares minimization.
    
It is used by SPalette to find valid spectra for certain design
criteria. The COOOL package by Lydia Deng and Wences Gouveia
(http://coool.mines.edu/) is used for optimization.
*/

using namespace coool;

class SOptimizer 
{
 public:
    /** constructor */
    SOptimizer();
    /** destructor */
    ~SOptimizer();

    /** searches for a vector x minmizing |Mx - y| while
	satisfying the constraints (upper and lower bounds).
	\param M matrix
	\param y vector to minimize for
	\param lb vector for lower bounds
	\param ub vector for upper bounds per component
	\param gradstep stepping for finite differences (default all=GradStep)
    */
    SVector minimize(const SMatrix& M, const SVector& y,
		     const SVector& lb, const SVector& ub);
    SVector minimize(const SMatrix& M, const SVector& y,
		     const SVector& lb, const SVector& ub, 
		     const SVector& gradstep);
    
    /** unconstrained minimization for |Mx - y|
	The result 
        \param M matrix
	\param y vector to minimize for
    */
    SVector minimize(const SMatrix& M, const SVector& y);

    //! enable/disable verbose output (by default off)
    void setVerbose(bool verbose=true) { m_Verbose = verbose;}
    //! set max number of iterations during optimization (default 100)
    void setItMax(int itmax) { m_ItMax=itmax;}
    //! set max n. of iterations during (cubic) line search (default 10)
    void setItMaxLine(int itmaxline) { m_ItMaxLine = itmaxline;}
    //! set threshold for objective function (default 0.001)
    void setTol(double tol) { m_Tol = tol;}
    //! set step size per component for finite differcing
    void setGradStep(float gradstep) {m_GradStep = gradstep;};
    
    //! get actual number of iterations for last minimization
    double getNIterations() const {return m_NIter;};
    //! get residue after last minimization
    double getResidue() const {return m_Residue;};

 protected:
    
    bool m_Verbose;
    int m_ItMax;
    int m_ItMaxLine;
    double m_Tol;
    float m_GradStep;
        
    double m_Residue;
    int m_NIter;
};
 
#endif

