#ifndef _SMATRIX_H_
#define _SMATRIX_H_

#include <iostream>
#include "vuSimpleTypes.h"

#include "vuColour.h"
#include "vuMatrix.h"
#include "Coool/DensMatrix.hh"
#include "LinAlg/LinAlg.h"


class SVector;

/** This class is (another) implementation of a matrix.  Unlike
    vuMatrix this class can be of arbitrary format. It's meant to act
    as converter to the matrix classes provided by the COOOL and the
    LinAlg package.
    
    This implementation is really inefficient, because there is a lot
    of memory copying and allocation going on. But it's ok for the
    stuff done in SOptimize. To see how a matrix class is really done, look
    at MTL (http://www.osl.iu.edu/research/mtl) or the one provided
    in LinAlg.

    The positions and sizes are written as rows first and then columns.
*/

class SMatrix 
{
 public:
    /** constructor */
    SMatrix(const dword nrows=1, const dword ncols=1);
    /** constructor with initialization */
    SMatrix(const dword nrows, const dword ncols, const double s);
    //!Copy constructor
    SMatrix(const SMatrix& m);
    //!Create from vuColour without alpha
    SMatrix(const vuColourN& col);
    //!Create from vuMatrix
    SMatrix(const vuMatrix& m);
    //!Create from a double*
    SMatrix(dword nrows, dword ncols, const double* data);
    //!Create from a float*
    SMatrix(dword nrows, dword ncols, const float* data);
    //!Create from linalg::Matrix
    SMatrix(const linalg::Matrix& m);
    //!Create from linalg::Vector
    SMatrix(const linalg::Vector& m);
    //!Create from coool::Matrix
    SMatrix(const coool::DensMatrix<double>& m);
    //!Create from coool::Vector
    SMatrix(const coool::Vector<double>& m);
    
    /** destructor */
    ~SMatrix();

    //!Assigns a matrix to the instance.
    SMatrix& operator=(const SMatrix& m);
    //!Assigns a constant scalar to all elements
    SMatrix& operator=(const double s);
    //!Assigns from float array
    SMatrix& operator=(const float* fdata);
    //!Assigns from double array
    SMatrix& operator=(const double* fdata);
    //!Assigns a matrix to the instance.
    SMatrix& operator=(const vuMatrix& m);
    //!Assigns a linalg::Matrix to the instance.
    SMatrix& operator=(const linalg::Matrix& m);
    //!Assigns a coool:DensMatrix to the instance.
    SMatrix& operator=(const coool::DensMatrix<double>& m);

    //!convert to a linalg::Matrix
    operator linalg::Matrix() const;
    //!convert to a coool::DensMatrix<double>
    operator coool::DensMatrix<double>() const;

    /**The access operator.
       To make it nice we could return a vector here, but therefore
       we'd need to reuse the memory (like LinAlg Matrix). */
    double* operator[](unsigned int index);
    
    //!The const access operator.
    const double* operator[](unsigned int index) const;

    //!Returns the data pointer.
    /*!Same as calling operator[0].
     */
    double* getData(void);
    double const * getData(void) const;

    /**Resizes the matrix.  No changes are made if the current
       dimensions aggree with the parameters.  Attention: The elements
       are not initialized. So call makeZero() on your own.  This
       function does pretty much the same as destroying the class and
       creating a new one.
    */
    void resize(dword nrows, dword ncols);
    
    //!returns number of rows
    dword getNCols() const {return m_NCols;};
    //!returns number of rows
    dword getNRows() const {return m_NRows;};

    //!Makes the matrix an identity matrix and returns a reference.
    SMatrix& makeIdentity(void);
    //!sets all elements of the matrix to zero
    SMatrix& makeZero(void);
    //!sets diagonal elements from vector diag
    SMatrix& makeDiag(const SVector& diag);
    //!creates a Toeplitz matrix using the elements of v as diagonal elements
    SMatrix& makeToeplitz(const SVector& v, bool symmetric=false);
    
    //!inserts the contents of another matrix at a certain position
    SMatrix& insert(const SMatrix& m, dword i=0, dword j=0);

    //note: makes a partial copy from (i,j) to (k,l)
    //SMatrix extract(dword i, dword j, dword k, dword l);
    
    //!horizontal concatenation of another matrix, rows must match
    SMatrix horizCat(const SMatrix& m) const;
    //!vertical concatenation with another matrix, columns must match
    SMatrix vertCat(const SMatrix& m) const;
    
    //!returns euclidic norm
    double norm() const;

    //!returns squared 2 euclidic norm
    double norm2() const;
    
    //!Returns the addition of the two matrices..
    SMatrix operator+(const SMatrix& m) const;

    //!Returns the difference of the two matrices..
    SMatrix operator-(const SMatrix& m) const;

    //!Multiplies a scalar by a matrix.
    friend SMatrix operator*(double s, const SMatrix& m);

    //!Returns the product of the two matrices.
    SMatrix operator*(const SMatrix& m) const;

    //!Returns the product of the instance with a scalar.
    SMatrix operator*(double s) const;
    
    //!Adds a matrix to the instance.
    SMatrix& operator+=(const SMatrix& m);
    //!Subtracts a matrix from the instance.
    SMatrix& operator-=(const SMatrix& m);
    //!Multiplies the instance by a matrix.
    SMatrix& operator*=(const SMatrix& m);
    //!Multiplies the instance by a scalar.
    SMatrix& operator*=(double s);

    //a friend, write matrix to stream
    friend ostream& operator<< (ostream& os, const SMatrix& A);
    // a friend, read matrix from stream
    friend istream& operator>> (istream& is, SMatrix& A);

 private:
    //!allocating memory and setting m_NRows and m_NCols
    void init(dword nrows, dword ncols);
    //!release the allocated memory
    void free();
    
 protected:
    //!number of rows and columns
    dword m_NRows, m_NCols;
    //!pointer to the two dimensional double array
    double *m_Data;
};

/** The SVector class is a Nx1 matrix, nothing else ;-) */
class SVector : public SMatrix 
{
 public:
    //!constructor for a Vector of dimension N
    SVector(dword N=1) : SMatrix(N,1) {};
    //!constructor for a Vector of dimension N, with constant values s
    SVector(dword N, double s) : SMatrix(N,1,s) {};
    //!constructor from double array
    SVector(dword N, const double* src) : SMatrix(N,1,src) {};
    //!constructor from float array
    SVector(dword N, const float* src) : SMatrix(N,1,src) {};
    //!copy constructor
    SVector(const vuColourN& col) : SMatrix(col) {};
    //!copy constructor
    SVector(const SVector& v) : SMatrix(v) {};
    //!copy constructor for a matrix, that has the apropriate Nx1 format
    SVector(const SMatrix& m) : SMatrix(m)
	{ if(m.getNCols() != 1)
	    throw "SVector: error, can't convert matrix with ncols != 1";
	};
    SVector(const coool::Vector<double> v) : SMatrix(v) {};

    //!returns the size of the vector
    dword getSize() const { return m_NRows;};
    
    //! Direct access to the elements
    double& operator[] (dword index) { return m_Data[index]; };
    //! Direct const access to the data
    const double operator[] (dword index) const { return m_Data[index]; };

    //!convert to a linalg::Vector
    operator linalg::Vector() const;
    //!convert to a coool::Vector<double>
    operator coool::Vector<double>() const;
};

#endif








