#include "SMatrix.h"

//#define max(a,b) (a>b ? a : b)
//#define min(a,b) (a<b ? a : b)
#undef min
#undef max

#ifndef min
#define min(a,b) ( a<b ? a : b )
#endif
#ifndef max
#define max(a,b) ( a>b ? a : b )
#endif

/*
template <class T>
const T& min(const T& a, const T& b) {
	return a<b ? a : b;
}
template <class T>
const T& max(const T& a, const T& b) {
	return a>b ? a : b;
}
*/

#define checkNRows(M) (m_NRows==M.m_NRows)
#define checkNCols(M) (m_NCols==M.m_NCols)
#define checkDims(M) (checkNRows(M) && checkNCols(M))

SMatrix::SMatrix(const dword nrows, const dword ncols)
{
    init(nrows, ncols);
}

SMatrix::SMatrix(const dword nrows, const dword ncols, const double s)
{
    init(nrows, ncols);
    operator=(s);
}

void SMatrix::init(dword nrows, dword ncols)
{
    m_NRows = nrows;
    m_NCols = ncols;
    if(nrows && ncols) m_Data = new double[ncols*nrows];
    else m_Data = 0;
}

void SMatrix::free()
{
    if(m_Data)
	delete m_Data;
    m_Data = 0;
    m_NRows = m_NCols = 0;
}

SMatrix::~SMatrix()
{
    free();
}

void SMatrix::resize(dword nrows, dword ncols)
{
    if((m_NRows != nrows) || (m_NCols != ncols))
    {
	free();
	init(nrows, ncols);
    }
}

//!Copy constructor
SMatrix::SMatrix(const SMatrix& m)
{
    init(m.m_NRows, m.m_NCols);
    operator=(m);
}

//!Create from a float*
SMatrix::SMatrix(dword nrows, dword ncols, const float* data)
{
    init(nrows, ncols);
    operator=(data);
}

//!Create from a double*
SMatrix::SMatrix(dword nrows, dword ncols, const double* data)
{
    init(nrows, ncols);
    operator=(data);
}

//!Create from vuColour without alpha
SMatrix::SMatrix(const vuColourN& col)
{
    init(col.nComponents()-1,1);
    double *dat = m_Data;
    const float * src = col.getData();
    for(dword i=0; i<m_NRows; i++, dat++, src++)
    {
	(*dat) = (*src);
    }
}

//!Create from vuMatrix
SMatrix::SMatrix(const vuMatrix& m)
{
    init(4,4);
    double *dat=m_Data;
    for(dword i=0; i<4; i++)
	for(dword j=0;j<4;j++, dat++)
	{
	    (*dat) = m[i][j];
	}
}

//!Create from linalg::Matrix
SMatrix::SMatrix(const linalg::Matrix& mat)
{
    // getting direct access to the elements, (undoing const)
    linalg::MatrixDA eth((linalg::Matrix&)mat);

    init(eth.q_row_upb()-eth.q_row_lwb()+1, 
	 eth.q_col_upb()-eth.q_col_lwb()+1);
    double *dat = m_Data;

    for(register int i=eth.q_row_lwb(); i<=eth.q_row_upb(); i++)
	for(register int j=eth.q_col_lwb(); j<=eth.q_col_upb(); j++, dat++)
	    *dat = eth(i,j);
}
 
//!Create from linalg::Matrix
SMatrix::SMatrix(const linalg::Vector& v)
{
    init(v.q_upb()-v.q_lwb()+1, 1);
    double *dat = m_Data;
    for(register int i=v.q_lwb(); i<=v.q_upb(); i++, dat++)
    {
	*dat = v(i);
    }
    
}
 
//!Create from coool::DensMatrix
SMatrix::SMatrix(const coool::DensMatrix<double>& m)
{
    init(m.numOfRows(), m.numOfCols());
    
    double *dat = m_Data;
    for(dword i=0; i<m_NRows; i++)
    {
	coool::Vector<double> row = m[i];
	for(dword j=0; j<m_NCols; j++, dat++)
	    *dat = row[j];
    }
}

SMatrix::SMatrix(const coool::Vector<double>& v)
{
    init(v.size(),1);
    double *dat = m_Data;
    for(dword i=0; i<m_NRows; i++, dat++)
    {
	*dat = v[i];
    }
}


//!Assigns a matrix to the instance.
SMatrix& SMatrix::operator=(const SMatrix& m)
{
    resize(m.m_NRows, m.m_NCols);
    double *dat = m_Data;
    double *src = m.m_Data;
    dword N = m_NRows*m_NCols;
    for(dword i=0; i<N; i++, dat++, src++)
    {
	(*dat) = (*src);
    }
    return *this;
}

//!Assigns scalar to the matrix elements
SMatrix& SMatrix::operator=(const double s)
{
    double *dat = m_Data;
    dword N = m_NRows*m_NCols;
    for(dword i=0; i<N; i++, dat++)
    {
	(*dat) = s;
    }
    return *this;
}

//!Assigns double array to the matrix elements
SMatrix& SMatrix::operator=(const float* fdata)
{
    double *dat = m_Data;
    float *src = (float*)fdata;
    dword N = m_NRows*m_NCols;
    for(dword i=0; i<N; i++, dat++, src++)
    {
	(*dat) = (*src);
    }
    return *this;
}

//!Assigns double array to the matrix elements
SMatrix& SMatrix::operator=(const double* fdata)
{
    double *dat = m_Data;
    double *src = (double*)fdata;
    dword N = m_NRows*m_NCols;
    for(dword i=0; i<N; i++, dat++, src++)
    {
	(*dat) = (*src);
    }
    return *this;
}

//!The access operator.
double* SMatrix::operator[](unsigned int index)
{
    return &m_Data[index * m_NCols];
}


//!The const access operator.
const double* SMatrix::operator[](unsigned int index) const
{
    return &m_Data[index * m_NCols];
}


//!Returns the data pointer.
/*!Same as calling operator[0].
 */
double* SMatrix::getData(void)
{
    return m_Data;
}

double const* SMatrix::getData(void) const
{
    return m_Data;
}

//!convert to a linalg::Matrix
SMatrix::operator linalg::Matrix() const
{
    linalg::Matrix mat(1,m_NRows, 1,m_NCols);
    linalg::MatrixDA eth(mat);
    double *dat = m_Data;
    for(register int i=eth.q_row_lwb(); i<=eth.q_row_upb(); i++)
	for(register int j=eth.q_col_lwb(); j<=eth.q_col_upb(); j++, dat++)
	    eth(i,j) = *dat;
    return mat;
}

//!convert to a coool::DensMatrix<double>
SMatrix::operator coool::DensMatrix<double>() const
{
    coool::DensMatrix<double> dmat(m_NRows, m_NCols);
    double *dat = m_Data;
    for(dword i=0; i<m_NRows; i++)
    {
	for(dword j=0; j<m_NCols; j++, dat++)
	    dmat[i][j] = *dat;
    }
    return dmat;
}

double SMatrix::norm2() const
{
    double *dat = m_Data;
    double res=0;
    dword N = m_NRows*m_NCols;
    for(dword i=0; i<N; i++, dat++)
	res += (*dat)*(*dat);
    return res;
}

double SMatrix::norm() const
{
    return sqrt(norm2());
}

//!Makes the matrix an identity matrix and returns a reference.
SMatrix& SMatrix::makeIdentity(void)
{
    makeZero();
    double *dat = m_Data;
    dword N = min(m_NRows, m_NCols);
    for(dword i=0; i<N; i++, dat+=m_NCols+1)
    {
	(*dat) = 1.0f;
    }
    return *this;
}

SMatrix& SMatrix::makeZero(void)
{
    operator=(0.0f);
    return *this;
}

//!sets diagonal elements from vector diag
SMatrix& SMatrix::makeDiag(const SVector& diag)
{
    makeZero();
    double *dat = m_Data;
    double const* src = diag.getData();
    dword N = min(m_NRows, m_NCols);
    for(dword i=0; i<N; i++, src++, dat+=m_NCols+1)
    {
	(*dat) = (*src);
    }
    return *this;
}

//!creates a Toeplitz matrix
SMatrix& SMatrix::makeToeplitz(const SVector& v, bool symmetric)
{
    if(v.getSize() < m_NCols)
	throw "SMatrix::makeToeplitz Error: dimensions don't agree.";
    if(!symmetric || m_NRows<m_NCols) makeZero();
    dword N = min(m_NRows, m_NCols);
    double *dat = m_Data;
//fill upper triangle
    for(dword j=0; j<N; j++)
    {
	const double *src = v.getData();
	dat+=j;
	for(dword i=j; i<m_NRows; i++, src++, dat++)
	{
	    (*dat) = (*src);
	}
    }
//fill lower triangle
    if(symmetric) {
	dat = m_Data+m_NCols;
	for(dword j=1; j<N; j++)
	{
	    const double *src = v.getData()+1;
	    for(dword i=0; i<j; i++, src++, dat--)
	    {
		(*dat) = (*src);
	    }
	    dat+=j+m_NCols+1;
	}
    }
    
    return *this;
}

//!inserts the contents of another matrix at a certain position
SMatrix& SMatrix::insert(const SMatrix& m, dword i, dword j)
{
    if(m.m_NRows+i > m_NRows  ||  m.m_NCols+j>m_NCols)
	throw "SMatrix::insert operand dimensions too big.";
    double *dst = m_Data + m_NCols*i + j;
    const double *src = m.m_Data;
    dword dstinc = m_NCols-m.m_NCols;
    for(dword a=0; a<m.m_NRows; a++, dst+=dstinc)
	for(dword b=0; b<m.m_NCols; b++, src++, dst++)
	    (*dst) = (*src);
    return *this;
}

//!horizontal concatenation of another matrix, rows must match
SMatrix SMatrix::horizCat(const SMatrix& m) const
{
    if(!checkNRows(m))
	throw "SMatrix::horizCat matrix dimensions must agree.";
    SMatrix ret(m_NRows, m_NCols+m.m_NCols);
    ret.insert(*this);
    ret.insert(m,0,m_NCols);
    
    return ret;
}


//!vertical concatenation with another matrix, columns must match
SMatrix SMatrix::vertCat(const SMatrix& m) const
{
    if(!checkNCols(m))
	throw "SMatrix::vertCat matrix dimensions must agree.";
    
    SMatrix ret(m_NRows+m.m_NRows, m_NCols);
    ret.insert(*this);
    ret.insert(m,m_NRows,0);
    
    return ret;
}

//!Returns the addition of the two matrices..
SMatrix SMatrix::operator+(const SMatrix& m) const
{
    if(checkDims(m)) {
	SMatrix res(*this);
	return res += m;
    } else throw "SMatrix::operator+ matrix dimensions don't agree.";
}

//!Returns the addition of the two matrices..
SMatrix SMatrix::operator-(const SMatrix& m) const
{
    if(checkDims(m)) {
	SMatrix res(*this);
	return res -= m;
    } else throw "SMatrix::operator- matrix dimensions don't agree.";
}

//!Returns the product of the two matrices.
SMatrix SMatrix::operator*(const SMatrix& m) const
{
    if(m_NCols==m.m_NRows) {
	SMatrix res(m_NRows, m.m_NCols, 0.0f);
	for(dword h=0; h<m.m_NCols; h++)
	    for(dword i=0; i<m_NRows; i++)
	    {
		double &r= res[i][h];
		double *me = &m_Data[m_NCols * i];
		double *it = &m.m_Data[h];
		for(dword j=0; j<m_NCols; j++, me++, it += m.m_NCols)
		{
		    r += (*me) * (*it);
		}
	    }
	return res;
    } else throw "SMatrix::operator* matrix dimensions don't agree.";
}


//!Returns the product of the instance with a scalar.
SMatrix SMatrix::operator*(double s) const
{
    SMatrix res(*this);
    return res *= s;
}

//!Multiplies a scalar by a matrix.
SMatrix operator*(const double s, const SMatrix& m)
{
    return m*s;
}

//!Adds a matrix to the instance.
SMatrix& SMatrix::operator+=(const SMatrix& m)
{
    if(checkDims(m)) {
	double *dat = m_Data;
	double *src = m.m_Data;
	dword N = m_NRows*m_NCols;
	for(dword i=0; i<N; i++, dat++, src++)
	{
	    (*dat) += (*src);
	}
	
	return *this;
    } else throw "SMatrix::operator+ matrix dimensions don't agree.";
}

//!Subtracts a matrix from the instance.
SMatrix& SMatrix::operator-=(const SMatrix& m)
{
    if(checkDims(m)) {
	double *dat = m_Data;
	double *src = m.m_Data;
	dword N = m_NRows*m_NCols;
	for(dword i=0; i<N; i++, dat++, src++)
	{
	    (*dat) -= (*src);
	}
	
	return *this;
    } else throw "SMatrix::operator- matrix dimensions don't agree.";
}

//!Multiplies the instance by a matrix.
SMatrix& SMatrix::operator*=(const SMatrix& m)
{
    if(checkDims(m) && (m_NCols==m_NRows)) {
	SMatrix tmp(*this);
	for(dword h=0; h<m_NCols; h++)
	    for(dword i=0; i<m_NRows; i++)
	    {
		double &r= m_Data[m_NCols * i + h];
		r = 0;
		double *a = &m_Data[m_NCols * i];
		double *b = &m.m_Data[h];
		for(dword j=0; j<m_NCols; j++, a++, b += m_NCols)
		{
		    r += (*a) * (*b);
		}
	    }
	return *this;
    } else throw "SMatrix::operator*= matrix dimensions don't agree.";
}

//!Multiplies the instance by a scalar.
SMatrix& SMatrix::operator*=(double s)
{
    double *dat = m_Data;
    dword N = m_NRows*m_NCols;
    for(dword i=0; i<N; i++, dat++)
    {
	(*dat) *= s;
    }
    return *this;
}

ostream& operator<<(ostream& ofp, const SMatrix& A)
{
    double *dat = A.m_Data;
    for (dword i = 0; i < A.m_NRows; i++)
    {
	for (dword j = 0; j < A.m_NCols; j++, dat++)
	    ofp << (*dat) << " ";

	ofp << endl;
    }
    return ofp;
}

istream& operator>>(istream& ifp, SMatrix& A)
{
    double *dat = A.m_Data;
    dword N = A.m_NRows* A.m_NCols;
    for (dword j = 0; j < N; j++, dat++)
	ifp >> (*dat);
    
    return ifp;
}

//------------------------------------------------------------------------------
// SVector stuff

SVector::operator coool::Vector<double>() const
{
    coool::Vector<double> v(m_NRows);
    for(dword i=0; i<m_NRows;i++)
    {
	v[i] = m_Data[i];
    }
    return v;
};

SVector::operator linalg::Vector() const
{
    linalg::Vector v(1,m_NRows);
    double *dat = m_Data;
    for(register int i=v.q_lwb(); i<=v.q_upb(); i++, dat++)
    {
	v(i) = *dat;
    }
    
    return v;
}
