#include <math.h>
#include <stdlib.h>
#include "vuNormalBlock.h"

vuNormalBlock::vuNormalBlock()
{
    m_Size = 0;
    m_Current = 0;
    m_Norms = 0;
    m_Dim = 0;
    m_Range = 0.0f;
    m_StdDev = 0.0f;
}

vuNormalBlock::vuNormalBlock(const vuNormalBlock& inst)
{
    dword i;

    m_Size = inst.m_Size;
    m_Current = inst.m_Current;
    m_Dim = inst.m_Dim;
    m_Range = inst.m_Range;
    m_StdDev = inst.m_StdDev;

    m_Norms = new float[m_Size*3];
    for(i=0;i<m_Size;++i)
        m_Norms[i] = inst.m_Norms[i];
}

vuNormalBlock::~vuNormalBlock()
{
    if (m_Norms)
        delete [] m_Norms;
}

void vuNormalBlock::setSize(dword size)
{
    m_Size = size;
    m_Current = 0;
    m_Range = 0.0f;
    m_StdDev = 0.0f;

    if (m_Norms)
        delete [] m_Norms;

    m_Norms = new float[m_Size*3];
}

dword vuNormalBlock::getSize(void) const
{
    return m_Size;
}

float vuNormalBlock::getRange(void) const
{
    return m_Range;
}

float vuNormalBlock::getStdDev(void) const
{
    return m_StdDev;
}

void vuNormalBlock::addNormal(const float* norm)
{
    m_Norms[m_Current++] = norm[0];
    m_Norms[m_Current++] = norm[1];
    m_Norms[m_Current++] = norm[2];
}

void vuNormalBlock::computeStats(void)
{
    float range[3];
    float stddev[3];
    float min;
    float max;
    float sum;
    float sumsqr;
    float tmp;
    dword dim;
    dword i;

    for(dim=0;dim<3;++dim)
    {
        min = 1000000.0f;
        max = -1000000.0f;
        sum = 0.0f;
        sumsqr = 0.0f;

        for(i=0;i<m_Size*3;i+=3)
        {
            tmp = m_Norms[i+dim];
            if (tmp < min)
                min = tmp;
            if (tmp > max)
                max = tmp;
            sum += tmp;
            sumsqr += tmp*tmp;
        }
        range[dim] = max-min;
        stddev[dim] = (sumsqr-((sum*sum)/(float)m_Size))/((float)m_Size-1);
    }

    m_Dim = 0;
    for(dim=1;dim<3;++dim)
    {
        if (range[dim] > range[m_Dim])
            m_Dim = dim;
        else if (range[dim] == range[m_Dim])
        {
            if (stddev[dim] > stddev[m_Dim])
                m_Dim = dim;
        }
    }

    m_Range = range[m_Dim];
    m_StdDev = stddev[m_Dim];
}

bool vuNormalBlock::medianSplit(vuNormalBlock& left, vuNormalBlock& right)
{
    float median;
    dword half;
    dword h1;
    dword h2;

    QuickSort(0, m_Size-1);

    if (m_Size&0x01)
        half = (m_Size-1)>>1;
    else
        half = m_Size>>1;
    median = m_Norms[half*3+m_Dim];

    h1 = half-1;
    while((m_Norms[h1*3+m_Dim]==median) && (h1>0))
        --h1;

    h2 = half+1;
    while((m_Norms[h2*3+m_Dim]==median) && (h2<(m_Size-1)))
        ++h2;

    if ((h2-half) < (half-h1))
        half = h2;
    else
        half = h1+1;

    if (half == 0 || half == m_Size)
        return false;

    left.setSize(half);
    right.setSize(m_Size-half);

    half *= 3;

    for(h1=0;h1<half;++h1)
        left.m_Norms[h1] = m_Norms[h1];

    for(h1=half;h1<m_Size*3;++h1)
        right.m_Norms[h1-half] = m_Norms[h1];

    left.computeStats();
    right.computeStats();

    return true;
}

void vuNormalBlock::computeAverage(float* f) const
{
    float len;
    dword i;

    f[0] = 0.0f;
    f[1] = 0.0f;
    f[2] = 0.0f;
    for(i=0;i<m_Size*3;i+=3)
    {
        f[0] += m_Norms[i];
        f[1] += m_Norms[i+1];
        f[2] += m_Norms[i+2];
    }

    len = (float)sqrt((double)((f[0]*f[0])+(f[1]*f[1])+(f[2]*f[2])));
    if (len > 0.0f)
    {
        f[0] /= len;
        f[1] /= len;
        f[2] /= len;
    }
}

vuNormalBlock& vuNormalBlock::operator=(const vuNormalBlock& rhs)
{
    if (this != &rhs)
    {
        dword i;

        m_Size = rhs.m_Size;
        m_Current = rhs.m_Current;
        m_Range = rhs.m_Range;
        m_StdDev = rhs.m_StdDev;

        if (m_Norms)
            delete [] m_Norms;
        m_Norms = new float[m_Size*3];

        for(i=0;i<m_Size;++i)
            m_Norms[i] = rhs.m_Norms[i];
    }
    return *this;
}

int vuNormalBlock::Partition(int p, int r)
{
    float t;
    float x;
    int i;
    int j;

    x = m_Norms[p*3+m_Dim];
    i = p-1;
    j = r+1;
    while(true)
    {
        do{ --j; } while(m_Norms[j*3+m_Dim] > x);
        do{ ++i; } while(m_Norms[i*3+m_Dim] < x);
        if(i < j)
        {
            t = m_Norms[i*3];  m_Norms[i*3] = m_Norms[j*3];  m_Norms[j*3] = t;
            t = m_Norms[i*3+1];  m_Norms[i*3+1] = m_Norms[j*3+1];  m_Norms[j*3+1] = t;
            t = m_Norms[i*3+2];  m_Norms[i*3+2] = m_Norms[j*3+2];  m_Norms[j*3+2] = t;
        }
        else
            return j;
    }
}

int vuNormalBlock::RandomPartition(int p, int r)
{
    float t;
    int i;
    
    i = (rand()%(r-p))+p+1;
    t = m_Norms[p*3];  m_Norms[p*3] = m_Norms[i*3];  m_Norms[i*3] = t;
    t = m_Norms[p*3+1];  m_Norms[p*3+1] = m_Norms[i*3+1];  m_Norms[i*3+1] = t;
    t = m_Norms[p*3+2];  m_Norms[p*3+2] = m_Norms[i*3+2];  m_Norms[i*3+2] = t;
    return Partition(p, r);
}

void vuNormalBlock::QuickSort(int p, int r)
{
    int q;
    if (p < r)
    {
        q = RandomPartition(p, r);
        QuickSort(p, q);
        QuickSort(q+1, r);
    }
}
