#include "vuUDSphere.h"
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#define TWO_PI      6.283185307179586

vuUDSphere::vuUDSphere(int _size)
{
  m_Tolerance = 0.001;
  m_Size      = _size + 1; //the first point is the center, so it doesn't count
  m_P         = new Point3d[m_Size];
  m_dt        = TWO_PI / ((double)m_Size);

  initPoints();
}

vuUDSphere::~vuUDSphere()
{
  delete [] m_P;
}

//**********************************************************//
//**** PRIVATE methods: The physical simulation         ****//
//**********************************************************//

void vuUDSphere::makeRandomPoint(Point3d& p)
{
  double x, y, z;
  double dist;

  // Create random point
  x = ((double)(rand()%RAND_MAX))/((double)(RAND_MAX-1))*2.0 - 1.0;
  y = ((double)(rand()%RAND_MAX))/((double)(RAND_MAX-1))*2.0 - 1.0;
  z = ((double)(rand()%RAND_MAX))/((double)(RAND_MAX-1))*2.0 - 1.0;
  dist = sqrt(x*x + y*y + z*z);
  x /= dist;
  y /= dist;
  z /= dist;

  // Init point with random vector
  p.set(x, y, z);
}

void vuUDSphere::initPoints(void)
{
  int i;

  m_P[0].x = 0.0;
  m_P[0].y = 0.0;
  m_P[0].z = 0.0;
  for(i=1;i<m_Size;++i)
    makeRandomPoint(m_P[i]);
}

void vuUDSphere::computeForces(Point3d& p0, Point3d& p1)
{
  double x, y, z;
  double d;

  x = p1.x - p0.x;
  y = p1.y - p0.y;
  z = p1.z - p0.z;

  d = x*x + y*y + z*z;

  if (d == 0.0)
    d = 0.0001;
  d = d*sqrt(d);

  x /= d;
  y /= d;
  z /= d;

  p0.fx -= x;
  p0.fy -= y;
  p0.fz -= z;

  p1.fx += x;
  p1.fy += y;
  p1.fz += z;
}

//**********************************************************//
//****                   PUBLIC methods                 ****//
//**********************************************************//

bool vuUDSphere::advancePoints(void)
{
  double sx, sy, sz;
  double min;
  double t;
  double dt2;
  int i;
  int j;
  bool result;

  result = false;

  // Set forces to zero
  for(i=1;i<m_Size;++i)
    {
      m_P[i].fx = 0.0;
      m_P[i].fy = 0.0;
      m_P[i].fz = 0.0;
    }

  // Compute forces
  for(i=1;i<m_Size;++i)
    for(j=i+1;j<m_Size;++j)
      computeForces(m_P[i], m_P[j]);

  dt2 = m_dt*m_dt;

  // Apply forces
  for(i=1;i<m_Size;++i)
    {
      sx = m_P[i].x;
      sy = m_P[i].y;
      sz = m_P[i].z;

      m_P[i].x += m_P[i].fx * dt2;
      m_P[i].y += m_P[i].fy * dt2;
      m_P[i].z += m_P[i].fz * dt2;

      // Keep on unit sphere
      t = sqrt(m_P[i].x*m_P[i].x + m_P[i].y*m_P[i].y + m_P[i].z*m_P[i].z);
      m_P[i].x /= t;
      m_P[i].y /= t;
      m_P[i].z /= t;

      // Check if total movement is less than tolerance
      if (!result)
        {
	  sx -= m_P[i].x;
	  sy -= m_P[i].y;
	  sz -= m_P[i].z;
	  t = sx*sx + sy*sy + sz*sz;
	  if (t > m_Tolerance*m_Tolerance)
	    result = true;
        }
    }

  // Move closest to (0,1,0) to (0,1,0)
  min = m_P[1].x*m_P[1].x + (m_P[1].y-1.0)*(m_P[1].y-1.0) + m_P[1].z*m_P[1].z;
  j = 1;
  for(i=2;i<m_Size;++i)
    {
      t = m_P[i].x*m_P[i].x + (m_P[i].y-1.0)*(m_P[i].y-1.0) + m_P[i].z*m_P[i].z;
      if (t < min)
        {
	  min = t;
	  j = i;
        }
    }
  m_P[j].x = 0.0;
  m_P[j].y = 1.0;
  m_P[j].z = 0.0;

  return result;
}

bool vuUDSphere::doesFileExist() {
  FILE *fp;
  char name[32];

  sprintf(name, "%i.nrm", m_Size-1);
  fp = fopen(name, "rb");
  if (fp) {
    fclose(fp);
    return true;
  }
  else
    return false;
}

void vuUDSphere::writeToFile()
{
  FILE* fp;
  float n[3];
  char name[32];
  int i;
  sprintf(name, "%i.nrm", m_Size-1);
  fp = fopen(name, "wb");
  if (fp) {
    for(i=1;i<m_Size;++i) {
      n[0] = (float)(m_P[i].x);
      n[1] = (float)(m_P[i].y);
      n[2] = (float)(m_P[i].z);
      fwrite(n, sizeof(float)*3, 1, fp);
    }
    fclose(fp);
  }
}

void vuUDSphere::readFromFile()
{
  FILE* fp;
  float n[3];
  char name[32];
  int i;
  sprintf(name, "%i.nrm", m_Size-1);
  fp = fopen(name, "rb");
  if (fp) {
    for(i=1;i<m_Size;++i) {
      fread(n, sizeof(float)*3, 1, fp);
      m_P[i].set(n[0], n[1], n[2]);
    }
    fclose(fp);
  }
}

int vuUDSphere::getSize(void)
{
  return m_Size-1;
}

Point3d  vuUDSphere::getPointAt(int idx)
{
  assert(idx <= m_Size-1);
  return m_P[idx+1];
}

void vuUDSphere::calculate(void) {
  while(advancePoints());
}

void vuUDSphere::lazyCalculate(void) {
  if (doesFileExist())
    readFromFile();
  else {
    calculate();
    writeToFile();
  }
}
